package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.kuiper.entity.FrontFileEntity;
import com.bizunited.platform.kuiper.starter.repository.FrontFileRepository;
import com.bizunited.platform.kuiper.starter.service.FrontFileSevice;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.transaction.Transactional;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import static com.bizunited.platform.common.constant.Constants.LINE_SEPARATOR;
import static com.bizunited.platform.common.constant.MigrateDataConstants.FRONT_FILE_FILENAME;

@Service("FrontFileServiceImpl")
public class FrontFileServiceImpl implements FrontFileSevice {

  private static final Logger LOGGER = LoggerFactory.getLogger(FrontFileServiceImpl.class);

  @Autowired 
  private VenusFileService venusFileService;
  @Autowired 
  private FrontFileRepository frontFileRepository;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public FrontFileEntity create(String frontFileEntityStr, String fileContent) {
    Validate.notBlank(frontFileEntityStr, "前端文件实体json不能为空！");
    Validate.notBlank(fileContent, "前端文件内容不能为空！");
    FrontFileEntity entity;
    try {
      JSON json = JSON.parseObject(frontFileEntityStr);
      entity = JSON.toJavaObject(json, FrontFileEntity.class);
    } catch (RuntimeException e) {
      throw new IllegalArgumentException(e);
    }
    Validate.notNull(entity, "前端文件json转化实体不能为空！");
    Validate.notBlank(entity.getName(), "前端文件实体名称不能为空！");
    FrontFileEntity existNameEntity = frontFileRepository.findByNameAndProjectName(entity.getName(), platformContext.getAppName());
    Validate.isTrue(null == existNameEntity,"存在重复名称，请检查！");
    // 保存JSON到文件路径
    this.saveFile(entity, fileContent);

    entity.setCreateTime(new Date());
    entity.setFileStatus(0);
    entity.setProjectName(platformContext.getAppName());
    return frontFileRepository.saveAndFlush(entity);
  }

  /**
   * 保存文件
   * @param entity
   * @param fileContent
   */
  private FrontFileEntity saveFile(FrontFileEntity entity, String fileContent) {
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String fileRename = uuid + ".txt";
    String relativePath =
        StringUtils.join("/frontFile/", folderName, "/", (new Random().nextInt(100) % 10));
    byte[] content;
    content = fileContent.getBytes(StandardCharsets.UTF_8);
    venusFileService.saveFile(relativePath, fileRename, fileRename, content);
    entity.setFilePath(relativePath);
    entity.setFileName(fileRename);
    return entity;
  }

  @Override
  @Transactional
  public FrontFileEntity update(String frontFileEntityStr, String fileContent) {
    Validate.notBlank(frontFileEntityStr, "前端文件实体json不能为空！");
    Validate.notBlank(fileContent, "前端文件内容不能为空！");
    FrontFileEntity entity;
    try {
      JSON json = JSON.parseObject(frontFileEntityStr);
      entity = JSON.toJavaObject(json, FrontFileEntity.class);
    } catch (Exception e) {
      throw new IllegalArgumentException(e);
    }
    Validate.notNull(entity, "前端文件json转化实体不能为空！");
    Validate.notBlank(entity.getId(), "更新时，前端文件实体ID不能为空！");
    Optional<FrontFileEntity> op = frontFileRepository.findById(entity.getId());
    FrontFileEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity, "未根据ID找到现有实体，请检查！");

    Validate.notBlank(entity.getName(), "前端文件实体名称不能为空！");
    Validate.notNull(entity.getFileStatus(), "前端文件实体状态不能为空！");

    this.saveFile(existEntity, fileContent);

    existEntity.setName(entity.getName());
    existEntity.setFileClassify(entity.getFileClassify());
    existEntity.setFileDesc(entity.getFileDesc());
    return frontFileRepository.saveAndFlush(existEntity);
  }

  @Override
  @Transactional
  public FrontFileEntity saveEntity(FrontFileEntity frontFileEntity) {
    Validate.notNull(frontFileEntity,"前端文件信息不能为空！");
    Validate.isTrue(StringUtils.isBlank(frontFileEntity.getId()),"前端文件新建时ID必须为空！");
    Validate.notBlank(frontFileEntity.getName(),"前端文件名称不能为空！");
    FrontFileEntity existName = frontFileRepository.findByNameAndProjectName(frontFileEntity.getName(), platformContext.getAppName());
    Validate.isTrue(null == existName,"前端文件名不能重复！");
    Validate.notBlank(frontFileEntity.getFilePath(),"前端文件文件路径不能为空！");
    Validate.notBlank(frontFileEntity.getFileName(),"前端文件文件名不能为空！");
    Validate.notNull(frontFileEntity.getFileStatus(),"前端文件状态不能为空！");
    frontFileEntity.setCreateTime(new Date());
    frontFileEntity.setProjectName(platformContext.getAppName());
    return frontFileRepository.saveAndFlush(frontFileEntity);
  }

  @Override
  @Transactional
  public FrontFileEntity updateEntity(FrontFileEntity frontFileEntity) {
    Validate.notNull(frontFileEntity,"前端文件信息不能为空！");
    Validate.isTrue(StringUtils.isNotBlank(frontFileEntity.getId()),"更新文件新建时ID不能为空！");
    FrontFileEntity existFront = frontFileRepository.findById(frontFileEntity.getId()).orElse(null);
    Validate.notNull(existFront,"前端文件根据ID查询不到信息！");
    Validate.notBlank(frontFileEntity.getName(),"前端文件名称不能为空！");
    // 验证是否变动了 name, 如果变动了，则检查新的NAME是否存在
    if(!StringUtils.equals(frontFileEntity.getName(),existFront.getName())){
      FrontFileEntity existName = frontFileRepository.findByNameAndProjectName(frontFileEntity.getName(), platformContext.getAppName());
      Validate.isTrue(existName == null,"修改后的前端文件名已重复，请修改");
    }
    existFront.setName(frontFileEntity.getName());
    existFront.setFileClassify(frontFileEntity.getFileClassify());
    existFront.setFileDesc(frontFileEntity.getFileDesc());
    existFront.setFilePath(frontFileEntity.getFilePath());
    existFront.setFileName(frontFileEntity.getFileName());
    existFront.setCreateTime(frontFileEntity.getCreateTime());
    existFront.setFileStatus(frontFileEntity.getFileStatus());
    return frontFileRepository.save(existFront);
  }

  @Override
  @Transactional
  public FrontFileEntity disable(String frontFileId) {
    Validate.notBlank(frontFileId, "传入前端文件ID不能为空!!");
    Optional<FrontFileEntity> op = frontFileRepository.findById(frontFileId);
    FrontFileEntity entity = op.orElse(null);
    Validate.notNull(entity, "查询前端文件为空，请检查!!");

    entity.setFileStatus(1);
    return frontFileRepository.save(entity);
  }

  @Override
  @Transactional
  public FrontFileEntity enable(String frontFileId) {
    Validate.notBlank(frontFileId, "传入前端文件ID不能为空！");
    Optional<FrontFileEntity> op = frontFileRepository.findById(frontFileId);
    FrontFileEntity entity = op.orElse(null);
    Validate.notNull(entity, "查询前端文件为空，请检查！");
    entity.setFileStatus(0);
    return frontFileRepository.save(entity);
  }

  @Override
  @Transactional
  public void deleteByName(String frontFileName) {
    FrontFileEntity frontFileEntity = frontFileRepository.findByNameAndProjectName(frontFileName, platformContext.getAppName());
    if(null == frontFileEntity){
      return;
    }
    frontFileRepository.delete(frontFileEntity);
    frontFileRepository.flush();
    // 最后试图删除文件
    venusFileService.deleteFile(frontFileEntity.getFilePath(),frontFileEntity.getFileName(),frontFileEntity.getFileName());
  }

  @Override
  public String findContentById(String frontFileId) {
    Validate.notBlank(frontFileId, "传入前端文件ID不能为空！");
    Optional<FrontFileEntity> op = frontFileRepository.findById(frontFileId);
    FrontFileEntity entity = op.orElse(null);
    Validate.notNull(entity, "查询前端文件为空，请检查！");
    byte[] content = venusFileService.readFileContent(entity.getFilePath(), entity.getFileName());
    return new String(content, StandardCharsets.UTF_8);
  }


  @Override
  public String findContentByName(String frontFileName) {
    Validate.notBlank(frontFileName, "传入前端文件名称不能为空！");
    FrontFileEntity entity = frontFileRepository.findByNameAndProjectName(frontFileName, platformContext.getAppName());
    if(null == entity){
      return null;
    }
    byte[] content = venusFileService.readFileContent(entity.getFilePath(), entity.getFileName());
    if(null != content){
      return new String(content, StandardCharsets.UTF_8);
    }else{
      return "";
    }
  }

  @Override
  public FrontFileEntity findByName(String name) {
    if(StringUtils.isBlank(name)) {
      return null;
    }
    return this.frontFileRepository.findByNameAndProjectName(name, platformContext.getAppName());
  }

  @Override
  public FrontFileEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.frontFileRepository.findById(id).orElse(null);
  }

  @Override
  public Page<FrontFileEntity> queryPage(Pageable pageable, String name, String fileClassify, Integer fileStatus) {
    Validate.notNull(pageable, "分页信息不能为空");
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (StringUtils.isNotBlank(fileClassify)) {
      conditions.put("fileClassify", fileClassify);
    }
    if (null != fileStatus) {
      conditions.put("fileStatus", fileStatus);
    }
    return frontFileRepository.queryPage(pageable, conditions);
  }
  
  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.frontFileRepository.countByIds(ids);
  }

  /**
   * 根据指定的id数组查询
   * @param frontFileIds
   * @return
   */
  @Override
  public Set<FrontFileEntity> findByIds(String[] frontFileIds) {
    if(ArrayUtils.isEmpty(frontFileIds)) {
      return Sets.newHashSet();
    }
    return frontFileRepository.findByIds(frontFileIds);
  }

  /**
   * 查询所有页面函数
   * @return
   */
  @Override
  public List<FrontFileEntity> findAll() {
    List<FrontFileEntity> frontFileEntities = frontFileRepository.findAll();
    if(!CollectionUtils.isEmpty(frontFileEntities)){
      frontFileEntities = frontFileEntities.stream().filter(frontFileEntity -> StringUtils.equals(frontFileEntity.getProjectName(), platformContext.getAppName())).collect(Collectors.toList());
    }
    return frontFileEntities;
  }

  /**
   * 导入页面函数
   * @param importModel
   */
  @Override
  @Transactional(Transactional.TxType.REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry frontFile = zipFile.getEntry(FRONT_FILE_FILENAME);
    if(frontFile == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
      return;
    }

    try (InputStream is = zipFile.getInputStream(frontFile)) {
      ObjectInputStream ois = new ObjectInputStream(is);
      List<FrontFileEntity> frontFiles = (List<FrontFileEntity>) ois.readObject();
      for(FrontFileEntity item : frontFiles){
        // 将脚本文件从之前保存的ZIP文件特定路径下获取出
        ZipEntry scriptFile = ZipFileUtils.getZipEntry(zipFile, item.getFilePath(), item.getFileName());
        InputStream inputStream = zipFile.getInputStream(scriptFile);
        byte[] bytes = IOUtils.toByteArray(inputStream);
        String content = new String(bytes, StandardCharsets.UTF_8);
        item.setContent(content);
      }
      if(CollectionUtils.isEmpty(frontFiles)) {
        importModel.appendLine("导入的数据为空");
      } else {
        this.importData(frontFiles, importModel);
      }
    } catch (IOException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      importModel.append("读取业务数据失败：").append(e.getMessage()).append(LINE_SEPARATOR);
    }
  }

  /**
   * 批量导入页面函数
   * @param frontFiles
   * @param importModel
   */
  private void importData(List<FrontFileEntity> frontFiles, MigrateImportModel importModel) {
    importModel.setTotalCount(frontFiles.size());
    for (int i = 0; i < frontFiles.size(); i++) {
      FrontFileEntity frontFile = frontFiles.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(frontFile, importModel);
    }
  }

  /**
   * 导入页面函数（单个处理）
   * @param frontFile
   * @param importModel
   */
  private void importData(FrontFileEntity frontFile, MigrateImportModel importModel) {
    importModel.appendLine(String.format("开始导入数据：文件名称=%s", frontFile.getName()));
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    FrontFileEntity dbFrontFile = frontFileRepository.findByNameAndProjectName(frontFile.getName(), platformContext.getAppName());
    frontFile.setId(null);
    if(dbFrontFile != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("页面函数已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbFrontFile != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("编码规则已存在，进行更新覆盖");
      this.handleUpdateData(frontFile, dbFrontFile, importModel);
      return;
    }
    if(dbFrontFile == null) {
      this.handleCreateData(frontFile, importModel);
    }
  }

  /**
   * 导入-创建页面函数
   * @param frontFile
   * @param importModel
   */
  private void handleCreateData(FrontFileEntity frontFile, MigrateImportModel importModel) {
    importModel.appendLine("导入页面函数");
    importModel.appendLine("开始新增页面函数");
    String frontFileEntityStr = JSON.toJSONString(frontFile);
    this.create(frontFileEntityStr, frontFile.getContent());
    importModel.addCreateCount();
  }

  /**
   * 导入-更新页面函数
   * @param frontFile
   * @param dbFrontFile
   * @param importModel
   */
  private void handleUpdateData(FrontFileEntity frontFile, FrontFileEntity dbFrontFile, MigrateImportModel importModel) {
    frontFile.setId(dbFrontFile.getId());
    String frontFileEntityStr = JSON.toJSONString(frontFile);
    this.update(frontFileEntityStr, frontFile.getContent());
    importModel.appendLine("更新成功");
    importModel.addUpdateCount();
  }

  /**
   * 不分页查询
   * @param name
   * @param fileClassify
   * @param fileStatus
   * @return
   */
  @Override
  public List<FrontFileEntity> findAllByConditions(String name, String fileClassify, Integer fileStatus) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (StringUtils.isNotBlank(fileClassify)) {
      conditions.put("fileClassify", fileClassify);
    }
    if (null != fileStatus) {
      conditions.put("fileStatus", fileStatus);
    }
    return frontFileRepository.findAllByConditions(conditions);
  }

  /**
   * 根据文件路径和文件名查询实体
   * @param filePath
   * @param fileName
   * @return
   */
  @Override
  public FrontFileEntity findByFilePathAndFileName(String filePath, String fileName){
    if(StringUtils.isAnyBlank(filePath,fileName)) {
      return null;
    }
    return frontFileRepository.findByFilePathAndFileName(filePath,fileName);
  }
}
