package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.kuiper.entity.InstanceContextEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.starter.repository.InstanceContextRepository;
import com.bizunited.platform.kuiper.starter.service.InstanceContextService;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 表单实例上下文服务层实现
 * @Author: Paul Chan
 * @Date: 2019-04-19 11:37
 * @Description:
 */
@Service("InstanceContextServiceImpl")
public class InstanceContextServiceImpl implements InstanceContextService {
  @Autowired
  private InstanceContextRepository instanceContextRepository;
  @Autowired
  private InstanceService instanceService;
  @Autowired
  private PlatformContext platformContext;

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceContextService#save(java.lang.String, java.util.List)
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public List<InstanceContextEntity> save(String instanceId, List<InstanceContextEntity> contexts){
    Validate.notBlank(instanceId, "instanceId is blank");
    InstanceEntity instance = instanceService.findById(instanceId);
    Validate.notNull(instance, "not found instance");
    if(CollectionUtils.isEmpty(contexts)){
      return findByInstanceId(instanceId);
    }
    List<InstanceContextEntity> oldContexts = this.findByInstanceId(instanceId);
    Set<String> names = new HashSet<>();
    Map<String, InstanceContextEntity> oldContextsMap = oldContexts.stream().collect(Collectors.toMap(InstanceContextEntity::getName, item -> item, (a, b) -> b, () -> new HashMap<>(16)));
    for (InstanceContextEntity context : contexts) {
      Validate.notNull(context.getName(), "name not be null");
      Validate.notNull(context.getValue(), "value not be null");
      Validate.isTrue(!names.contains(context.getName()), "有重复的name值，请检查数据");
      names.add(context.getName());
      InstanceContextEntity oldContext = oldContextsMap.get(context.getName());
      if(oldContext != null){
        oldContext.setValue(context.getValue());
        oldContext.setModifyTime(new Date());
        oldContext.setProjectName(platformContext.getAppName());
        instanceContextRepository.save(oldContext);
        continue;
      }
      context.setCreateTime(new Date());
      context.setModifyTime(new Date());
      context.setInstance(instance);
      context.setProjectName(platformContext.getAppName());
      instanceContextRepository.save(context);
      oldContexts.add(context);
    }
    return oldContexts;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceContextService#findByInstanceId(java.lang.String)
   */
  @Override
  public List<InstanceContextEntity> findByInstanceId(String instanceId){
    if(StringUtils.isBlank(instanceId)){
      return Lists.newArrayList();
    }
    return instanceContextRepository.findByInstanceId(instanceId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceContextService#findByInstanceIdAndName(java.lang.String, java.lang.String)
   */
  @Override
  public InstanceContextEntity findByInstanceIdAndName(String instanceId, String name){
    if(StringUtils.isBlank(instanceId) || StringUtils.isBlank(name)){
      return null;
    }
    return instanceContextRepository.findByInstanceContextIdAndName(instanceId, name);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteByInstanceIdAndName(String instanceId, String name) {
    Validate.notBlank(instanceId, "表单实例ID不能为空");
    Validate.notBlank(name, "name不能为空");
    InstanceContextEntity instanceContext = instanceContextRepository.findByInstanceContextIdAndName(instanceId, name);
    if(instanceContext != null){
      instanceContextRepository.delete(instanceContext);
    }
  }

}
