package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.InstanceViewEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.InstanceRepository;
import com.bizunited.platform.kuiper.starter.repository.internal.InstanceViewRepositoryCustom;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;

/**
 * 表单实例功能的默认服务层实现
 * @author yinwenjie
 */
@Service("InstanceServiceImpl")
public class InstanceServiceImpl implements InstanceService {
  @Autowired
  private InstanceRepository instanceRepository;
  @Autowired
  private InstanceViewRepositoryCustom instanceViewRepositoryCustom;
  @Autowired
  private UserService userService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private TemplateLayoutService templateLayoutService;
  @Autowired
  private InstanceActivityService instanceActivityService;
  @Autowired
  private ServicableMethodService servicableMethodService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(InstanceServiceImpl.class);
  
  @Override
  @Transactional
  public InstanceEntity createByTemplateId(String templateId , String taskCode , String account) {
    /*
     * 创建表单实例的过程为：
     * 1、首先进行传入的instance对象的判定，主要是id、模板等信息的判断
     * 2、构造和创建实例
     * 3、创建实例后需要查找并创建一个当前模板下一个名字为create的可见性(活动实例)，并按照这个可见性创建一个活动
     * */
    // 1、=========
    Validate.notBlank(account , "当前操作者用户信息必须传入");
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未发现操作者信息，请检查!!");
    TemplateEntity currentTemplate = this.templateService.findById(templateId);
    Validate.notNull(currentTemplate , "未找设定的模板信息");
    return this.createInstance(templateId , taskCode , account);
  }

  @Override
  @Transactional
  public InstanceEntity createByTemplateCode(String templateCode, String cversion, String taskCode, String account) {
    Validate.notBlank(templateCode, "未发现指定的模板业务编号");
    Validate.notBlank(cversion, "未发现指定的模板业务版本号");
    Validate.notBlank(account, "当前操作者用户信息必须传入");
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user, "未发现操作者信息，请检查!!");
    TemplateEntity template = templateService.findByCodeAndCversion(templateCode, cversion);
    Validate.notNull(template, "未发现指定的模版");
    return this.createInstance(template.getId(), taskCode, account);
  }

  @Override
  @Transactional
  public InstanceEntity createByTemplateCode(String templateCode, String taskCode , String account) {
    // 1、=========
    Validate.notBlank(templateCode , "未发现指定的模板业务编号");
    List<TemplateEntity> templates = this.templateService.findByCode(templateCode);
    Validate.notEmpty(templates , "未发现指定模板业务编号的任何模板信息");
    Validate.notBlank(account , "当前操作者用户信息必须传入");
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未发现操作者信息，请检查!!");

    if(templates.size() == 1) {
      return this.createInstance(templates.get(0).getId() , taskCode , account);
    }
    // 根据创建时间，拿到当前默认的版本信息
    TemplateEntity currentTemplate;
    currentTemplate = templates.stream().filter(TemplateEntity::getDefaultVersion).findFirst().orElse(null);
    // 如果没有找到默认的版本信息，则使用最新创建时间的版本
    if(currentTemplate == null) {
      currentTemplate = templates.stream().sorted(Comparator.comparing(TemplateEntity::getCreateTime).reversed()).findFirst().orElse(null);
    }
    Validate.notNull(currentTemplate , "未发现任何模板信息，请检查!!");
    return this.createInstance(currentTemplate.getId() , taskCode, account);
  }

  /**
   * @param currentTemplateId 当前模板编号
   * @param taskCode 指定的可能传入的，将要创建的create名称的活动实例的唯一外部业务编号
   * @param account 当前人账号
   * @return
   */
  private InstanceEntity createInstance(String currentTemplateId , String taskCode , String account) {
    // 当前模板的布局信息必须正确
    Validate.notBlank(currentTemplateId , "未找设定的模板编号信息");
    TemplateEntity currentTemplate = this.templateService.findById(currentTemplateId);
    Validate.notNull(currentTemplate ,"未找到模板id对应的模板信息，请检查!!");
    JSONObject templateLayout = templateLayoutService.findDetailsByTemplateId(currentTemplateId , 1);
    Validate.notNull(templateLayout , "当前指定的表单模板，还未确认布局信息，无法创建实例。请检查!!");
    Validate.isTrue(currentTemplate.getTstatus() == 1 , "当前指定模板的状态不正确，请检查（模板状态不能为禁用）!");
    Validate.notBlank(account ,"创建者信息必须传入，请检查!!");
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未发现操作者信息，请检查!!");
    
    // 2、=========
    Date currentTime = new Date();
    InstanceEntity currentInstance = new InstanceEntity();
    currentInstance.setCreateTime(currentTime);
    currentInstance.setCreator(account);
    currentInstance.setDomain(currentTemplate.getDomain());
    currentInstance.setProjectName(currentTemplate.getProjectName());
    currentInstance.setTemplate(currentTemplate);
    currentInstance.setProjectName(platformContext.getAppName());
    this.instanceRepository.save(currentInstance);
    
    // 3、========
    // 创建一个create的活动
    TemplateVisibilityEntity visibility = this.templateVisibilityService.findByTemplateIdAndVisibilityName(currentTemplate.getId(), "create");
    Validate.notNull(visibility , "表单实例参考的模板下，没有名称为“create”的可见性配置(活动)，请检查！！");
    InstanceActivityEntity activity = new InstanceActivityEntity();
    activity.setInstance(currentInstance);
    activity.setCreateTime(currentTime);
    // 需要修改为保存用户的账户名
    activity.setCreator(user.getAccount());
    activity.setModifyer(user.getAccount());
    activity.setModifyTime(currentTime);
    activity.setPreviousActivity(null);
    activity.setTemplate(currentTemplate);
    activity.setTemplateVisibility(visibility);
    // 存入可能的taskCode
    if(!StringUtils.isBlank(taskCode)) {
      activity.setTaskCode(taskCode);
    }
    this.instanceActivityService.create(activity , account);
    currentInstance.setActivities(Arrays.asList(activity));
    return currentInstance;
  }

  @Override
  @Transactional
  public InstanceEntity createByTemplateCode(String templateCode, String taskCode , String account, String serviceName, JSONObject formData, Map<String, Object> params) {
    /*
     * .该方法实际上有两个关键步骤：
     * 1、首先按照templateCode基于指定的模板生成相关的表单实例、表单活动实例信息
     * 注意：如果当前模板是动态模板，则报错，并要求使用动态表单的数据添加服务
     * 2、如果当前模板是静态模板，则进行静态实例的保存调用
     * (注意：本方法中V1.2和V1.1两个版本的处理方式完全不同)
     * */
    Validate.notNull(formData , "传入的表单数据不可能为空!!");
    Validate.notBlank(templateCode , "未发现指定的模板业务编号");
    TemplateEntity template = this.templateService.findByDefaultVersion(templateCode);
    Validate.notNull(template , "未发现指定模板业务编号的默认模板信息，最可能的情况是没有设定默认模板!!");
    Validate.notBlank(account ,"创建者信息必须传入，请检查!!");
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未发现操作者信息，请检查!!");
    // 对服务源信息进行校验
    Validate.notBlank(serviceName, "服务源名称不能为空，请检查!!");
    ServicableMethodInfo servicableMethodInfo = servicableMethodService.findDetailsByName(serviceName);
    Validate.notNull(servicableMethodInfo,"根据服务名%s，没有查询到服务的相关信息，请检查!!" , serviceName);
    
    // 1、=======
    InstanceEntity currentInstance = this.createInstance(template.getCode(), taskCode, account);
    String currentInstanceId = currentInstance.getId();
    List<InstanceActivityEntity> activities = currentInstance.getActivities();
    Validate.notNull(activities , "没有成功创建表单活动实例，请检查!!");
    Validate.isTrue(activities.size() == 1 , "未知的实例初始化错误（创建了多个活动），请检查!!");
    InstanceActivityEntity currentActivity = activities.get(0);
    String currentActivityId = currentActivity.getId();
    TemplateEntity currentTemplate = currentInstance.getTemplate();
    Validate.notNull(currentTemplate , "错误的创建过程，未发现关联的表单模板信息!!");
    String templateId = currentTemplate.getId();
    Validate.notBlank(templateId , "错误的创建过程，未发现关联的表单模板信息!!");
    // 验证模板类型，只能是静态模板
    String templateType = currentTemplate.getType();
    Validate.isTrue(StringUtils.equals(templateType, "static"), "错误的模板类型[%s]，请检查!!" , templateType);
    
    // 2、=======
 // 操作类型（opType）、表单实例id(formInstanceId)、实例活动id（instanceActivityId）、业务表单信息的json结构（formData）、
    InvokeParams invokeParams = new InvokeParams();
    if(params != null) {
      invokeParams.putInvokeParams(params);
    }
    invokeParams.putInvokeParam("opType", "create");
    invokeParams.putInvokeParam("instanceActivityId" , currentActivityId);
    invokeParams.putInvokeParam("instanceId", currentInstanceId);
    if(formData != null) {
      invokeParams.setJsonData(formData);
    }
    try {
      this.servicableMethodService.invoke(serviceName, null,invokeParams);
    } catch (InvokeProxyException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    return currentInstance;
  }

  @Override
  @Transactional
  public void updateTemplate(TemplateEntity template) {
    Validate.notNull(template, "模板对象不能为空！！！");
    Validate.notBlank(template.getCode(), "模板编号不能为空！！！");
    Validate.notBlank(template.getId(), "模版ID不能为空！！！");
    Set<InstanceEntity> instances = instanceRepository.findByTemplateCode(template.getCode());
    if(CollectionUtils.isEmpty(instances)) {
      return;
    }
    for (InstanceEntity instance : instances) {
      instance.setTemplate(template);
      List<InstanceActivityEntity> activities = instance.getActivities();
      if(!CollectionUtils.isEmpty(activities)) {
        instanceActivityService.updateTemplate(activities, template);
      }
      instanceRepository.save(instance);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceService#findById(java.lang.String)
   */
  @Override
  public InstanceEntity findDetailsById(String instanceId) {
    if(StringUtils.isBlank(instanceId)) {
      return null;
    }
    
    return this.instanceRepository.findDetailsById(instanceId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceService#findByConditions(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.util.Date, java.util.Date, java.lang.String, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<InstanceViewEntity> findByConditions(String code, String instanceId, String cversion, String domain, String templateName, String projectName,
                                                   Date beginCreateTime , Date endCreateTime , String userName , Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    conditions.put(PROJECT_NAME, platformContext.getAppName());
    if (StringUtils.isNotBlank(code)) {
      conditions.put("code", code);
    }
    if (StringUtils.isNotBlank(instanceId)) {
      conditions.put("instanceId", instanceId);
    }
    if (StringUtils.isNotBlank(cversion)) {
      conditions.put("cversion", cversion);
    }
    if (StringUtils.isNotBlank(domain)) {
      conditions.put("domain", domain);
    }
    if (StringUtils.isNotBlank(templateName)) {
      conditions.put("templateName", templateName);
    }
    if (beginCreateTime != null) {
      conditions.put("beginCreateTime", beginCreateTime);
    }
    if (endCreateTime != null) {
      conditions.put("endCreateTime", endCreateTime);
    }
    if (StringUtils.isNotBlank(userName)) {
      conditions.put("userName", userName);
    }
    conditions.put("projectName", platformContext.getAppName());
    
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    return instanceViewRepositoryCustom.queryPage(pageable, conditions);
  }

  /**
   * 根据实例id，查询实例基本信息
   */
  @Override
  public InstanceEntity findById(String id) {
    Validate.notBlank(id , "传入的实例id不能为空，请检查!!");
    Optional<InstanceEntity> op = this.instanceRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public Page<InstanceEntity> findByTemplateId(String templateId, Pageable pageable) {
    if(StringUtils.isBlank(templateId)) {
      return new PageImpl<>(new ArrayList<>(), pageable, 0);
    }
    return instanceRepository.findByTemplateId(templateId, pageable);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceService#countByTemplateId(java.lang.String)
   */
  @Override
  public int countByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return 0;
    }
    return this.instanceRepository.countByTemplateId(templateId);
  }
}