package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.starter.repository.ListTemplateRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.Principal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.LIST_TEMPLATE_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * ListTemplateEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("ListTemplateEntityServiceImpl")
public class ListTemplateServiceImpl implements ListTemplateService { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ListTemplateServiceImpl.class);
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private UserService userService;
  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private ListTemplateRepository listTemplateEntityRepository;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 列表模板布局信息文件内容存放的相对路径
   */
  private static final String LAYOUTDIR = "/listTemplate/layout";
  private static final String EXTENDTYPE = ".txt";
  
  @Transactional
  @Override
  public ListTemplateEntity init(ListTemplateEntity listTemplateEntity, Principal principal) {
    String userName = principal.getName();
    Validate.notBlank(userName, "未发现指定的用户名信息");
    UserVo creator = this.userService.findByAccount(userName);
    Validate.notNull(creator, "错误的创建人信息，请检查!!");
    listTemplateEntity.setCreator(userName);
    listTemplateEntity.setProjectName(platformContext.getAppName());
    return this.initForm(listTemplateEntity);
    
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
  }
  
  private ListTemplateEntity initForm(ListTemplateEntity listTemplateEntity) { 
    this.createValidation(listTemplateEntity);
    Long countByCode = this.listTemplateEntityRepository.countByCode(listTemplateEntity.getCode());
    //如果是第一个版本，就设置为默认版本
    listTemplateEntity.setDefaultVersion((null ==countByCode || countByCode == 0L));
    // ===============================
    //  和业务有关的验证填写在这个区域
    listTemplateEntity.setModifyTime(new Date());
    // ===============================
    listTemplateEntity.setProjectName(platformContext.getAppName());
    this.listTemplateEntityRepository.saveAndFlush(listTemplateEntity);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return listTemplateEntity;
  }
  /**
   * 在创建一个新的列表模板模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ListTemplateEntity listTemplate) { 
    Validate.notNull(listTemplate, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(listTemplate.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    listTemplate.setId(null);
    Validate.notBlank(listTemplate.getCode(), "列表模板的唯一编号不能为空！");
    Validate.notBlank(listTemplate.getCversion(), "列表模板版本信息不能为空！");
    Validate.notBlank(listTemplate.getName(), "列表模板的中文说明不能为空！");
    listTemplate.setTstatus(1);
    if(listTemplate.getDataSource() == null) {
      listTemplate.setDataSource(2);
    }
    listTemplate.setCreateTime(new Date());
    //2020/9/11 修改，部署多服务是，
    Set<ListTemplateEntity> exsitCode = this.listTemplateEntityRepository.findByCode(listTemplate.getCode());
    Validate.isTrue(CollectionUtils.isEmpty(exsitCode), "当前指定编号【%s】已存在", listTemplate.getCode());
    // 查询唯一键的可用性
    ListTemplateEntity exsit = this.listTemplateEntityRepository.findByCodeAndCversion(listTemplate.getCode(), listTemplate.getCversion());
    Validate.isTrue(exsit == null, "当前指定编号【%s】，指定版本下【%s】的列表模板信息已存在，请检查" , listTemplate.getCode(), listTemplate.getCversion());
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(listTemplate.getCode() == null || listTemplate.getCode().length() < 128 , "列表模板的唯一编号,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(listTemplate.getProjectName() == null || listTemplate.getProjectName().length() < 128 , "项目名,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(listTemplate.getCversion() == null || listTemplate.getCversion().length() < 64 , "不同的版本信息,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(listTemplate.getName() == null || listTemplate.getName().length() < 128 , "列表模板的中文说明,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(listTemplate.getRelativePath() == null || listTemplate.getRelativePath().length() < 256 , "列表模板文件的相对目录,填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(listTemplate.getFileName() == null || listTemplate.getFileName().length() < 256 , "保存的列表模板布局文件文件名,填入值超过了限定长度(256)，请检查!");
  }
  @Transactional
  @Override
  public ListTemplateEntity update(ListTemplateEntity listTemplateEntity) { 
    return this.updateForm(listTemplateEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
  }
  
  private ListTemplateEntity updateForm(ListTemplateEntity listTemplateEntity) {     
    this.updateValidation(listTemplateEntity);
    // ===================基本信息
    String currentId = listTemplateEntity.getId();
    Optional<ListTemplateEntity> op = this.listTemplateEntityRepository.findById(currentId);
    ListTemplateEntity currentListTemplateEntity = op.orElse(null);
    Validate.notNull(currentListTemplateEntity,"未发现指定的原始模型对象信");

    // 开始重新赋值——一般属性
    currentListTemplateEntity.setCode(listTemplateEntity.getCode());
    currentListTemplateEntity.setProjectName(platformContext.getAppName());
    currentListTemplateEntity.setCversion(listTemplateEntity.getCversion());
    currentListTemplateEntity.setName(listTemplateEntity.getName());
    currentListTemplateEntity.setTstatus(listTemplateEntity.getTstatus());
    currentListTemplateEntity.setDataSource(listTemplateEntity.getDataSource());
    currentListTemplateEntity.setRelativePath(listTemplateEntity.getRelativePath());
    currentListTemplateEntity.setFileName(listTemplateEntity.getFileName());
    currentListTemplateEntity.setModifyTime(new Date());
    this.listTemplateEntityRepository.saveAndFlush(currentListTemplateEntity);
    return currentListTemplateEntity;
  }
  /**
   * 在更新一个已有的ListTemplateEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ListTemplateEntity listTemplateEntity) { 
    Validate.isTrue(!StringUtils.isBlank(listTemplateEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(listTemplateEntity.getCode(), "列表模板的唯一编号不能为空！");
    Validate.notBlank(listTemplateEntity.getCversion(), "不同的版本信息不能为空！");
    Validate.notBlank(listTemplateEntity.getName(), "列表模板的中文说明不能为空！");
    Validate.notNull(listTemplateEntity.getTstatus(), "列表模板状态不能为空！");
    Validate.notNull(listTemplateEntity.getDataSource(), "数据源类型：1、服务源；2、数据视图不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(listTemplateEntity.getCode() == null || listTemplateEntity.getCode().length() < 128 , "列表模板的唯一编号,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(listTemplateEntity.getProjectName() == null || listTemplateEntity.getProjectName().length() < 128 , "项目名,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(listTemplateEntity.getCversion() == null || listTemplateEntity.getCversion().length() < 64 , "不同的版本信息,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(listTemplateEntity.getName() == null || listTemplateEntity.getName().length() < 128 , "列表模板的中文说明,填入值超过了限定长度(128)，请检查!");
  } 
  /**
   * 将object读成bytes内容
   * @param obj
   * @return
   */
  private byte[] writeObjectToBytes(Object obj){
    byte[] bytes;
    try (ByteArrayOutputStream out = new ByteArrayOutputStream();
        ObjectOutputStream sOut = new ObjectOutputStream(out);){
        sOut.writeObject(obj);
        sOut.flush();
        bytes= out.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new byte[] {};
    }
    return bytes;
  }
  /**
   * 将bytes内容读取成json信息
   * @param bytes 文件字节
   * @return
   */
  private JSONObject readBytesToJson(byte[] bytes) {
    Object layoutObject;
    try (ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ObjectInputStream sIn = new ObjectInputStream(in);){
      layoutObject = sIn.readObject();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return null;
    }
    if(layoutObject == null) {
      return null;
    }
    
    return JSON.parseObject(JSON.toJSONString(layoutObject));
  }
  @Override
  @Transactional
  public ListTemplateEntity updateContentById(String id, JSONObject content) {
    Validate.notBlank(id, "更新列表模板内容时，列表模板编号必须传入!!");
    Validate.notNull(content, "更新列表模板内容时，JSON结构的内容本身必须传入");
    
    // 确定文件名、文件保存的相对路径等信息
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(id, EXTENDTYPE);
    byte[] objectContents = writeObjectToBytes(content);
    this.venusFileService.saveFile(relativePath, fileName, fileName, objectContents);
    
    // 修改文件信息在数据库中的描述
    Optional<ListTemplateEntity> op = this.listTemplateEntityRepository.findById(id);
    ListTemplateEntity listTemplate = op.orElse(null);
    Validate.notNull(listTemplate, "错误的列表模板编号信息，请检查");

    listTemplate.setFileName(fileName);
    listTemplate.setRelativePath(relativePath);
    listTemplate.setModifyTime(new Date());
    this.listTemplateEntityRepository.saveAndFlush(listTemplate);
    return listTemplate;
  }

  @Override
  public JSONObject findContentById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    Optional<ListTemplateEntity> op = this.listTemplateEntityRepository.findById(id);
    Validate.isTrue(op.isPresent(), "错误的列表模板编号信息，请检查!");
    
    // 读取文件内容
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(id, EXTENDTYPE);
    byte[] contentBytes = this.venusFileService.readFileContent(relativePath, fileName);
    Validate.notNull(contentBytes, "未读取到指定列表模板的布局描述信息[%s]，请检查!" , id);
    return this.readBytesToJson(contentBytes);
  }

  @Override
  public ListTemplateEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.listTemplateEntityRepository.findDetailsById(id);
  }
  @Override
  public Set<ListTemplateEntity> findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return Sets.newHashSet();
    }
    return this.listTemplateEntityRepository.findByCode(code);
  }
  @Override
  public ListTemplateEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    Optional<ListTemplateEntity> op = this.listTemplateEntityRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  public Page<ListTemplateEntity> findByConditions(Pageable pageable, Map<String, Object> conditions) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    if(conditions == null) {
      conditions = new HashMap<>();
    }
    conditions.put(PROJECT_NAME, platformContext.getAppName());
    return this.listTemplateEntityRepository.findByConditions(pageable, conditions);
  }
  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.listTemplateEntityRepository.countByIds(ids);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<ListTemplateEntity> op = this.listTemplateEntityRepository.findById(id);
    op.ifPresent(listTemplateEntity -> this.listTemplateEntityRepository.delete(listTemplateEntity));
  }

  @Override
  public JSONObject findContentByCodeAndCversion(String code, String cversion) {
    if (StringUtils.isAnyBlank(code,cversion)){
      return null;
    }
    ListTemplateEntity templateEntity = this.listTemplateEntityRepository.findDetailsByCodeAndCversion(code, cversion);
    Validate.notNull(templateEntity, "错误的列表模板编号信息，请检查!!");

    // 读取文件内容
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(templateEntity.getId(), EXTENDTYPE);
    byte[] contentBytes = this.venusFileService.readFileContent(relativePath, fileName);
    Validate.notNull(contentBytes, "未读取到指定列表模板的布局描述信息[%s]，请检查", code);
    return this.readBytesToJson(contentBytes);
  }

  @Override
  public ListTemplateEntity findByCodeAndCversion(String code, String cversion){
    if (StringUtils.isAnyBlank(code,cversion)){
      return null;
    }
    return listTemplateEntityRepository.findDetailsByCodeAndCversion(code, cversion);
  }

  @Override
  @Transactional
  public ListTemplateEntity updateContentByCodeAndCversion(String code, String cversion, JSONObject content) {
    Validate.notBlank(code, "更新列表模板内容时，列表模板编码必须传入!!");
    Validate.notBlank(cversion, "更新列表模板内容时，列表模板版本号必须传入!!");
    Validate.notNull(content, "更新列表模板内容时，JSON结构的内容本身必须传入!!");

    // 修改文件信息在数据库中的描述
    ListTemplateEntity templateEntity = this.listTemplateEntityRepository.findDetailsByCodeAndCversion(code, cversion);
    Validate.notNull(templateEntity, "错误的列表模板编号信息，请检查!!");

    // 确定文件名、文件保存的相对路径等信息
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(templateEntity.getId(), EXTENDTYPE);
    byte[] objectContents = writeObjectToBytes(content);
    this.venusFileService.saveFile(relativePath, fileName, fileName, objectContents);

    templateEntity.setFileName(fileName);
    templateEntity.setRelativePath(relativePath);
    this.listTemplateEntityRepository.saveAndFlush(templateEntity);
    return templateEntity;
  }

  @Override
  public JSONObject findDefaultContentByCode(String code) {
    if (StringUtils.isBlank(code)){
      return null;
    }
    ListTemplateEntity templateEntity = this.listTemplateEntityRepository.findDefaultByCode(code);
    Validate.notNull(templateEntity, "没有找到当前列表模板的默认版本，请设置默认版本!!");

    // 读取文件内容
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(templateEntity.getId(), EXTENDTYPE);
    byte[] contentBytes = this.venusFileService.readFileContent(relativePath, fileName);
    Validate.notNull(contentBytes, "未读取到指定列表模板的布局描述信息[%s]，请检查", code);
    return this.readBytesToJson(contentBytes);
  }

  @Override
  @Transactional
  public void updateDefaultVersion(String code, String cversion) {
    Validate.notBlank(code, "更新列表模板内容时，列表模板编码必须传入!!");
    Validate.notBlank(cversion, "更新列表模板内容时，列表模板版本号必须传入!!");
    ListTemplateEntity detailsByCode = this.listTemplateEntityRepository.findDetailsByCodeAndCversion(code, cversion);

    Validate.notNull(detailsByCode, "未找到指定的列表模板信息，请检查!!");
    Validate.isTrue(detailsByCode.getTstatus() == 1, "当前指定模板的状态不正确，请检查（模板状态不能为禁用）!");

    //把当前code对应的所有版本先置为非默认版本0
    this.listTemplateEntityRepository.updateDefaultVersion(code, Boolean.FALSE);
    this.listTemplateEntityRepository.updateDefaultVersion(code, cversion, Boolean.TRUE, new Date());
  }

  @Override
  public ListTemplateEntity findDefaultByCode(String code) {
    if (StringUtils.isBlank(code)){
      return null;
    }
    return this.listTemplateEntityRepository.findDefaultByCode(code);
  }

  @Override
  public List<ListTemplateEntity> findAllByConditions(ListTemplateEntity condition) {
    return listTemplateEntityRepository.findAllByConditions(condition);
  }

  @Override
  public List<ListTemplateEntity> findByIds(String[] ids) {
    if(ArrayUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    return listTemplateEntityRepository.findByIds(ids);
  }

  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry templateEntry = zipFile.getEntry(LIST_TEMPLATE_FILENAME);
    if(templateEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
      return;
    }
    try (InputStream is = zipFile.getInputStream(templateEntry);
         ObjectInputStream ois = new ObjectInputStream(is)) {
      List<ListTemplateEntity> templates = (List<ListTemplateEntity>) ois.readObject();
      if(CollectionUtils.isEmpty(templates)) {
        importModel.appendLine("导入的数据为空，请检查数据");
        return;
      }
      this.importData(templates, importModel);
    } catch (IOException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      importModel.append("读取模版数据失败：").appendLine(e.getMessage());
    }
  }

  /**
   * 批量导入模版数据
   * @param templates
   * @param importModel
   */
  private void importData(List<ListTemplateEntity> templates, MigrateImportModel importModel) {
    importModel.setTotalCount(templates.size());
    for (int i = 0; i < templates.size(); i++) {
      ListTemplateEntity template = templates.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(template, importModel);
    }
  }

  /**
   * 导入单条模版数据
   * @param template
   * @param importModel
   */
  private void importData(ListTemplateEntity template, MigrateImportModel importModel) {
    importModel.appendLine(String.format("开始导入数据：编号=%s,版本号=%s", template.getCode(), template.getCversion()));
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    ListTemplateEntity dbTemplate = listTemplateEntityRepository.findByCodeAndCversion(template.getCode(), template.getCversion());
    template.setId(null);
    if(dbTemplate != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("模版数据已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbTemplate != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("模版数据已存在，执行更新导入");
      this.handleUpdateData(template, dbTemplate, importModel);
      return;
    }
    if(dbTemplate != null && ImportExecuteModeEnum.ADD == executeMode) {
      this.handleCreateDataOnExist(template, importModel);
      return;
    }
    if(dbTemplate == null) {
      this.handleCreateData(template, importModel);
      return;
    }
  }

  /**
   * 处理系统已存在的新增数据
   * @param template
   * @param importModel
   * @return
   */
  private ListTemplateEntity handleCreateDataOnExist(ListTemplateEntity template, MigrateImportModel importModel) {
    importModel.appendLine("模版数据已存在，执行版本升级导入");
    String newVersion;
    int count = 1;
    long versionCount;
    do {
      newVersion = StringUtils.join(template.getCversion(), "_", count);
      count++;
      versionCount = listTemplateEntityRepository.countByCodeAndVersion(template.getCode(), newVersion);
    } while (versionCount > 0);
    importModel.append("生成新的版本号：").appendLine(newVersion);
    importModel.appendLine("导入模版数据");
    template.setCversion(newVersion);
    return this.handleCreateData(template, importModel);
  }

  /**
   * 处理新增数据
   * @param template
   * @param importModel
   * @return
   */
  private ListTemplateEntity handleCreateData(ListTemplateEntity template, MigrateImportModel importModel) {
    importModel.appendLine("导入新增模版数据：编号=%s，版本=%s", template.getCode(), template.getCversion());
    template.setDefaultVersion(false);
    Principal principal = SecurityUtils.getPrincipal();
    ListTemplateEntity dbTemplate = this.init(template, principal);
    this.importTemplateFile(template, dbTemplate, importModel);
    importModel.appendLine("导入新增模版成功");
    importModel.addCreateCount();
    return dbTemplate;
  }

  /**
   * 处理更新的数据
   * @param template
   * @param dbTemplate
   * @param importModel
   */
  private ListTemplateEntity handleUpdateData(ListTemplateEntity template, ListTemplateEntity dbTemplate, MigrateImportModel importModel) {
    template.setId(dbTemplate.getId());
    dbTemplate = this.update(template);
    this.importTemplateFile(template, dbTemplate, importModel);
    importModel.appendLine("更新模版成功");
    importModel.addUpdateCount();
    return dbTemplate;
  }

  /**
   * 导入列表模版文件
   * @param template
   * @param importModel
   */
  private void importTemplateFile(ListTemplateEntity template, ListTemplateEntity dbTemplate, MigrateImportModel importModel) {
    try {
      importModel.appendLine("获取模版文件数据");
      byte[] bytes = ZipFileUtils.readZipFile(importModel.getZipFile(), template.getRelativePath(), template.getFileName());
      Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "模版文件内容为空，请检查数据");
      JSONObject json = this.readBytesToJson(bytes);
      importModel.appendLine("导入模版文件数据");
      this.updateContentById(dbTemplate.getId(), json);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException("读取模版文件失败", e);
    }
  }

}
