package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.core.service.NebulaStaticPersistentService;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateEventEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateLayoutEntity;
import com.bizunited.platform.kuiper.entity.TemplateMaintainerEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityAttributesEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityButtonsEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum;
import com.bizunited.platform.kuiper.starter.model.StaticTemplateMergeModel;
import com.bizunited.platform.kuiper.starter.repository.TemplateGroupRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateItemRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplatePropertyRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRelationRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityAttributesRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateVisibilityButtonRepository;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.kuiper.starter.service.StaticTemplateMergeService;
import com.bizunited.platform.kuiper.starter.service.TemplateEventService;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.kuiper.starter.service.TemplateMaintainerService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.saturn.model.PersistentClass;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Service
public class StaticTemplateMergeServiceImpl implements StaticTemplateMergeService {

  @Autowired
  private TemplateService templateService;
  @Autowired
  private UserService userService;
  @Autowired
  private NebulaStaticPersistentService nebulaStaticPersistentService;
  @Autowired
  private TemplateRepository templateRepository;
  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  @Autowired
  private TemplatePropertyRepository templatePropertyRepository;
  @Autowired
  private TemplateRelationRepository templateRelationRepository;
  @Autowired
  private TemplateItemRepository templateItemRepository;
  @Autowired
  private TemplateGroupRepository templateGroupRepository;
  @Autowired
  private TemplateLayoutService templateLayoutService;
  @Autowired
  private TemplateEventService templateEventService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private TemplateVisibilityAttributesRepository templateVisibilityAttributesRepository;
  @Autowired
  private TemplateVisibilityButtonRepository templateVisibilityButtonRepository;
  @Autowired
  private TemplateMaintainerService templateMaintainerService;
  @Autowired
  private PlatformContext platformContext;

  /** 扩展字段标识正则1 */
  private static final String EXTEND_PATTERN = "extend\\d+";
  /** 扩展字段标识正则2 */
  private static final String EXT_PATTERN = "ext\\d+";
  /** 表单级别的事件，这些事件的名称是固定的。主要用于表单层级的操作 */
  private static final Set<String> INNER_EVENT_NAME = Sets.newHashSet("onLoad","beforeSubmit","afterSubmit");

  /**
   * 合并逻辑
   * 1.首先进行基础的验证并构造必要信息
   * 2.构造新模型信息，含基础属性、一般属性、关联属性、分组属性、明细属性合并
   * 3.布局信息合并
   * 4.事件信息合并
   * 5.可见性合并
   * 6.绑定表单维护人员
   * 注意：合并信息以基准表单模型为准，设基准表单模型信息为A，待合并表单模型为B.
   * （表单模型信息为：基础属性、一般属性、关联属性、分组属性、明细属性、布局、事件、可见性）
   * 属性又分为：基准字段、扩展字段
   * ============================
   * 当某基准字段A中有，B中无时，以A为准
   * 当某基准字段A中无，B中有时，B的信息覆盖A
   * 当某基准字段A中有，B中有时，B的信息覆盖A
   * 当某基准字段A中无，B中无时，不做变动
   * ============================
   * 当某扩展字段A中有，B中无时，以A为准
   * 当某扩展字段A中无，B中有时，B的信息覆盖到A，且可见性隐藏
   * 当某扩展字段A中有，B中有时，以A为准
   * 当某扩展字段A中无，B中无时，不做变动
   */
  @Override
  @Transactional
  public TemplateEntity merge(StaticTemplateMergeModel model) {
    // 1、=======基础验证
    this.validate(model);
    // 2、=======构造新模型信息，含基础属性、一般属性、关联属性、分组属性、明细属性合并
    //表单基础属性
    this.buildTemplateProperties(model);
    //一般属性
    Set<TemplatePropertyEntity> refProperties = model.getReferenceTemplate().getProperties();
    Set<TemplatePropertyEntity> mergedProperties = model.getMergedTemplate().getProperties();
    this.processProerties(model,refProperties,mergedProperties,model.getReferenceTemplate().getPersistentClassName(),model.getNewTemplate());
    //关联属性
    Set<TemplateRelationEntity> refRelations = model.getReferenceTemplate().getRelations();
    Set<TemplateRelationEntity> mergedRelations = model.getMergedTemplate().getRelations();
    if(!(CollectionUtils.isEmpty(refRelations) && CollectionUtils.isEmpty(mergedRelations))){
      this.processRelations(model,refRelations,mergedRelations,model.getReferenceTemplate().getPersistentClassName(),model.getNewTemplate());
    }
    //明细属性
    Set<TemplateItemEntity> refItems = model.getReferenceTemplate().getItemRelations();
    Set<TemplateItemEntity> mergedItems = model.getMergedTemplate().getItemRelations();
    if(!(CollectionUtils.isEmpty(refItems) && CollectionUtils.isEmpty(mergedItems))){
      this.processItems(model,refItems,mergedItems,model.getReferenceTemplate().getPersistentClassName(),model.getNewTemplate());
    }
    //分组属性
    Set<TemplateGroupEntity> refGroups = model.getReferenceTemplate().getGroupRelations();
    Set<TemplateGroupEntity> mergedGroups = model.getMergedTemplate().getGroupRelations();
    if(!(CollectionUtils.isEmpty(refGroups) && CollectionUtils.isEmpty(mergedGroups))){
      this.processGroup(model,refGroups,mergedGroups,model.getReferenceTemplate().getPersistentClassName(),model.getNewTemplate());
    }
    // 3、=======布局信息合并
    this.processTemplateLayouts(model);
    // 4、=======事件信息合并
    this.processEvents(model);
    // 5、=======可见性合并
    this.processVisibilities(model);
    // 6、=======绑定表单维护人员
    Set<TemplateMaintainerEntity> maintainers = templateMaintainerService.findByTemplateId(model.getMergedTemplateId());
    if(!CollectionUtils.isEmpty(maintainers)) {
      String[] accounts = maintainers.stream().map(TemplateMaintainerEntity::getUserAccount).collect(Collectors.toList()).toArray(new String[maintainers.size()]);
      templateMaintainerService.binding(model.getNewTemplate().getId(), accounts);
    }
    return model.getNewTemplate();
  }


  private void validate(StaticTemplateMergeModel model){
    Validate.notBlank(model.getReferenceTemplateId(),"基准表单模板id不能为空，请检查!!");
    Validate.notBlank(model.getMergedTemplateId(),"待合并表单模板id不能为空，请检查!!");
    Validate.notBlank(model.getNewTemplateName(),"新表单模板名称不能为空，请检查!!");
    Validate.notBlank(model.getNewTemplateVersion(),"新表单模板版本不能为空，请检查!!");
    Validate.isTrue(!StringUtils.equals(model.getReferenceTemplateId(),model.getMergedTemplateId()),"基准表单模板id和待合并表单模板id不能相同，请检查!!");
    //查询模板详情信息
    TemplateEntity refTemplateEntity = templateService.findDetailsById(model.getReferenceTemplateId());
    Validate.notNull(refTemplateEntity,"根据基准表单模板id【%s】，未能获取到表单信息，请检查!!",model.getReferenceTemplateId());
    TemplateEntity templateEntity = templateService.findDetailsById(model.getMergedTemplateId());
    Validate.notNull(templateEntity,"根据待合并表单模板id【%s】，未能获取到表单信息，请检查!!",model.getMergedTemplateId());
    //边界验证
    Validate.isTrue(StringUtils.equals(refTemplateEntity.getType() , FormTemplateTypeEnum.STATIC.getType()) &&
            StringUtils.equals(templateEntity.getType() , FormTemplateTypeEnum.STATIC.getType()), "目前本接口只支持基于静态模型的静态模板合并");
    Validate.isTrue(StringUtils.equals(refTemplateEntity.getPersistentClassName(),templateEntity.getPersistentClassName()),"合并表单模板的模型描述名称必须一致");
    Validate.isTrue(StringUtils.equals(refTemplateEntity.getCode(),templateEntity.getCode()),"合并表单模板的编码必须一致");
    Validate.isTrue(!StringUtils.equals(refTemplateEntity.getCversion(),templateEntity.getCversion()),"合并表单模板的版本不能相同");
    Validate.isTrue(refTemplateEntity.getTstatus() == 1 && templateEntity.getTstatus() == 1,"基准表单模板 或 待合并模板的状态不是启用状态，请检查!!");
    //验证重复性
    TemplateEntity entity = templateService.findByCodeAndCversion(refTemplateEntity.getCode(),model.getNewTemplateVersion());
    Validate.isTrue(entity == null ,"设定的合并表单模板版本编号【%s】重复，请检查!!",model.getNewTemplateVersion());
    //检查操作人信息
    Validate.notBlank(model.getAccount() , "必须指定操作者信息!!");
    UserVo user = userService.findByAccount(model.getAccount());
    Validate.notNull(user , "未找到任何创建者信息，请检查!!");
    //检查
    PersistentClass currentPersistentClass = nebulaStaticPersistentService.findByPersistentClass(refTemplateEntity.getPersistentClassName());
    Validate.notNull(currentPersistentClass , "没有找到当前静态模型描述，请检查!!");

    //设置merge功能所需的所有数据
    model.setReferenceTemplate(refTemplateEntity);
    model.setMergedTemplate(templateEntity);
    model.setCurrentPersistentClass(currentPersistentClass);
  }


  /** 构建表单模板基础属性数据*/
  private void buildTemplateProperties(StaticTemplateMergeModel model){
    TemplateEntity newTemplate = new TemplateEntity();
    model.setNewTemplate(newTemplate);
    TemplateEntity refTemplateEntity = model.getReferenceTemplate();
    newTemplate.setCode(refTemplateEntity.getCode());
    newTemplate.setCreateTime(new Date());
    newTemplate.setCreator(model.getAccount());
    newTemplate.setCversion(model.getNewTemplateVersion());
    newTemplate.setDomain(refTemplateEntity.getDomain());
    newTemplate.setFormStyle(refTemplateEntity.getFormStyle());
    newTemplate.setName(model.getNewTemplateName());
    newTemplate.setPersistentClassName(refTemplateEntity.getPersistentClassName());
    newTemplate.setProjectName(platformContext.getAppName());
    newTemplate.setRepositoryEntity(refTemplateEntity.getRepositoryEntity());
    newTemplate.setTstatus(refTemplateEntity.getTstatus());
    newTemplate.setType(refTemplateEntity.getType());
    newTemplate.setModifyTime(new Date());
    newTemplate.setProjectName(platformContext.getAppName());
    templateRepository.saveAndFlush(newTemplate);
  }

  /** 抽象集合交集*/
  private <E,R> Set<R> intersections(Collection<E> refCollections , Collection<E> mergedCollections , Function<E,R> uniqueKeyFunc){
    Set<R> refInfos = refCollections.stream().map(uniqueKeyFunc).collect(Collectors.toSet());
    Set<R> infos = mergedCollections.stream().map(uniqueKeyFunc).collect(Collectors.toSet());
    return Sets.intersection(refInfos,infos);
  }

  /** 抽象集合差集，正向和反向*/
  private <E,R> Set<R> differences(Collection<E> refCollections , Collection<E> mergedCollections , Function<E,R> uniqueKeyFunc , boolean refMerged){
    Set<R> refInfos = refCollections.stream().map(uniqueKeyFunc).collect(Collectors.toSet());
    Set<R> infos = mergedCollections.stream().map(uniqueKeyFunc).collect(Collectors.toSet());
    return refMerged ? Sets.difference(refInfos,infos) : Sets.difference(infos,refInfos);
  }

  /** 处理表单模板一般属性字段，不含扩展字段*/
  private Set<TemplatePropertyEntity> processBaseFields(StaticTemplateMergeModel model,Set<TemplatePropertyEntity> refProperties,Set<TemplatePropertyEntity> mergedProperties,String persistentClassName,Object refObject){
    this.buildPropertiesMap(model,refProperties,mergedProperties,persistentClassName);
    Set<TemplatePropertyEntity> sets = Sets.newHashSet();
    Set<String> collections = Sets.union(model.getIntersections(),model.getMergedRefDiff());
    //当某基准字段A中有，B中有时，B的信息覆盖A
    //当某基准字段A中无，B中有时，B的信息覆盖A
    for(String key : collections){
      TemplatePropertyEntity mergedProerty = model.getMergedMap().get(key);
      this.validateField(key,model.getPersistentPropertyMap().get(key),mergedProerty);
      TemplatePropertyEntity newTemplatePropertyEntity = kuiperToolkitService.copyObjectByWhiteList(mergedProerty,TemplatePropertyEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem");
      newTemplatePropertyEntity.setId(null);
      this.setTemplatePropertesRefObject(refObject,newTemplatePropertyEntity);
      sets.add(newTemplatePropertyEntity);
      this.mergeRecord(model,refObject,key,model.getMergedRefDiff().contains(key));
    }
    //当某基准字段A中有，B中无时，以A为准
    for(String key : model.getRefMergedDiff()){
      TemplatePropertyEntity refProerty = model.getRefMap().get(key);
      this.validateField(key,model.getPersistentPropertyMap().get(key),refProerty);
      TemplatePropertyEntity newTemplatePropertyEntity = kuiperToolkitService.copyObjectByWhiteList(refProerty,TemplatePropertyEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem");
      newTemplatePropertyEntity.setId(null);
      this.setTemplatePropertesRefObject(refObject,newTemplatePropertyEntity);
      sets.add(newTemplatePropertyEntity);
    }
    return sets;
  }

  /** 处理表单模板一般属性字段，只含有扩展字段*/
  private Set<TemplatePropertyEntity> processExtFields(StaticTemplateMergeModel model,Set<TemplatePropertyEntity> refProperties,Set<TemplatePropertyEntity> mergedProperties,String persistentClassName,Object refObject){
    this.buildPropertiesMap(model,refProperties,mergedProperties,persistentClassName);
    Set<TemplatePropertyEntity> sets = Sets.newHashSet();
    //当某扩展字段A中有，不管B中有没有，都以A为准
    model.getIntersections().addAll(model.getRefMergedDiff());
    for(String key : model.getIntersections()){
      TemplatePropertyEntity refProerty = model.getRefMap().get(key);
      this.validateField(key,model.getPersistentPropertyMap().get(key),refProerty);
      TemplatePropertyEntity newTemplatePropertyEntity = kuiperToolkitService.copyObjectByWhiteList(refProerty,TemplatePropertyEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem");
      newTemplatePropertyEntity.setId(null);
      this.setTemplatePropertesRefObject(refObject,newTemplatePropertyEntity);
      sets.add(newTemplatePropertyEntity);
    }
    //当某基准字段A中无，B中有时，B的信息覆盖A
    for(String key : model.getMergedRefDiff()){
      TemplatePropertyEntity mergedProerty = model.getMergedMap().get(key);
      this.validateField(key,model.getPersistentPropertyMap().get(key),mergedProerty);
      TemplatePropertyEntity newTemplatePropertyEntity = kuiperToolkitService.copyObjectByWhiteList(mergedProerty,TemplatePropertyEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem");
      newTemplatePropertyEntity.setId(null);
      this.setTemplatePropertesRefObject(refObject,newTemplatePropertyEntity);
      sets.add(newTemplatePropertyEntity);
      this.mergeRecord(model,refObject,key,true);
    }
    return sets;
  }


  private void buildPropertiesMap(StaticTemplateMergeModel model,Set<TemplatePropertyEntity> refProperties,Set<TemplatePropertyEntity> mergedProperties,String persistentClassName){
    Map<String,TemplatePropertyEntity> refMap = refProperties.stream().collect(Collectors.toMap(TemplatePropertyEntity::getPropertyName,e -> e));
    Map<String,TemplatePropertyEntity> mergedMap = mergedProperties.stream().collect(Collectors.toMap(TemplatePropertyEntity::getPropertyName,e -> e));
    PersistentClass currentPersistentClass = nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(currentPersistentClass , "没有找到当前静态模型描述，请检查!!");
    Map<String, PersistentProperty> persistentClassMap = currentPersistentClass.getProperties().stream().collect(Collectors.toMap(PersistentProperty::getPropertyName,e -> e));
    model.setMaps(refMap,mergedMap,persistentClassMap);
  }

  /** 根据当前值，设置表单模型中一般属性的当前关联关系*/
  private void setTemplatePropertesRefObject(Object refObject,TemplatePropertyEntity propertyEntity){
    if(refObject instanceof TemplateEntity){
      propertyEntity.setCurrentTemplate((TemplateEntity) refObject);
    }else if(refObject instanceof TemplateGroupEntity){
      propertyEntity.setCurrentGroup((TemplateGroupEntity) refObject);
    }else if(refObject instanceof TemplateItemEntity){
      propertyEntity.setCurrentItem((TemplateItemEntity) refObject);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 根据当前值，设置表单模型中ManyToOne或ManyToMany属性的当前关联关系*/
  private void setTemplateRelationRefObject(Object refObject,TemplateRelationEntity relationEntity){
    if(refObject instanceof TemplateEntity){
      relationEntity.setCurrentTemplate((TemplateEntity) refObject);
    }else if(refObject instanceof TemplateGroupEntity){
      relationEntity.setCurrentGroup((TemplateGroupEntity) refObject);
    }else if(refObject instanceof TemplateItemEntity){
      relationEntity.setCurrentItem((TemplateItemEntity) refObject);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 根据当前值，设置表单模型中明细OneToMany属性的当前关联关系*/
  private void setTemplateItemRefObject(Object refObject,TemplateItemEntity itemEntity){
    if(refObject instanceof TemplateEntity){
      itemEntity.setParentTemplate((TemplateEntity) refObject);
    }else if(refObject instanceof TemplateGroupEntity){
      itemEntity.setParentGroup((TemplateGroupEntity) refObject);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 根据当前值，设置表单模型中分组OneToOne属性的当前关联关系*/
  private void setTemplateGroupRefObject(Object refObject,TemplateGroupEntity groupEntity){
    if(refObject instanceof TemplateEntity){
      groupEntity.setParentTemplate((TemplateEntity) refObject);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 处理集合的交集、正反向差集，并临时存储在上下文。这些属性会多次复用*/
  private void proertiesFields(StaticTemplateMergeModel model,Set<String> intersections , Set<String> refMergedDiff , Set<String> mergedRefDiff , Predicate<String> fieldFunc){
    Set<String> intersectionFields;
    Set<String> refMergedDiffFields;
    Set<String> mergedRefDiffFields;
    if(fieldFunc == null){
      model.setSets(intersections,refMergedDiff,mergedRefDiff);
    }else{
      intersectionFields = intersections.stream().filter(fieldFunc).collect(Collectors.toSet());
      refMergedDiffFields = refMergedDiff.stream().filter(fieldFunc).collect(Collectors.toSet());
      mergedRefDiffFields = mergedRefDiff.stream().filter(fieldFunc).collect(Collectors.toSet());
      model.setSets(intersectionFields,refMergedDiffFields,mergedRefDiffFields);
    }
  }



  private void validateField(String key , PersistentProperty persistentProperty,TemplatePropertyEntity templatePropertyEntity){
    Validate.notNull(persistentProperty,"字段【%s】未能在当前模型描述中找到，请检查!!",key);
    Validate.isTrue(StringUtils.equals(templatePropertyEntity.getPropertyClassName(),persistentProperty.getPropertyClass()),"表单模型【%s】字段类型与当前模型描述类信息不匹配",key);
    Validate.isTrue(StringUtils.equals(templatePropertyEntity.getPropertyDbName(),persistentProperty.getPropertyDbName()),"表单模型【%s】字段类型与当前模型描述数据库字段信息不匹配",key);
  }

  private void validateField(String key , PersistentRelation persistentRelation,TemplateRelationEntity templateRelationEntity){
    Validate.notNull(persistentRelation,"字段【%s】未能在当前模型描述中找到，请检查!!",key);
    Validate.isTrue(StringUtils.equals(templateRelationEntity.getPropertyClassName(),persistentRelation.getPropertyClass()),"表单模型【%s】字段类型与当前模型描述类信息不匹配",key);
  }

  private void validateField(String key , PersistentRelation persistentRelation,TemplateItemEntity templateItemEntity){
    Validate.notNull(persistentRelation,"字段【%s】未能在当前模型描述中找到，请检查!!",key);
    Validate.isTrue(StringUtils.equals(templateItemEntity.getPropertyClassName(),persistentRelation.getPropertyClass()),"表单模型【%s】字段类型与当前模型描述类信息不匹配",key);
  }

  private void validateField(String key , PersistentRelation persistentRelation,TemplateGroupEntity templateGroupEntity){
    Validate.notNull(persistentRelation,"字段【%s】未能在当前模型描述中找到，请检查!!",key);
    Validate.isTrue(StringUtils.equals(templateGroupEntity.getPropertyClassName(),persistentRelation.getPropertyClass()),"表单模型【%s】字段类型与当前模型描述类信息不匹配",key);
  }

  /** 处理表单关联属性的字段*/
  private Set<TemplateRelationEntity> processRelationsFields(StaticTemplateMergeModel model , Set<TemplateRelationEntity> refRelations , Set<TemplateRelationEntity> mergedRelations , String persistentClassName,Object refObject){
    Map<String,TemplateRelationEntity> refMap = refRelations.stream().collect(Collectors.toMap(TemplateRelationEntity::getPropertyName,e -> e));
    Map<String,TemplateRelationEntity> mergedMap = mergedRelations.stream().collect(Collectors.toMap(TemplateRelationEntity::getPropertyName,e -> e));
    PersistentClass currentPersistentClass = nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(currentPersistentClass , "没有找到当前静态模型描述，请检查!!");
    Map<String, PersistentRelation> persistentClassMap = currentPersistentClass.getRelations().stream().collect(Collectors.toMap(PersistentRelation::getPropertyName,e -> e));
    Set<TemplateRelationEntity> sets = Sets.newHashSet();
    Set<String> collections = Sets.union(model.getIntersections(),model.getMergedRefDiff());
    //当某基准字段A中有，B中有时，B的信息覆盖A
    //当某基准字段A中无，B中有时，B的信息覆盖A
    for(String key : collections){
      TemplateRelationEntity mergedRelation = mergedMap.get(key);
      this.validateField(key,persistentClassMap.get(key),mergedRelation);
      TemplateRelationEntity newTemplateRelationEntity = kuiperToolkitService.copyObjectByWhiteList(mergedRelation,TemplateRelationEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem","currentView");
      newTemplateRelationEntity.setId(null);
      this.setTemplateRelationRefObject(refObject,newTemplateRelationEntity);
      sets.add(newTemplateRelationEntity);
      this.mergeRecord(model,refObject,key,model.getMergedRefDiff().contains(key));
    }
    //当某基准字段A中有，B中无时，以A为准
    for(String key : model.getRefMergedDiff()){
      TemplateRelationEntity refRelation = refMap.get(key);
      this.validateField(key,persistentClassMap.get(key),refRelation);
      TemplateRelationEntity newTemplateRelationEntity = kuiperToolkitService.copyObjectByWhiteList(refRelation,TemplateRelationEntity.class, HashSet.class, ArrayList.class,"currentTemplate","currentGroup","currentItem","currentView");
      newTemplateRelationEntity.setId(null);
      this.setTemplateRelationRefObject(refObject,newTemplateRelationEntity);
      sets.add(newTemplateRelationEntity);
    }
    return sets;
  }

  /** 处理表单明细属性的字段*/
  private Set<TemplateItemEntity> processItemFields(StaticTemplateMergeModel model , Set<TemplateItemEntity> refItems , Set<TemplateItemEntity> mergedItems , String persistentClassName,Object refObject){
    Map<String,TemplateItemEntity> refMap = refItems.stream().collect(Collectors.toMap(TemplateItemEntity::getPropertyName,e -> e));
    Map<String,TemplateItemEntity> mergedMap = mergedItems.stream().collect(Collectors.toMap(TemplateItemEntity::getPropertyName,e -> e));
    PersistentClass currentPersistentClass = nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(currentPersistentClass , "没有找到当前静态模型描述，请检查!!");
    Map<String, PersistentRelation> persistentClassMap = currentPersistentClass.getRelations().stream().collect(Collectors.toMap(PersistentRelation::getPropertyName,e -> e));
    Set<TemplateItemEntity> sets = Sets.newHashSet();
    Set<String> collections = Sets.union(model.getIntersections(),model.getMergedRefDiff());
    //当某基准字段A中有，B中有时，B的信息覆盖A
    //当某基准字段A中无，B中有时，B的信息覆盖A
    for(String key : collections){
      TemplateItemEntity mergedItem = mergedMap.get(key);
      this.validateField(key,persistentClassMap.get(key),mergedItem);
      TemplateItemEntity newTemplateItemEntity = kuiperToolkitService.copyObjectByWhiteList(mergedItem,TemplateItemEntity.class, HashSet.class, ArrayList.class,"parentTemplate","parentGroup","properties","relations");
      newTemplateItemEntity.setId(null);
      this.setTemplateItemRefObject(refObject,newTemplateItemEntity);
      sets.add(newTemplateItemEntity);
      this.mergeRecord(model,refObject,key,model.getMergedRefDiff().contains(key));
    }
    //当某基准字段A中有，B中无时，以A为准
    for(String key : model.getRefMergedDiff()){
      TemplateItemEntity refItem = refMap.get(key);
      this.validateField(key,persistentClassMap.get(key),refItem);
      TemplateItemEntity newTemplateItemEntity = kuiperToolkitService.copyObjectByWhiteList(refItem,TemplateItemEntity.class, HashSet.class, ArrayList.class,"parentTemplate","parentGroup","properties","relations");
      newTemplateItemEntity.setId(null);
      this.setTemplateItemRefObject(refObject,newTemplateItemEntity);
      sets.add(newTemplateItemEntity);
    }
    return sets;
  }

  /** 处理分组属性的字段*/
  private Set<TemplateGroupEntity> processGroupFields(StaticTemplateMergeModel model , Set<TemplateGroupEntity> refGroups , Set<TemplateGroupEntity> mergedGroups , String persistentClassName,Object refObject){
    Map<String,TemplateGroupEntity> refMap = refGroups.stream().collect(Collectors.toMap(TemplateGroupEntity::getPropertyName,e -> e));
    Map<String,TemplateGroupEntity> mergedMap = mergedGroups.stream().collect(Collectors.toMap(TemplateGroupEntity::getPropertyName,e -> e));
    PersistentClass currentPersistentClass = nebulaStaticPersistentService.findByPersistentClass(persistentClassName);
    Validate.notNull(currentPersistentClass , "没有找到当前静态模型描述，请检查!!");
    Map<String, PersistentRelation> persistentClassMap = currentPersistentClass.getRelations().stream().collect(Collectors.toMap(PersistentRelation::getPropertyName,e -> e));
    Set<TemplateGroupEntity> sets = Sets.newHashSet();
    Set<String> collections = Sets.union(model.getIntersections(),model.getMergedRefDiff());
    //当某基准字段A中有，B中有时，B的信息覆盖A
    //当某基准字段A中无，B中有时，B的信息覆盖A
    for(String key : collections){
      TemplateGroupEntity mergedGroup = mergedMap.get(key);
      this.validateField(key,persistentClassMap.get(key),mergedGroup);
      TemplateGroupEntity newTemplateGroupEntity = kuiperToolkitService.copyObjectByWhiteList(mergedGroup,TemplateGroupEntity.class, HashSet.class, ArrayList.class,"parentTemplate","properties","relations","itemRelations");
      newTemplateGroupEntity.setId(null);
      this.setTemplateGroupRefObject(refObject,newTemplateGroupEntity);
      sets.add(newTemplateGroupEntity);
      this.mergeRecord(model,refObject,key,model.getMergedRefDiff().contains(key));
    }
    //当某基准字段A中有，B中无时，以A为准
    for(String key : model.getRefMergedDiff()){
      TemplateGroupEntity refGroup = refMap.get(key);
      this.validateField(key,persistentClassMap.get(key),refGroup);
      TemplateGroupEntity newTemplateGroupEntity = kuiperToolkitService.copyObjectByWhiteList(refGroup,TemplateGroupEntity.class, HashSet.class, ArrayList.class,"parentTemplate","properties","relations","itemRelations");
      newTemplateGroupEntity.setId(null);
      this.setTemplateGroupRefObject(refObject,newTemplateGroupEntity);
      sets.add(newTemplateGroupEntity);
    }
    return sets;
  }


  /** 处理表单一般属性核心逻辑*/
  private void processProerties(StaticTemplateMergeModel model,Set<TemplatePropertyEntity> refProperties,Set<TemplatePropertyEntity> mergedProperties,String persistentClassName,Object refObject){

    Set<String> intersections = this.intersections(refProperties,mergedProperties,TemplatePropertyEntity::getPropertyName);
    Set<String> refMergedDiff = this.differences(refProperties,mergedProperties,TemplatePropertyEntity::getPropertyName,true);
    Set<String> mergedRefDiff = this.differences(refProperties,mergedProperties,TemplatePropertyEntity::getPropertyName,false);
    this.proertiesFields(model,intersections,refMergedDiff,mergedRefDiff,e -> this.matchExt(false,e));//基准字段
    Set<TemplatePropertyEntity> properties = this.processBaseFields(model,refProperties,mergedProperties,persistentClassName,refObject);//基准字段
    this.proertiesFields(model,intersections,refMergedDiff,mergedRefDiff,e -> this.matchExt(true,e));//扩展字段
    Set<TemplatePropertyEntity> extProperties = this.processExtFields(model,refProperties,mergedProperties,persistentClassName,refObject);//扩展字段
    properties.addAll(extProperties);
    if(!CollectionUtils.isEmpty(properties)) {
      properties.forEach(p -> p.setProjectName(platformContext.getAppName()));
    }
    templatePropertyRepository.saveAll(properties);
    templatePropertyRepository.flush();
    if(refObject instanceof TemplateEntity){
      ((TemplateEntity)refObject).setProperties(properties);
    }else if(refObject instanceof TemplateGroupEntity){
      ((TemplateGroupEntity)refObject).setProperties(properties);
    }else if(refObject instanceof TemplateItemEntity){
      ((TemplateItemEntity)refObject).setProperties(properties);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 处理表单关联属性核心逻辑*/
  private void processRelations(StaticTemplateMergeModel model,Set<TemplateRelationEntity> refRelations , Set<TemplateRelationEntity> mergedRelations , String persistentClassName,Object refObject){
    Set<String> intersectionRelations = this.intersections(refRelations,mergedRelations,TemplateRelationEntity::getPropertyName);
    Set<String> refMergedDiffRelations = this.differences(refRelations,mergedRelations,TemplateRelationEntity::getPropertyName,true);
    Set<String> mergedRefDiffRelations = this.differences(refRelations,mergedRelations,TemplateRelationEntity::getPropertyName,false);
    this.proertiesFields(model,intersectionRelations,refMergedDiffRelations,mergedRefDiffRelations,null);
    Set<TemplateRelationEntity> relations = this.processRelationsFields(model,refRelations,mergedRelations,persistentClassName,refObject);
    if(!CollectionUtils.isEmpty(relations)) {
      relations.forEach(r -> r.setProjectName(platformContext.getAppName()));
    }
    templateRelationRepository.saveAll(relations);
    templateRelationRepository.flush();
    if(refObject instanceof TemplateEntity){
      ((TemplateEntity)refObject).setRelations(relations);
    }else if(refObject instanceof TemplateGroupEntity){
      ((TemplateGroupEntity)refObject).setRelations(relations);
    }else if(refObject instanceof TemplateItemEntity){
      ((TemplateItemEntity)refObject).setRelations(relations);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 处理表单明细属性核心逻辑*/
  private void processItems(StaticTemplateMergeModel model,Set<TemplateItemEntity> refItems , Set<TemplateItemEntity> mergedItems , String persistentClassName,Object refObject){
    Set<String> intersectionItems = this.intersections(refItems,mergedItems,TemplateItemEntity::getPropertyName);
    Set<String> refMergedDiffItems = this.differences(refItems,mergedItems,TemplateItemEntity::getPropertyName,true);
    Set<String> mergedRefDiffItems = this.differences(refItems,mergedItems,TemplateItemEntity::getPropertyName,false);
    this.proertiesFields(model,intersectionItems,refMergedDiffItems,mergedRefDiffItems,null);
    Set<TemplateItemEntity> items = this.processItemFields(model,refItems,mergedItems,persistentClassName,refObject);
    for(TemplateItemEntity item : items){
      item.setProjectName(platformContext.getAppName());
      templateItemRepository.saveAndFlush(item);
      //一般属性
      TemplateItemEntity ref = refItems.stream().filter(e -> StringUtils.equals(e.getPropertyName(),item.getPropertyName())).findFirst().orElse(null);
      TemplateItemEntity merged = mergedItems.stream().filter(e -> StringUtils.equals(e.getPropertyName(),item.getPropertyName())).findFirst().orElse(null);
      Set<TemplatePropertyEntity> refProperties = (ref == null ? Sets.newHashSet() : ref.getProperties());
      Set<TemplatePropertyEntity> mergedProperties = (merged == null ? Sets.newHashSet() : merged.getProperties());
      this.processProerties(model,refProperties,mergedProperties,item.getPropertyClassName(),item);
      //关联属性
      Set<TemplateRelationEntity> refRelations = (ref == null ? Sets.newHashSet() : ref.getRelations());
      Set<TemplateRelationEntity> mergedRelations = (merged == null ? Sets.newHashSet() : merged.getRelations());
      if(!(CollectionUtils.isEmpty(refRelations) && CollectionUtils.isEmpty(mergedRelations))){
        this.processRelations(model,refRelations,mergedRelations,item.getPropertyClassName(),item);
      }
    }
    if(refObject instanceof TemplateEntity){
      ((TemplateEntity)refObject).setItemRelations(items);
    }else if(refObject instanceof TemplateGroupEntity){
      ((TemplateGroupEntity)refObject).setItemRelations(items);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }

  /** 处理表单分组属性核心逻辑*/
  private void processGroup(StaticTemplateMergeModel model,Set<TemplateGroupEntity> refGroups,Set<TemplateGroupEntity> mergedGroups,String persistentClassName,Object refObject){
    Set<String> intersectionGroups = this.intersections(refGroups,mergedGroups,TemplateGroupEntity::getPropertyName);
    Set<String> refMergedDiffGroups = this.differences(refGroups,mergedGroups,TemplateGroupEntity::getPropertyName,true);
    Set<String> mergedRefDiffGroups = this.differences(refGroups,mergedGroups,TemplateGroupEntity::getPropertyName,false);
    this.proertiesFields(model,intersectionGroups,refMergedDiffGroups,mergedRefDiffGroups,null);
    Set<TemplateGroupEntity> groups = this.processGroupFields(model,refGroups,mergedGroups,persistentClassName,refObject);
    for(TemplateGroupEntity group : groups){
      group.setProjectName(platformContext.getAppName());
      templateGroupRepository.saveAndFlush(group);
      //一般属性
      TemplateGroupEntity ref = refGroups.stream().filter(e -> StringUtils.equals(e.getPropertyName(),group.getPropertyName())).findFirst().orElse(null);
      TemplateGroupEntity merged = mergedGroups.stream().filter(e -> StringUtils.equals(e.getPropertyName(),group.getPropertyName())).findFirst().orElse(null);
      Set<TemplatePropertyEntity> refProperties = (ref == null ? Sets.newHashSet() : ref.getProperties());
      Set<TemplatePropertyEntity> mergedProperties = (merged == null ? Sets.newHashSet() : merged.getProperties());
      this.processProerties(model,refProperties,mergedProperties,group.getPropertyClassName(),group);
      //关联属性
      Set<TemplateRelationEntity> refRelations = (ref == null ? Sets.newHashSet() : ref.getRelations());
      Set<TemplateRelationEntity> mergedRelations = (merged == null ? Sets.newHashSet() : merged.getRelations());
      if(!(CollectionUtils.isEmpty(refRelations) && CollectionUtils.isEmpty(mergedRelations))){
        this.processRelations(model,refRelations,mergedRelations,group.getPropertyClassName(),group);
      }
      //明细属性
      Set<TemplateItemEntity> refItems = (ref == null ? Sets.newHashSet() : ref.getItemRelations());
      Set<TemplateItemEntity> mergedItems = (ref == null ? Sets.newHashSet() : ref.getItemRelations());
      if(!(CollectionUtils.isEmpty(refItems) && CollectionUtils.isEmpty(mergedItems))){
        this.processItems(model,refItems,mergedItems,group.getPropertyClassName(),group);
      }
    }
    if(refObject instanceof TemplateEntity){
      ((TemplateEntity)refObject).setGroupRelations(groups);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
  }


  /** 记录后端表单模型变动的字段项，并存储在上下文，已被布局文件解析时参考*/
  private void mergeRecord(StaticTemplateMergeModel model , Object refObject , String propertyName , boolean isAdd){
    String newPropertyName;
    if(refObject instanceof TemplatePropertyEntity || refObject instanceof TemplateEntity){
      newPropertyName = propertyName;
    }else if(refObject instanceof TemplateRelationEntity){
      newPropertyName = this.recordMergeRelations((TemplateRelationEntity)refObject,propertyName);
    }else if(refObject instanceof TemplateItemEntity){
      newPropertyName = this.recordMergeItems((TemplateItemEntity) refObject,propertyName);
    }else if(refObject instanceof TemplateGroupEntity){
      newPropertyName = this.recordMergeGroups((TemplateGroupEntity) refObject,propertyName);
    }else{
      throw new IllegalArgumentException("未知的模型数据，类型不匹配，请检查!!");
    }
    model.getMergeInfo().put(newPropertyName,isAdd);
  }

  /** 一般属性，构造前端布局信息的字段名称，该名称是把关联信息扁平化的拼装值*/
  private String recordMergeProperties(String propertyName){
    return propertyName;
  }

  /** 关联属性，构造前端布局信息的字段名称，该名称是把关联信息扁平化的拼装值*/
  private String recordMergeRelations(TemplateRelationEntity refObject,String propertyName){
    TemplateEntity template = refObject.getCurrentTemplate();
    TemplateItemEntity item = refObject.getCurrentItem();
    TemplateGroupEntity group = refObject.getCurrentGroup();
    if(template != null){
      return this.recordMergeProperties(propertyName);
    }
    if(item != null){
      return this.recordMergeItems(item,propertyName);
    }
    return this.recordMergeGroups(group,propertyName);
  }

  /** 明细属性，构造前端布局信息的字段名称，该名称是把关联信息扁平化的拼装值*/
  private String recordMergeItems(TemplateItemEntity refObject,String propertyName){
    TemplateEntity parentTemplate = refObject.getParentTemplate();
    TemplateGroupEntity parentGroupTemplate = refObject.getParentGroup();
    if(parentTemplate != null){
      return this.recordMergeProperties(StringUtils.join(refObject.getPropertyName(),"-",propertyName));
    }
    return this.recordMergeGroups(parentGroupTemplate,StringUtils.join(refObject.getPropertyName(),"-",propertyName));
  }

  /** 分组属性，构造前端布局信息的字段名称，该名称是把关联信息扁平化的拼装值*/
  private String recordMergeGroups(TemplateGroupEntity refObject,String propertyName){
    return this.recordMergeProperties(StringUtils.join(refObject.getPropertyName(),"-",propertyName));
  }

  /**
   *  处理前端布局文件逻辑
   * 需分别解析前端基准布局文件与前端待合并布局文件，并相互之间进行对比
   * */
  private void processTemplateLayouts(StaticTemplateMergeModel model){
    TemplateEntity refTemplate = model.getReferenceTemplate();
    TemplateEntity mergedTemplate = model.getMergedTemplate();
    Set<TemplateLayoutEntity> refLayouts = templateLayoutService.findByTemplateId(refTemplate.getId());
    Set<TemplateLayoutEntity> mergedLayouts = templateLayoutService.findByTemplateId(mergedTemplate.getId());
    Validate.isTrue(!CollectionUtils.isEmpty(refLayouts) && !CollectionUtils.isEmpty(mergedLayouts),"合并模板时，发现布局文件缺失，请检查!!");
    Validate.isTrue(refLayouts.size() == mergedLayouts.size(),"合并模板时，发现布局文件数不一致，请检查!!");
    refLayouts.forEach(v -> Validate.isTrue(mergedLayouts.stream().anyMatch(e -> v.getLayoutType().equals(e.getLayoutType())),"合并模板时，发现布局文件类型不匹配，请检查!!"));
    JSONObject refLayout1 = templateLayoutService.findDetailsByTemplateId(refTemplate.getId(),1);
    Validate.notNull(refLayout1,"表单模型【%s】版本【%s】  PC布局文件不存在，请检查!!",refTemplate.getName(),refTemplate.getCversion());
    JSONObject mergedLayout1 = templateLayoutService.findDetailsByTemplateId(mergedTemplate.getId(),1);
    Validate.notNull(mergedLayout1,"表单模型【%s】版本【%s】  PC布局文件不存在，请检查!!",mergedTemplate.getName(),mergedTemplate.getCversion());

    //解析基准布局文件，并存储上下文
    this.resolveLayout(model,refLayout1,true);
    //解析带合并布局文件，并存储上下文
    this.resolveLayout(model,mergedLayout1,false);
    //生成并构建新的前端布局文件JSON(替换了原有的基准布局文件)
    this.resolveJson(model,refLayout1);
    try {
      TemplateLayoutEntity layoutEntity = templateLayoutService.save(model.getNewTemplate().getId(),1,refLayout1);
      model.getNewTemplate().setTemplateLayout(layoutEntity);
    } catch (IOException e) {
      throw new IllegalArgumentException(String.format("保存PC布局文件失败：%s",e.getMessage()));
    }

    //当mobile布局文件都存在的情况下，以下逻辑才生效
    JSONObject refLayout2 = templateLayoutService.findDetailsByTemplateId(refTemplate.getId(),2);
    if(refLayout2 != null){
      JSONObject mergedLayout2 = templateLayoutService.findDetailsByTemplateId(mergedTemplate.getId(),2);
      if(mergedLayout2 != null){
        this.resolveLayout(model,refLayout2,true);
        this.resolveLayout(model,mergedLayout2,false);
        this.resolveJson(model,refLayout2);
        try {
          TemplateLayoutEntity layoutEntity2 = templateLayoutService.save(model.getNewTemplate().getId(),2,refLayout2);
          model.getNewTemplate().setMobileTemplateLayout(layoutEntity2);
        } catch (IOException e) {
          throw new IllegalArgumentException(String.format("保存Mobile布局文件失败：%s",e.getMessage()));
        }
      }
    }

    //当print布局文件都存在的情况下，以下逻辑才生效
    JSONObject refLayout3 = templateLayoutService.findDetailsByTemplateId(refTemplate.getId(),3);
    if(refLayout3 != null){
      JSONObject mergedLayout3 = templateLayoutService.findDetailsByTemplateId(mergedTemplate.getId(),3);
      if(mergedLayout3 != null){
        this.resolveLayout(model,refLayout3,true);
        this.resolveLayout(model,mergedLayout3,false);
        this.resolveJson(model,refLayout3);
        try {
          TemplateLayoutEntity layoutEntity3 = templateLayoutService.save(model.getNewTemplate().getId(),3,refLayout3);
          model.getNewTemplate().setPrintTemplateLayout(layoutEntity3);
        } catch (IOException e) {
          throw new IllegalArgumentException(String.format("保存Print布局文件失败：%s",e.getMessage()));
        }
      }
    }
  }

  /**
   * 分情况解析前端布局文件信息，并存储上下文
   * 1.空单元格
   * 2.按钮
   * 3.明细
   * 4.label标签
   * 5.字段
   * 注：
   * 1.由于前端的布局json结构含有cell_row元素，并且属性名称在最深层次，需要反向解析，以属性名称为key
   * 2.空单元格和按钮为单独的json数组
   * 3.label、字段、明细分别存储
   * */
  private void resolveLayout(StaticTemplateMergeModel model,JSONObject layout,boolean isRefJson){
    JSONObject layoutCells = layout.getJSONObject("cells");
    JSONObject jsonResolveInfo;
    JSONObject labelInfo;
    JSONObject fieldInfo;
    if(isRefJson){
      jsonResolveInfo = model.getRefJsonResolveInfo();
      labelInfo = model.getRefLabelJsonInfo();
      fieldInfo = model.getRefFieldJsonInfo();
    }else{
      jsonResolveInfo = model.getMergedJsonResolveInfo();
      labelInfo = model.getMergedLabelJsonInfo();
      fieldInfo = model.getMergedFieldJsonInfo();
    }
    jsonResolveInfo.clear();
    labelInfo.clear();
    fieldInfo.clear();
    jsonResolveInfo.put("emptyCells",new JSONArray());
    jsonResolveInfo.put("buttons",new JSONArray());

    for(String cellKey : layoutCells.keySet()){
      JSONObject cellValue = layoutCells.getJSONObject(cellKey);
      JSONObject json = new JSONObject();
      json.put("row,cell",cellKey);

      //空单元格
      if(cellValue.isEmpty()){
        json.put("value",cellValue);
        jsonResolveInfo.getJSONArray("emptyCells").add(json);
        continue;
      }
      //按钮
      String controllerType = cellValue.getString("type");
      if(StringUtils.equals(controllerType,"button")){
        json.put("value",cellValue);
        jsonResolveInfo.getJSONArray("buttons").add(json);
        continue;
      }else if(StringUtils.equals(controllerType,"table")){//明细
        String itemRelation = cellValue.getString("field");
        json.put("itemRelation",itemRelation);
        json.put("value",cellValue);
        jsonResolveInfo.put(itemRelation,json);
        continue;
      }
      //label标签
      String label = cellValue.getString("for");
      if(StringUtils.isNotBlank(label)){
        json.put("label",label);
        json.put("value",cellValue);
        labelInfo.put(label,json);
        continue;
      }
      //字段
      String field = cellValue.getString("field");
      if(StringUtils.isNotBlank(field)){
        json.put("field",field);
        json.put("value",cellValue);
        fieldInfo.put(field,json);
        continue;
      }
      throw new IllegalArgumentException("未知情况的前端布局文件解析，请检查!!");
    }
  }

  /**
   * 根据基准模板的布局json，替换refLayout生成新的前端布局json信息
   * 分情况逐步解析生成新布局信息
   * 1.按钮：单独的逻辑，只对新增的按钮重新生成
   * 2.label：由于字段或明细都有label，所以当新增加了按钮时，需要同时辨别新增(字段或明细)。如果是更新label，不需要同步更新(字段或明细)。
   * 3.字段：只做更新的逻辑，新增的逻辑在label已处理
   * 4.明细：只做更新的逻辑，新增的逻辑在label已处理
   * 注意：明细的逻辑采取的暴力方式，即把待合并布局json直接替换基准布局，没有对明细中的各个字段进行单独的处理
   */
  private void resolveJson(StaticTemplateMergeModel model,JSONObject refLayout){
    JSONObject refJsonResolveInfo = model.getRefJsonResolveInfo();
    JSONObject mergedJsonResolveInfo = model.getMergedJsonResolveInfo();
    JSONObject refLabelInfo = model.getRefLabelJsonInfo();
    JSONObject mergedLabelInfo = model.getMergedLabelJsonInfo();
    JSONObject refFieldInfo = model.getRefFieldJsonInfo();
    JSONObject mergedFieldInfo = model.getMergedFieldJsonInfo();
    //按钮
    JSONArray refButtons = refJsonResolveInfo.getJSONArray("buttons");
    JSONArray mergedButtons = mergedJsonResolveInfo.getJSONArray("buttons");
    Set<String> refButtonNames = refButtons.stream().map(e -> ((JSONObject) e).getJSONObject("value").getString("name")).collect(Collectors.toSet());
    Set<String> mergedButtonNames = mergedButtons.stream().map(e -> ((JSONObject) e).getJSONObject("value").getString("name")).collect(Collectors.toSet());
    Set<String> needAddButtonNames = Sets.difference(mergedButtonNames,refButtonNames);
    for (Object mergedButton : mergedButtons) {
      JSONObject button = ((JSONObject) mergedButton).getJSONObject("value");
      if (needAddButtonNames.contains(button.getString("name"))) {
        this.processLayoutButton(refLayout, button);
      }
    }

    //label
    Set<String> refLabelKeys = refLabelInfo.keySet();
    Set<String> mergedLabelKeys = mergedLabelInfo.keySet();
    Set<String> intersections = Sets.intersection(refLabelKeys,mergedLabelKeys);
    Set<String> mergedRefDifs = Sets.difference(mergedLabelKeys,refLabelKeys);
    for(String addLabelKey : mergedRefDifs){
      JSONObject label = mergedLabelInfo.getJSONObject(addLabelKey).getJSONObject("value");
      JSONObject field = mergedFieldInfo.getJSONObject(addLabelKey) == null ? null : mergedFieldInfo.getJSONObject(addLabelKey).getJSONObject("value");
      JSONObject item = mergedJsonResolveInfo.getJSONObject(addLabelKey) == null ? null : mergedJsonResolveInfo.getJSONObject(addLabelKey).getJSONObject("value");
      this.processLayoutLabel(label,refLayout,true,null);
      JSONObject cells = refLayout.getJSONObject("cells");
      JSONObject rows = refLayout.getJSONObject("rows");
      if(field != null){
        cells.put(StringUtils.join(String.valueOf(rows.size() - 1), ",", 2), field);
      }
      if(item != null){
        cells.put(StringUtils.join(String.valueOf(rows.size() - 1), ",", 2), item);
      }
    }
    for(String updateLabelKey : intersections){
      if(this.matchExt(false,updateLabelKey)){
        continue;
      }
      String rowcell = refLabelInfo.getJSONObject(updateLabelKey).getString("row,cell");
      JSONObject label = mergedLabelInfo.getJSONObject(updateLabelKey).getJSONObject("value");
      this.processLayoutLabel(label,refLayout,false,rowcell);
    }

    //字段
    Set<String> refFieldKeys = refFieldInfo.keySet();
    Set<String> mergedFieldKeys = mergedFieldInfo.keySet();
    intersections = Sets.intersection(refFieldKeys,mergedFieldKeys);
//    mergedRefDifs = Sets.difference(mergedFieldKeys,refFieldKeys);

    for(String updateFieldKey : intersections){
      if(this.matchExt(false,updateFieldKey)){
        continue;
      }
      String rowcell = refFieldInfo.getJSONObject(updateFieldKey).getString("row,cell");
      JSONObject field = mergedFieldInfo.getJSONObject(updateFieldKey).getJSONObject("value");
      this.processLayoutField(field,refLayout,false,rowcell);
    }

    //明细
    Set<String> refItemKeys = refJsonResolveInfo.keySet().stream().filter(e -> !StringUtils.equals(e,"buttons") && !StringUtils.equals(e,"emptyCells")).collect(Collectors.toSet());
    Set<String> mergedItemKeys = mergedJsonResolveInfo.keySet().stream().filter(e -> !StringUtils.equals(e,"buttons") && !StringUtils.equals(e,"emptyCells")).collect(Collectors.toSet());
    intersections = Sets.intersection(refItemKeys,mergedItemKeys);
//    mergedRefDifs = Sets.difference(mergedItemKeys,refItemKeys);

    for(String updateItemKey : intersections){
      String refRowcell = refJsonResolveInfo.getJSONObject(updateItemKey).getString("row,cell");
      JSONObject item = mergedJsonResolveInfo.getJSONObject(updateItemKey).getJSONObject("value");
      this.processLayoutItem(item,refLayout,false,refRowcell);
    }
  }


  /**
   * 处理按钮逻辑：
   * 新增时，把当前表单模板的最后一行数据往下再移动一行，并且原有行的数据替换为新值
   */
  private void processLayoutButton(JSONObject refLayout,JSONObject button){
    JSONObject rows = refLayout.getJSONObject("rows");
    JSONObject cells = refLayout.getJSONObject("cells");
    Set<String> cellKeys = cells.keySet();
    //获取最后一行cells
    Set<String> lastRowCells = cellKeys.stream().filter(e -> StringUtils.startsWith(e,StringUtils.join(rows.size(),","))).collect(Collectors.toSet());
    for(String lastRowCell : lastRowCells){
      String[] splits = lastRowCell.split(",");
      //赋值最后一行的cell
      cells.put(StringUtils.join(String.valueOf(rows.size() + 1),",",splits[1]),cells.remove(lastRowCell));
    }
    //赋值最后一行的row
    rows.put(String.valueOf(rows.size() + 1),rows.getJSONObject(String.valueOf(rows.size())));
    //替换原有的cell（注意rows的数据已经加了一行，需要减去）
    cells.put(StringUtils.join(String.valueOf(rows.size() -1),",",1),button);
    JSONObject newRowValue = new JSONObject();
    newRowValue.put("column",new Integer[]{24});
    rows.put(String.valueOf(rows.size() - 1),newRowValue);
  }


  /**
   * 处理label逻辑：
   * 新增时，把当前表单模板的最后一行数据往下再移动一行，原有行的数据替换为新的label值，并且辨别该label是否是字段或明细，同时新增字段或明细的数据
   * 更新时，直接替换原有的单元格值为新值。字段或明细不在此处理
   */
  private void processLayoutLabel(JSONObject label,JSONObject refLayout,boolean isAdd,String rowcell){
    JSONObject rows = refLayout.getJSONObject("rows");
    JSONObject cells = refLayout.getJSONObject("cells");
    Set<String> cellKeys = cells.keySet();
    if(!isAdd){
      cells.replace(rowcell, label);
      return;
    }
    //获取最后一行cells
    Set<String> lastRowCells = cellKeys.stream().filter(e -> StringUtils.startsWith(e,StringUtils.join(rows.size(),","))).collect(Collectors.toSet());
    for(String lastRowCell : lastRowCells){
      String[] splits = lastRowCell.split(",");
      //赋值最后一行的cell
      cells.put(StringUtils.join(String.valueOf(rows.size() + 1),",",splits[1]),cells.remove(lastRowCell));
    }
    //赋值最后一行的row
    rows.put(String.valueOf(rows.size() + 1),rows.getJSONObject(String.valueOf(rows.size())));
    //替换原有的cell（注意rows的数据已经加了一行，需要减去）
    cells.put(StringUtils.join(String.valueOf(rows.size() - 1), ",", 1), label);
    JSONObject newRowValue = new JSONObject();
    newRowValue.put("column",new Integer[]{3,21});
    rows.put(String.valueOf(rows.size() - 1),newRowValue);
  }

  /**
   * 处理字段逻辑：
   * 只做更新，因为新增的逻辑已经在label中进行处理了。直接替换原有的单元格值为新值。
   */
  private void processLayoutField(JSONObject field,JSONObject refLayout,boolean isAdd,String rowcell){
    JSONObject rows = refLayout.getJSONObject("rows");
    JSONObject cells = refLayout.getJSONObject("cells");
    Set<String> cellKeys = cells.keySet();
    if(!isAdd){
      cells.replace(rowcell, field);
      return;
    }
    //获取最后一行cells
    Set<String> lastRowCells = cellKeys.stream().filter(e -> StringUtils.startsWith(e,StringUtils.join(rows.size(),","))).collect(Collectors.toSet());
    for(String lastRowCell : lastRowCells){
      String[] splits = lastRowCell.split(",");
      //赋值最后一行的cell
      cells.put(StringUtils.join(String.valueOf(rows.size() + 1),",",splits[1]),cells.remove(lastRowCell));
    }
    //赋值最后一行的row
    rows.put(String.valueOf(rows.size() + 1),rows.getJSONObject(String.valueOf(rows.size())));
    //替换原有的cell（注意rows的数据已经加了一行，需要减去）
    cells.put(StringUtils.join(String.valueOf(rows.size() - 1), ",", 1), field);
    if(StringUtils.equals(field.getString("controltype"),"title")){
      rows.getJSONObject(String.valueOf(rows.size() - 1)).put("group",field.getString("field"));
    }
  }

  /**
   * 处理明细逻辑：
   * 只做更新，因为新增的逻辑已经在label中进行处理了。直接替换原有的单元格值为新值。
   * 注：明细的处理比较暴力，直接覆盖原有值，没有对明细中的字段再进行深层次的判断
   */
  private void processLayoutItem(JSONObject item,JSONObject refLayout,boolean isAdd,String rowcell){
    JSONObject rows = refLayout.getJSONObject("rows");
    JSONObject cells = refLayout.getJSONObject("cells");
    Set<String> cellKeys = cells.keySet();
    if(!isAdd){
      cells.replace(rowcell, item);
      return;
    }
    //获取最后一行cells
    Set<String> lastRowCells = cellKeys.stream().filter(e -> StringUtils.startsWith(e,StringUtils.join(rows.size(),","))).collect(Collectors.toSet());
    for(String lastRowCell : lastRowCells){
      String[] splits = lastRowCell.split(",");
      //赋值最后一行的cell
      cells.put(StringUtils.join(String.valueOf(rows.size() + 1),",",splits[1]),cells.remove(lastRowCell));
    }
    //赋值最后一行的row
    rows.put(String.valueOf(rows.size() + 1),rows.getJSONObject(String.valueOf(rows.size())));
    //替换原有的cell（注意rows的数据已经加了一行，需要减去）
    cells.put(StringUtils.join(String.valueOf(rows.size() - 1), ",", 1), item);
  }


  /**
   * 表单模板事件的处理逻辑
   * 一般控件事件：
   *    交集时：需要判断当前字段是否扩展字段，如果是扩展字段，需要以基础模板为准。如果不是扩展字段，需要以待合并模板数据为准
   *    正向差集：以基础模板为准
   *    反向差集：以待合并模板数据为准
   * 表单事件：onLoad,beforeSubmit,afterSubmit
   *    正向差集：以基础模板为准
   *    交集和反向差集：以待合并模板数据为准
   */
  private void processEvents(StaticTemplateMergeModel model){
    TemplateEntity refTemplate = model.getReferenceTemplate();
    TemplateEntity mergedTemplate = model.getMergedTemplate();
    //允许事件为空，无需校验
    Set<TemplateEventEntity> refEvents = templateEventService.findDetailsByTemplateId(refTemplate.getId());
    Set<TemplateEventEntity> mergedEvents = templateEventService.findDetailsByTemplateId(mergedTemplate.getId());
    Set<TemplateEventEntity> refInnerEvents = refEvents.stream().filter(e -> INNER_EVENT_NAME.contains(e.getOnEvent())).collect(Collectors.toSet());
    Set<TemplateEventEntity> mergedInnerEvents = mergedEvents.stream().filter(e -> INNER_EVENT_NAME.contains(e.getOnEvent())).collect(Collectors.toSet());
    Set<TemplateEventEntity> refContrllerEvents = refEvents.stream().filter(e -> !INNER_EVENT_NAME.contains(e.getOnEvent())).collect(Collectors.toSet());
    Set<TemplateEventEntity> mergedContrllerEvents = mergedEvents.stream().filter(e -> !INNER_EVENT_NAME.contains(e.getOnEvent())).collect(Collectors.toSet());

    Set<String> intersections = this.intersections(refContrllerEvents,mergedContrllerEvents,TemplateEventEntity::getAttributeName);
    Set<String> refMergedDiff = this.differences(refContrllerEvents,mergedContrllerEvents,TemplateEventEntity::getAttributeName,true);
    Set<String> mergedRefDiff = this.differences(refContrllerEvents,mergedContrllerEvents,TemplateEventEntity::getAttributeName,false);

    List<TemplateEventEntity> newEvents = Lists.newArrayList();
    //交集
    for(String propertyName : intersections){
      if(this.matchExt(true,propertyName)){
        TemplateEventEntity mergedEvent = mergedContrllerEvents.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
        Validate.notNull(mergedEvent,"未找到待合并模板事件%s",propertyName);
        TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(mergedEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
        newEventEntity.setId(null);
        newEventEntity.setTemplate(model.getNewTemplate());
        newEvents.add(newEventEntity);
      }else{
        TemplateEventEntity refMergedEvent = refContrllerEvents.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
        Validate.notNull(refMergedEvent,"未找到基准模板事件%s",propertyName);
        TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(refMergedEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
        newEventEntity.setId(null);
        newEventEntity.setTemplate(model.getNewTemplate());
        newEvents.add(newEventEntity);
      }
    }
    //正向差集
    for(String propertyName : refMergedDiff){
      TemplateEventEntity refMergedEvent = refContrllerEvents.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
      Validate.notNull(refMergedEvent,"未找到基准模板事件%s",propertyName);
      TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(refMergedEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
      newEventEntity.setId(null);
      newEventEntity.setTemplate(model.getNewTemplate());
      newEvents.add(newEventEntity);
    }
    //反向差集
    for(String propertyName : mergedRefDiff){
      TemplateEventEntity mergedEvent = mergedContrllerEvents.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
      Validate.notNull(mergedEvent,"未找到待合并模板事件%s",propertyName);
      TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(mergedEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
      newEventEntity.setId(null);
      newEventEntity.setTemplate(model.getNewTemplate());
      newEvents.add(newEventEntity);
    }

    //辨别表单事件类型
    intersections = this.intersections(refInnerEvents,mergedInnerEvents,TemplateEventEntity::getOnEvent);
    refMergedDiff = this.differences(refInnerEvents,mergedInnerEvents,TemplateEventEntity::getOnEvent,true);
    mergedRefDiff = this.differences(refInnerEvents,mergedInnerEvents,TemplateEventEntity::getOnEvent,false);
    for(String eventName : Sets.union(intersections,mergedRefDiff)){
      TemplateEventEntity mergedInnerEvent = mergedInnerEvents.stream().filter(e -> StringUtils.equals(eventName,e.getOnEvent())).findFirst().orElse(null);
      Validate.notNull(mergedInnerEvent,"未找到待合并模板事件%s",eventName);
      TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(mergedInnerEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
      newEventEntity.setId(null);
      newEventEntity.setTemplate(model.getNewTemplate());
      newEvents.add(newEventEntity);
    }

    for(String eventName : refMergedDiff){
      TemplateEventEntity refInnerEvent = refInnerEvents.stream().filter(e -> StringUtils.equals(eventName,e.getOnEvent())).findFirst().orElse(null);
      Validate.notNull(refInnerEvent,"未找到基准模板事件%s",eventName);
      TemplateEventEntity newEventEntity = kuiperToolkitService.copyObjectByWhiteList(refInnerEvent,TemplateEventEntity.class,HashSet.class,ArrayList.class,"");
      newEventEntity.setId(null);
      newEventEntity.setTemplate(model.getNewTemplate());
      newEvents.add(newEventEntity);
    }

    //执行保存
    templateEventService.save(model.getNewTemplate().getId(),newEvents);
  }


  /**
   * 表单模板可见性的处理逻辑
   * 1.先创建create可见性，且只创建一个create可见性，其他可见性不创建
   * 2.再处理可见性属性
   * 3.最后处理可见性按钮
   *
   * 交集时：需要判断当前字段是否扩展字段，如果是扩展字段，需要以基础模板为准。如果不是扩展字段，需要以待合并模板数据为准
   * 正向差集：以基础模板为准
   * 反向差集：以待合并模板数据为准，且可见性属性设为隐藏
   */
  private void processVisibilities(StaticTemplateMergeModel model){
    TemplateEntity refTemplate = model.getReferenceTemplate();
    TemplateEntity mergedTemplate = model.getMergedTemplate();
    Set<TemplateVisibilityEntity> refVisibilities = templateVisibilityService.findDetailsByTemplateId(refTemplate.getId());
    Validate.notEmpty(refVisibilities,"基准表单模板的可见性不能为空，请检查!!");
    Set<TemplateVisibilityEntity> mergedVisibilities = templateVisibilityService.findDetailsByTemplateId(mergedTemplate.getId());
    Validate.notEmpty(mergedVisibilities,"待合并表单模板的可见性不能为空，请检查!!");
    TemplateVisibilityEntity refVisibility = refVisibilities.stream().filter( e -> StringUtils.equals(e.getVisibilityName(),"create")).findAny().orElse(null);
    TemplateVisibilityEntity mergedVisibility = mergedVisibilities.stream().filter( e -> StringUtils.equals(e.getVisibilityName(),"create")).findAny().orElse(null);
    Validate.isTrue(refVisibility != null && mergedVisibility != null,"合并模板时，未发现create可见性信息，请检查!!");
    Set<TemplateVisibilityEntity> newVisibilities = Sets.newHashSet();
    this.buildTemplateVisibility(newVisibilities,refVisibility,mergedVisibility,model.getNewTemplate());
    Set<TemplateVisibilityEntity> visibilities = templateVisibilityService.save(model.getNewTemplate(),newVisibilities);
    model.getNewTemplate().setVisibility(visibilities);
  }

  private void buildTemplateVisibility(Set<TemplateVisibilityEntity> sets,TemplateVisibilityEntity refVisibility,TemplateVisibilityEntity mergedVisibility,TemplateEntity template){
    //只创建create可见性
    TemplateVisibilityEntity newVisibility = kuiperToolkitService.copyObjectByWhiteList(mergedVisibility,TemplateVisibilityEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
    newVisibility.setId(null);
    newVisibility.setTemplate(template);
    newVisibility.setAttributes(Sets.newHashSet());
    newVisibility.setButtons(Sets.newHashSet());
    sets.add(newVisibility);

    //可见性属性
    Set<TemplateVisibilityAttributesEntity> refAttributes = templateVisibilityAttributesRepository.findByVisiblId(refVisibility.getId());
    Validate.notEmpty(refAttributes,"基准表单模板的可见性属性不能为空，请检查!!");
    Set<TemplateVisibilityAttributesEntity> mergedAttributes = templateVisibilityAttributesRepository.findByVisiblId(mergedVisibility.getId());
    Validate.notEmpty(mergedAttributes,"待合并表单模板的可见性属性不能为空，请检查!!");
    Set<String> intersections = this.intersections(refAttributes,mergedAttributes,TemplateVisibilityAttributesEntity::getAttributeName);
    Set<String> refMergedDiff = this.differences(refAttributes,mergedAttributes,TemplateVisibilityAttributesEntity::getAttributeName,true);
    Set<String> mergedRefDiff = this.differences(refAttributes,mergedAttributes,TemplateVisibilityAttributesEntity::getAttributeName,false);
    //交集
    for(String propertyName : intersections){
      if(this.matchExt(true,propertyName)){
        TemplateVisibilityAttributesEntity mergedAttribute = mergedAttributes.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
        Validate.notNull(mergedAttribute,"未找到待合并模板可见性属性%s",propertyName);
        TemplateVisibilityAttributesEntity newAttributeEntity = kuiperToolkitService.copyObjectByWhiteList(mergedAttribute,TemplateVisibilityAttributesEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
        newAttributeEntity.setId(null);
        newAttributeEntity.setTemplateVisibility(newVisibility);
        newVisibility.getAttributes().add(newAttributeEntity);
      }else{
        TemplateVisibilityAttributesEntity refAttribute = refAttributes.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
        Validate.notNull(refAttribute,"未找到基准模板可见性属性%s",propertyName);
        TemplateVisibilityAttributesEntity newAttributeEntity = kuiperToolkitService.copyObjectByWhiteList(refAttribute,TemplateVisibilityAttributesEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
        newAttributeEntity.setId(null);
        newAttributeEntity.setTemplateVisibility(newVisibility);
        newVisibility.getAttributes().add(newAttributeEntity);
      }
    }
    //正向差集
    for(String propertyName : refMergedDiff){
      TemplateVisibilityAttributesEntity refAttribute = refAttributes.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
      Validate.notNull(refAttribute,"未找到基准模板可见性属性%s",propertyName);
      TemplateVisibilityAttributesEntity newAttributeEntity = kuiperToolkitService.copyObjectByWhiteList(refAttribute,TemplateVisibilityAttributesEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
      newAttributeEntity.setId(null);
      newAttributeEntity.setTemplateVisibility(newVisibility);
      newVisibility.getAttributes().add(newAttributeEntity);
    }
    //反向差集
    for(String propertyName : mergedRefDiff){
      TemplateVisibilityAttributesEntity mergedAttribute = mergedAttributes.stream().filter(e -> StringUtils.equals(propertyName,e.getAttributeName())).findFirst().orElse(null);
      Validate.notNull(mergedAttribute,"未找到待合并模板可见性属性%s",propertyName);
      TemplateVisibilityAttributesEntity newAttributeEntity = kuiperToolkitService.copyObjectByWhiteList(mergedAttribute,TemplateVisibilityAttributesEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
      newAttributeEntity.setId(null);
      if(this.matchExt(false,propertyName)){
        newAttributeEntity.setVisibilityType(3);
      }
      newAttributeEntity.setTemplateVisibility(newVisibility);
      newVisibility.getAttributes().add(newAttributeEntity);
    }


    //按钮的处理逻辑：
    // 1.当有交集和反向差集时，以带合并模板的按钮数据为准
    // 2.当有正向差集时，以基准模板的按钮数据为准
    Set<TemplateVisibilityButtonsEntity> refButtons = templateVisibilityButtonRepository.findByVisiblId(refVisibility.getId());
    Validate.notEmpty(refAttributes,"基准表单模板的可见性按钮不能为空，请检查!!");
    Set<TemplateVisibilityButtonsEntity> mergedButtons = templateVisibilityButtonRepository.findByVisiblId(mergedVisibility.getId());
    Validate.notEmpty(mergedAttributes,"待合并表单模板的可见性按钮不能为空，请检查!!");
    intersections = this.intersections(refButtons,mergedButtons,TemplateVisibilityButtonsEntity::getControllerId);
    refMergedDiff = this.differences(refButtons,mergedButtons,TemplateVisibilityButtonsEntity::getControllerId,true);
    mergedRefDiff = this.differences(refButtons,mergedButtons,TemplateVisibilityButtonsEntity::getControllerId,false);
    //交集和反向差集的并集
    for(String propertyName : Sets.union(intersections,mergedRefDiff)){
      TemplateVisibilityButtonsEntity mergedButton = mergedButtons.stream().filter(e -> StringUtils.equals(propertyName,e.getControllerId())).findFirst().orElse(null);
      Validate.notNull(mergedButton,"未找到待合并模板按钮控件%s",propertyName);
      TemplateVisibilityButtonsEntity newButtonEntity = kuiperToolkitService.copyObjectByWhiteList(mergedButton,TemplateVisibilityButtonsEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
      newButtonEntity.setId(null);
      newButtonEntity.setTemplateVisibility(newVisibility);
      newVisibility.getButtons().add(newButtonEntity);
    }
    //正向差集
    for(String propertyName : refMergedDiff){
      TemplateVisibilityButtonsEntity refButton = refButtons.stream().filter(e -> StringUtils.equals(propertyName,e.getControllerId())).findFirst().orElse(null);
      Validate.notNull(refButton,"未找到基准模板按钮控件%s",propertyName);
      TemplateVisibilityButtonsEntity newButtonEntity = kuiperToolkitService.copyObjectByWhiteList(refButton,TemplateVisibilityButtonsEntity.class,HashSet.class,ArrayList.class,StringUtils.EMPTY);
      newButtonEntity.setId(null);
      newButtonEntity.setTemplateVisibility(newVisibility);
      newVisibility.getButtons().add(newButtonEntity);
    }
  }

  /**
   * 匹配是否扩展字段
   * @param matchExt 当为false时，匹配不是扩展字段； 当为true时，匹配扩展字段
   * @param fieldName
   * @return
   */
  private boolean matchExt(boolean matchExt,String fieldName){
    return matchExt ? ( Pattern.matches(EXT_PATTERN,fieldName) || Pattern.matches(EXTEND_PATTERN,fieldName)) :
            (!Pattern.matches(EXT_PATTERN,fieldName) && !Pattern.matches(EXTEND_PATTERN,fieldName));
  }
}

