package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.core.service.NebulaStaticPersistentService;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.starter.repository.TemplateGroupRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplatePropertyRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRelationRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateGroupService;
import com.bizunited.platform.kuiper.starter.service.TemplateItemService;
import com.bizunited.platform.saturn.model.PersistentClass;
import com.bizunited.platform.saturn.model.PersistentProperty;
import com.bizunited.platform.saturn.model.PersistentRelation;
import com.bizunited.platform.saturn.model.PersistentRelation.RelationType;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Service("TemplateGroupServiceImpl")
public class TemplateGroupServiceImpl extends AbstractTemplateService implements TemplateGroupService {
  @Autowired
  private TemplateGroupRepository templateGroupRepository;
  @Autowired
  private TemplatePropertyRepository templatePropertyRepository;
  @Autowired
  private TemplateRelationRepository templateRelationRepository;
  @Autowired
  private NebulaStaticPersistentService nebulaStaticPersistentService;
  @Autowired
  private TemplateItemService templateItemService;
  @Autowired
  private PlatformContext platformContext;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateGroupService#initStaticGroups(com.bizunited.platform.kuiper.entity.TemplateEntity, com.bizunited.platform.saturn.model.PersistentRelation, com.bizunited.platform.saturn.model.PersistentClass)
   */
  @Override
  @Transactional
  public TemplateGroupEntity initStaticGroups(TemplateEntity parentTemplate , PersistentRelation parentRelation , PersistentClass persistentClass) {
    String parentClassName = parentRelation.getPersistentClassName();
    // 基于上层是主表单模型关联，进行的分组创建
    TemplateGroupEntity templateGroup = new TemplateGroupEntity();
    templateGroup.setPropertyClassName(parentRelation.getPropertyClass());
    // 分组项业务编号
    String propertyName = parentRelation.getPropertyName();
    Validate.notBlank(propertyName , "分组项属性不能为空，请检查!!");
    Validate.isTrue(propertyName.length() <= 64 , "属性名[%s]的长度需要小于64个字符，请检查!!" , propertyName);
    
    templateGroup.setParentClassName(parentClassName);
    templateGroup.setPkage(persistentClass.getPkage());
    templateGroup.setRepositoryEntity(persistentClass.getRepositoryEntity());
    templateGroup.setSimpleClassName(persistentClass.getSimpleClassName());
    templateGroup.setType("static");
    // 和动态模型相关的信息设置为空字符串
    templateGroup.setParentTableName("");
    templateGroup.setTableName("");
    templateGroup.setParentTemplate(parentTemplate);
    templateGroup.setId(null);
    templateGroup.setParentTableName("");
    templateGroup.setQueryService("");
    // 字段属性和字段中文描述
    templateGroup.setPropertyDesc(parentRelation.getPropertyDesc());
    templateGroup.setPropertyName(propertyName);
    templateGroup.setProjectName(platformContext.getAppName());
    this.templateGroupRepository.save(templateGroup);
    
    // 明细中的基本属性
    Set<TemplatePropertyEntity> templateProperties = this.initStaticProperties(persistentClass.getProperties());
    templateGroup.setProperties(templateProperties);
    if(!templateProperties.isEmpty()) {
      templateProperties.stream().forEach(item -> item.setCurrentGroup(templateGroup));
      templatePropertyRepository.saveAll(templateProperties);
      templateGroup.setProperties(templateProperties);
    }
    
    // 直接关联属性(只需要处理ManyToMany关系和ManyToOne关系)
    List<PersistentRelation> relations = persistentClass.getRelations();
    if(relations == null || relations.isEmpty()) {
      return templateGroup;
    }
    List<PersistentRelation> currentManyToRelations = relations.stream()
        .filter(item -> (item.getRelationType() == RelationType.ManyToOne || item.getRelationType() == RelationType.ManyToMany)).collect(Collectors.toList());
    Set<TemplateRelationEntity> templateRelations = this.initStaticRelations(parentClassName , currentManyToRelations);
    templateGroup.setRelations(templateRelations);
    if(!templateRelations.isEmpty()) {
      templateRelations.stream().forEach(item -> item.setCurrentGroup(templateGroup));
      templateRelationRepository.saveAll(templateRelations);
      templateGroup.setRelations(templateRelations);
    }
    
    // 关联的明细属性（既是OneToMany类型的关联）
    List<PersistentRelation> persistentOneToManyRelations = relations.stream().filter(item -> item.getRelationType() == RelationType.OneToMany).collect(Collectors.toList());
    if(persistentOneToManyRelations != null && !persistentOneToManyRelations.isEmpty()) {
      Set<TemplateItemEntity> items = new HashSet<>();
      for (PersistentRelation persistentOneToManyRelationItem : persistentOneToManyRelations) {
        String oneToManyPersistentClassName = persistentOneToManyRelationItem.getPropertyClass();
        PersistentClass oneToManyPersistentClass = nebulaStaticPersistentService.findByPersistentClass(oneToManyPersistentClassName);
        Validate.notNull(oneToManyPersistentClass , "在初始化主静态模型时发现NULL关联信息，请联系表单引擎开发团队!!");
        TemplateItemEntity currentTemplateItem = this.templateItemService.initStaticItems(null, templateGroup, persistentOneToManyRelationItem, oneToManyPersistentClass);
        items.add(currentTemplateItem);
      }
      templateGroup.setItemRelations(items);
    }
    
    return templateGroup;
  }
  
  /**
   * 根据模型的分组关联属性，组装模板分组信息中的关联属性——relations部分的信息
   * @param parentClassName
   * @param persistentRelations
   * @return
   */
  private Set<TemplateRelationEntity> initStaticRelations(String parentClassName , List<PersistentRelation> persistentRelations) {
    Set<TemplateRelationEntity> templateRelations = new LinkedHashSet<>();
    for (PersistentRelation persistentRelationItem : persistentRelations) {
      TemplateRelationEntity relationModel = super.initTemplateRelationEntity(persistentRelationItem, parentClassName);
      relationModel.setProjectName(platformContext.getAppName());
      templateRelations.add(relationModel);
    }
    return templateRelations;
  }
  
  /**
   * 根据模型的分组关联属性，组装模板分组信息中的一般属性
   * @param persistentClass 
   * @param persistentProperties
   * @return
   */
  private Set<TemplatePropertyEntity> initStaticProperties(List<PersistentProperty> persistentProperties) {
    Set<TemplatePropertyEntity> templateProperties = new LinkedHashSet<>();
    for (PersistentProperty persistentProperty : persistentProperties) {
      TemplatePropertyEntity propertyEntity = super.initTemplatePropertyEntity(persistentProperty);
      propertyEntity.setProjectName(platformContext.getAppName());
      templateProperties.add(propertyEntity);
    }
    return templateProperties;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateGroupService#initDynamicGroups(com.bizunited.platform.kuiper.entity.TemplateGroupEntity, com.bizunited.platform.kuiper.entity.TemplateEntity)
   */
  @Override
  @Transactional
  public TemplateGroupEntity initDynamicGroups(TemplateGroupEntity templateGroup , TemplateEntity parentTemplate) {
    // 首先是OneToOne本身的信息
    String propertyName = templateGroup.getPropertyName();
    Validate.notBlank(propertyName , "分组数据表的属性名必须填写!!");
    String propertyClassName = templateGroup.getPropertyClassName();
    if(StringUtils.isBlank(propertyClassName)) {
      propertyClassName = "java.lang.String";
      templateGroup.setPropertyClassName(propertyClassName);
    }
    String propertyDesc = templateGroup.getPropertyDesc();
    Validate.notBlank(propertyDesc , "分组数据表的属性说明必须填写");
    String groupType = templateGroup.getType();
    Validate.isTrue(StringUtils.equals(groupType, "dynamic") , "分组数据表的类型必须是动态数据表（dynamic）");
    String groupTableName = templateGroup.getTableName();
    Validate.notBlank(groupTableName , "分组数据表的数据说明信息必须指定!!");
    
    templateGroup.setId(null);
    templateGroup.setParentTemplate(parentTemplate);
    String parentTableName = parentTemplate.getTableName();
    templateGroup.setParentTableName(parentTableName);
    templateGroup.setRepositoryEntity(true);
    // 和静态模板有关的属性都只为空
    templateGroup.setSimpleClassName("");
    templateGroup.setQueryService("");
    templateGroup.setParentClassName("");
    templateGroup.setPkage("");
    templateGroup.setProjectName(platformContext.getAppName());
    this.templateGroupRepository.save(templateGroup);
    
    // 然后是分组数据表中的一般属性
    Set<TemplatePropertyEntity> templateProperties = templateGroup.getProperties();
    this.initDynamicProperties(templateGroup, templateProperties);
    // 然后是关联属性
    Set<TemplateRelationEntity> templateRelations = templateGroup.getRelations();
    this.initDynamicRelations(templateGroup, templateRelations);
    // 最后是分组下可能的明细编辑项
    Set<TemplateItemEntity> templateItems = templateGroup.getItemRelations();
    if(templateItems != null && !templateItems.isEmpty()) {
      for (TemplateItemEntity templateItem : templateItems) {
        templateItemService.initDynamicItems(templateItem, null, templateGroup, groupTableName);
      }
    }
    return templateGroup;
  }

  @Override
  public TemplateGroupEntity findParentById(String templateGroupId) {
    if(StringUtils.isBlank(templateGroupId)){
      return null;
    }
    return templateGroupRepository.findParentById(templateGroupId);
  }

  /**
   * 初始化一般模型
   * @param currentGroup
   * @param properties
   */
  private void initDynamicProperties(TemplateGroupEntity currentGroup , Set<TemplatePropertyEntity> properties) {
    for (TemplatePropertyEntity templateProperty : properties) {
      templateProperty.setId(null);
      templateProperty.setCurrentTemplate(null);
      templateProperty.setCurrentGroup(currentGroup);
      templateProperty.setCurrentItem(null);
      templateProperty.setProjectName(platformContext.getAppName());
      this.templatePropertyRepository.save(templateProperty);
    }
  }
  
  /**
   * 初始化关联模型，包括ManyToOne关联和ManyToMany关联
   * @param currentGroup
   * @param relations
   */
  private void initDynamicRelations(TemplateGroupEntity currentGroup , Set<TemplateRelationEntity> relations) {
    if(relations == null || relations.isEmpty()) {
      return;
    }
    for (TemplateRelationEntity templateRelation : relations) {
      templateRelation.setId(null);
      templateRelation.setCurrentGroup(currentGroup);
      templateRelation.setCurrentItem(null);
      templateRelation.setCurrentTemplate(null);
      // 将和静态模型有关的属性设置为空
      templateRelation.setQueryService("");
      templateRelation.setProjectName(platformContext.getAppName());
      this.templateRelationRepository.save(templateRelation);
    }
  }
}
