package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemExcelEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.TemplateItemExcelRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateItemExcelService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.ZipFile;

/**
 * TemplateItemExcelServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 19/Aug/2019 17:03
 */
@Service("TemplateItemExcelServiceImpl")
public class TemplateItemExcelServiceImpl implements TemplateItemExcelService {

  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateItemExcelServiceImpl.class);

  @Autowired
  private TemplateItemExcelRepository templateItemExcelRepository;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private VenusFileService fileUpdateService;
  @Autowired
  private PlatformContext platformContext;


  @Override
  @Transactional
  public TemplateItemExcelEntity save(TemplateItemExcelEntity templateItemExcelEntity) {
    Validate.notNull(templateItemExcelEntity, "创建时信息不能为空！");
    Validate.notBlank(templateItemExcelEntity.getTemplateId(), "创建时表单模板ID不能为空！");
    Validate.notBlank(templateItemExcelEntity.getTemplateItemId(), "创建时表单明细模板ID不能为空！");
    Validate.notBlank(templateItemExcelEntity.getFileName(), "创建时文件名称不能为空！");
    Validate.notBlank(templateItemExcelEntity.getOriginalFileName(), "创建时上传文件原始名称不能为空！");
    Validate.notBlank(templateItemExcelEntity.getFilePath(), "创建时上传文件相对路径不能为空！");
    Validate.notNull(templateItemExcelEntity.getCreateDate(), "创建时上传日期不能为空！");
    // 因为一个表单明细模板只能绑定一个EXCEL，所以要查询之前的关联，删除掉
    TemplateItemExcelEntity exist =
        templateItemExcelRepository.findByTemplateItemId(
            templateItemExcelEntity.getTemplateItemId());
    if (null != exist) {
      this.delete(exist.getId());
    }
    templateItemExcelEntity.setProjectName(platformContext.getAppName());
    return templateItemExcelRepository.save(templateItemExcelEntity);
  }

  @Override
  public List<TemplateItemExcelEntity> findByTemplate(String templateId) {
    if (StringUtils.isBlank(templateId)) {
      return new ArrayList<>();
    }
    return templateItemExcelRepository.findByTemplateId(templateId);
  }

  @Override
  public List<TemplateItemExcelEntity> findByTemplateGroup(String templateGroupId) {
    if (StringUtils.isBlank(templateGroupId)) {
      return new ArrayList<>();
    }
    return templateItemExcelRepository.findByTemplateGroupId(templateGroupId);
  }

  @Override
  public TemplateItemExcelEntity findByTemplateItem(String templateItemId) {
    if (StringUtils.isBlank(templateItemId)) {
      return null;
    }
    return templateItemExcelRepository.findByTemplateItemId(templateItemId);
  }

  @Override
  @Transactional
  public void delete(String templateItemExcelEntityId) {
    TemplateItemExcelEntity excelEntity =
        this.templateItemExcelRepository.findById(templateItemExcelEntityId).orElse(null);
    if (excelEntity == null) {
      return;
    }
    this.fileUpdateService.deleteFile(
        excelEntity.getFilePath(), excelEntity.getOriginalFileName(), excelEntity.getFileName());
    this.templateItemExcelRepository.deleteById(templateItemExcelEntityId);
    this.templateItemExcelRepository.flush();
  }

  @Override
  @Transactional
  public void importData(TemplateEntity template, List<TemplateItemExcelEntity> templateItemExcels, MigrateImportModel importModel) {
    if(CollectionUtils.isEmpty(templateItemExcels)) {
      return;
    }
    TemplateEntity dbTemplate = templateService.findDetailsById(template.getId());
    Validate.notNull(dbTemplate, "未获取到保存的模版对象");
    for (TemplateItemExcelEntity itemExcel : templateItemExcels) {
      importModel.appendLine("导入模版明细项excel模版：分组属性名=%s, 明细项属性名=%s", itemExcel.getGroupPropertyName(), itemExcel.getItemPropertyName());
      itemExcel.setTemplateId(dbTemplate.getId());
      if(StringUtils.isBlank(itemExcel.getTemplateGroupId())) {
        // 该明细项在模版下面
        this.importData(dbTemplate.getItemRelations(), itemExcel, importModel);
      } else {
        // 该明细项在分组下面
        this.importDataByGroup(template.getGroupRelations(), itemExcel, importModel);
      }
    }
  }

  /**
   * 导入表单模版分组下的明细项的excel模版
   * @param groups
   * @param itemExcel 保证传入的对象已经对表单模版ID做了赋值
   * @param importModel
   */
  private void importDataByGroup(Set<TemplateGroupEntity> groups, TemplateItemExcelEntity itemExcel, MigrateImportModel importModel) {
    Validate.notEmpty(groups, "表单模版的分组为空");
    Set<TemplateGroupEntity> collect = groups.stream().filter(g -> g.getPropertyName().equals(itemExcel.getGroupPropertyName())).collect(Collectors.toSet());
    Validate.notEmpty(collect, "明细项导入模版未找到对应的分组属性：%s", itemExcel.getGroupPropertyName());
    TemplateGroupEntity group = collect.iterator().next();
    itemExcel.setTemplateGroupId(group.getId());
    this.importData(group.getItemRelations(), itemExcel, importModel);
  }

  /**
   * 导入表单模版明细项的
   * @param items
   * @param itemExcel 保证传入的对象已经对表单模版ID和分组ID(如果是分组下的明细)做了赋值
   * @param importModel
   */
  private void importData(Set<TemplateItemEntity> items, TemplateItemExcelEntity itemExcel, MigrateImportModel importModel) {
    Validate.notEmpty(items, "表单模版的明细项为空:分组属性=%s", itemExcel.getGroupPropertyName());
    Set<TemplateItemEntity> excelItems = items.stream().filter(i -> i.getPropertyName().equals(itemExcel.getItemPropertyName())).collect(Collectors.toSet());
    Validate.notEmpty(excelItems, "明细项导入模版未找到对应的明细项属性：分组属性名=%s, 明细项属性名=%s", itemExcel.getGroupPropertyName(), itemExcel.getItemPropertyName());
    TemplateItemEntity item = excelItems.iterator().next();
    itemExcel.setTemplateItemId(item.getId());
    itemExcel.setCreateDate(new Date());
    itemExcel.setId(null);
    this.save(itemExcel);
    // 从压缩包中读取excel模版并保存到本地系统中
    ZipFile zipFile = importModel.getZipFile();
    try {
      byte[] bytes = ZipFileUtils.readZipFile(zipFile, itemExcel.getFilePath(), itemExcel.getFileName());
      fileUpdateService.saveFile(itemExcel.getFilePath(), itemExcel.getOriginalFileName(), itemExcel.getFileName(), bytes);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("读取明细项导入模版出错：%s/%s", itemExcel.getFilePath(), itemExcel.getFileName()), e);
    }
  }

}
