package com.bizunited.platform.kuiper.starter.service.migrate.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.kuiper.entity.FrontFileEntity;
import com.bizunited.platform.kuiper.entity.ListTemplateEntity;
import com.bizunited.platform.kuiper.entity.MigrateExportEntity;
import com.bizunited.platform.kuiper.entity.PageEntity;
import com.bizunited.platform.kuiper.entity.PageEventEntity;
import com.bizunited.platform.kuiper.entity.PageFlowEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateEventEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemExcelEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.TemplateLayoutTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.migrate.MigrateExportRepository;
import com.bizunited.platform.kuiper.starter.service.FrontFileSevice;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import com.bizunited.platform.kuiper.starter.service.PageFlowService;
import com.bizunited.platform.kuiper.starter.service.TemplateItemExcelService;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.kuiper.starter.service.migrate.MigrateExportService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipOutputStream;

import static com.bizunited.platform.common.constant.MigrateDataConstants.FORM_TEMPLATE_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.FRONT_FILE_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.LIST_TEMPLATE_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.PAGE_FLOW_FILENAME;

/**
 * 基础工具套件中重要数据的迁出服务实现
 * @author yinwenjie
 */
@Service("kuiperMigrateExportService")
public class MigrateExportServiceImpl implements MigrateExportService {
  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateExportServiceImpl.class);
  @Autowired
  private MigrateExportRepository migrateExportRepository;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private UserService userService;
  @Autowired
  private PageFlowService pageFlowService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private ListTemplateService listTemplateService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private TemplateLayoutService templateLayoutService;
  @Autowired
  private TemplateItemExcelService templateItemExcelService;
  @Autowired
  private FrontFileSevice frontFileSevice;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public MigrateExportEntity save(MigrateExportEntity export) {
    Validate.notNull(export.getDataType(), "数据类型不能为空");
    UserVo user = SecurityUtils.getCurrentUser();
    export.setCreateTime(new Date());
    export.setCreateUser(user);
    export.setCreator(user.getAccount());
    export.setProjectName(platformContext.getAppName());
    return migrateExportRepository.save(export);
  }

  /**
   * 保存导出日志
   * @param dataType
   * @param datas
   * @return
   */
  private MigrateExportEntity save(Integer dataType, String datas) {
    Validate.notNull(dataType, "数据类型不能为空");
    MigrateExportEntity export = new MigrateExportEntity();
    export.setDataType(dataType);
    export.setDatas(datas);
    MigrateExportService migrateExportService = applicationContext.getBean(MigrateExportService.class);
    return migrateExportService.save(export);
  }

  @Override
  public List<MigrateExportEntity> findDetailsByDataType(Integer dataType) {
    if(dataType == null) {
      return Lists.newArrayList();
    }
    List<MigrateExportEntity> exports = null;
    String projectName = platformContext.getAppName();
    if(StringUtils.isNotBlank(projectName)){
      exports = migrateExportRepository.findByDataTypeAndProjectName(dataType, projectName);
    }else {
      exports = migrateExportRepository.findByDataTypeAndBlankProjectName(dataType);
    }
    if(CollectionUtils.isEmpty(exports)) {
      return Lists.newArrayList();
    }
    Map<String, UserVo> userCache = new HashMap<>();
    for (MigrateExportEntity export : exports) {
      String creator = export.getCreator();
      UserVo user = userCache.get(creator);
      if(user == null) {
        user = userService.findByAccount(creator);
        userCache.put(creator, user);
      }
      export.setCreateUser(user);
    }
    return exports;
  }

  @Override
  public byte[] exportListTempate(String[] ids) {
    List<ListTemplateEntity> templates;
    if(ArrayUtils.isEmpty(ids)) {
      ListTemplateEntity condition = new ListTemplateEntity();
      condition.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      templates = listTemplateService.findAllByConditions(condition);
    } else {
      int count = listTemplateService.countByIds(ids);
      Validate.isTrue(ids.length == count, "指定导出的数据视图信息与数据库存储的信息不符，请检查");
      templates = listTemplateService.findByIds(ids);
    }
    if(CollectionUtils.isEmpty(templates)) {
      return new byte[0];
    }
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(templates);
        ZipFileUtils.writeZipFile(zos, LIST_TEMPLATE_FILENAME, objBaos.toByteArray());
      }
      for (ListTemplateEntity template : templates) {
        this.writeFileToZip(zos, template.getRelativePath(), template.getFileName(), String.format("未找到模版文件：编码=%s，版本=%s", template.getCode(), template.getCversion()));
      }
      zos.finish();
      this.save(MigrateDataTypeEnum.LIST_TEMPLATE.getType(), StringUtils.join(ids, ","));
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  @Override
  public byte[] exportPageFlow(String[] ids) {
    List<PageFlowEntity> pageFlows;
    if(ArrayUtils.isEmpty(ids)) {
      pageFlows = pageFlowService.findAll();
    } else {
      long count = pageFlowService.countByIds(ids);
      Validate.isTrue(count ==  ids.length, "指定导出的数据视图信息与数据库存储的信息不符，请检查");
      pageFlows = pageFlowService.findDetailsByIds(ids);
    }
    if(CollectionUtils.isEmpty(pageFlows)) {
      return new byte[0];
    }
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(pageFlows);
        ZipFileUtils.writeZipFile(zos, PAGE_FLOW_FILENAME, objBaos.toByteArray());
      }
      this.writePageFlowFile(zos, pageFlows);
      zos.finish();
      this.save(MigrateDataTypeEnum.PAGE_FLOW.getType(), StringUtils.join(ids, ","));
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  @Override
  public byte[] exportFormTemplate(String[] ids) {
    List<TemplateEntity> templates;
    if(ArrayUtils.isEmpty(ids)) {
      TemplateEntity condition = new TemplateEntity();
      condition.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      templates = templateService.findAllByConditions(condition);
    } else {
      int count = templateService.countByIds(ids);
      Validate.isTrue(count ==  ids.length, "指定导出的数据视图信息与数据库存储的信息不符，请检查");
      templates = templateService.findDetailsByIds(ids);
    }
    if(CollectionUtils.isEmpty(templates)) {
      return new byte[0];
    }
    Collection<TemplateEntity> collection = nebulaToolkitService.copyCollectionByWhiteList(templates, TemplateEntity.class, TemplateEntity.class, HashSet.class, ArrayList.class,
        "properties", "properties.currentGroup", "properties.currentItem", "relations", "relations.currentView",
        "relations.currentGroup", "relations.currentItem", "itemRelations.parentGroup", "itemRelations.properties", "itemRelations.relations",
        "itemRelations.relations.currentView", "event", "visibility", "visibility.attributes", "visibility.buttons", "maintainers");
    List<TemplateEntity> exportTemplates = new ArrayList<>(collection);
    // 加载表单模版的布局文件
    this.loadFormTemplateLayout(exportTemplates);
    Map<String, List<TemplateItemExcelEntity>> templateItemExcel = this.getTemplateItemExcel(exportTemplates);
    // 写入对象信息和相关文件到zip包中
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(exportTemplates);
        oos.writeObject(templateItemExcel);
        ZipFileUtils.writeZipFile(zos, FORM_TEMPLATE_FILENAME, objBaos.toByteArray());
      }
      this.writeTemplateFile(zos, exportTemplates, templateItemExcel);
      zos.finish();
      this.save(MigrateDataTypeEnum.FORM_TEMPLATE.getType(), StringUtils.join(ids, ","));
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  /**
   * 写模版中的文件到压缩文件中,事件的文件，excel导入配置的文件
   */
  private void writeTemplateFile(ZipOutputStream zos, List<TemplateEntity> exportTemplates, Map<String, List<TemplateItemExcelEntity>> templateItemExcel) throws IOException {
    for (TemplateEntity template : exportTemplates) {
      Set<TemplateEventEntity> events = template.getEvent();
      if(CollectionUtils.isEmpty(events)) {
        continue;
      }
      for (TemplateEventEntity event : events) {
        this.writeFileToZip(zos, event.getRelativePath(), event.getFileName(), String.format("未找到表单模版的事件内容：%s/%s", event.getRelativePath(), event.getFileName()));
      }
    }
    for (List<TemplateItemExcelEntity> value : templateItemExcel.values()) {
      if(CollectionUtils.isEmpty(value)) {
        continue;
      }
      for (TemplateItemExcelEntity excel : value) {
        this.writeFileToZip(zos, excel.getFilePath(), excel.getFileName(), String.format("未找到表单明细导入模版：%s/%s", excel.getFilePath(), excel.getFileName()));
      }
    }
  }

  /**
   * 获取表单明细项的excel导入模版，以表单模版的code|cversion为key，对应模版的excel导入配置为value保存在map中
   * 方便在导入的时候快速的找到对应模版的excel导入配置
   * @param exportTemplates
   * @return
   */
  private Map<String, List<TemplateItemExcelEntity>> getTemplateItemExcel(List<TemplateEntity> exportTemplates) {
    Map<String, List<TemplateItemExcelEntity>> excelMap = new HashMap<>();
    for (TemplateEntity template : exportTemplates) {
      List<TemplateItemExcelEntity> itemExcels = templateItemExcelService.findByTemplate(template.getId());
      String key = String.format("%s|%s", template.getCode(), template.getCversion());
      excelMap.put(key, itemExcels);
    }
    return excelMap;
  }

  /**
   * 加载表单模版的布局文件,PC、手机端、打印端
   * @param exportTemplates
   */
  private void loadFormTemplateLayout(List<TemplateEntity> exportTemplates) {
    for (TemplateEntity template : exportTemplates) {
      JSONObject pcLayout = templateLayoutService.findDetailsByTemplateId(template.getId(), TemplateLayoutTypeEnum.PC.getType());
      Validate.notNull(pcLayout, "未找到表单模版的布局文件：模版编号=%s, 版本号=%s", template.getCode(), template.getCversion());
      JSONObject mobileLayout = templateLayoutService.findDetailsByTemplateId(template.getId(), TemplateLayoutTypeEnum.MOBILE.getType());
      JSONObject printLayout = templateLayoutService.findDetailsByTemplateId(template.getId(), TemplateLayoutTypeEnum.PRINT.getType());
      template.setTemplateLayout(pcLayout);
      template.setMobileTemplateLayout(mobileLayout);
      template.setPrintTemplateLayout(printLayout);
    }
  }

  /**
   * 写页面流的文件到压缩文件中
   * @param zos
   * @param pageFlows
   * @throws IOException
   */
  private void writePageFlowFile(ZipOutputStream zos, List<PageFlowEntity> pageFlows) throws IOException {
    for (PageFlowEntity pageFlow : pageFlows) {
      this.writeFileToZip(zos, pageFlow.getRelativePath(), pageFlow.getFileName(), String.format("未找到页面流文件：%s", pageFlow.getCode()));
      Set<PageEntity> pages = pageFlow.getPages();
      if(CollectionUtils.isEmpty(pages)) {
        continue;
      }
      for (PageEntity page : pages) {
        this.writeFileToZip(zos, page.getRelativePath(), page.getFileName(), String.format("未找到页面文件：%s", page.getCode()));
        Set<PageEventEntity> events = page.getEvents();
        if(CollectionUtils.isEmpty(events)) {
          continue;
        }
        for (PageEventEntity event : events) {
          this.writeFileToZip(zos, event.getRelativePath(), event.getFileName(), String.format("未找到事件文件：%s", event.getEventId()));
        }
      }
    }
  }

  /**
   * 写文件到zip文件中
   * @param zos
   * @param relativePath
   * @param fileName
   * @param errorMsg
   */
  private void writeFileToZip(ZipOutputStream zos, String relativePath, String fileName, String errorMsg) throws IOException {
    byte[] pageBytes = venusFileService.readFileContent(relativePath, fileName);
    Validate.isTrue(ArrayUtils.isNotEmpty(pageBytes), "%s, {文件路径：%s,文件名:%s}", errorMsg, relativePath, fileName);
    ZipFileUtils.writeZipFile(zos, relativePath, fileName, pageBytes);
  }

  /**
   * 导出指定编号的页面函数。并以压缩文件（zip文件）下载的形式返回内容
   * @param frontFileIds
   * @return
   */
  @Override
  public byte[] exportFrontFile(String[] frontFileIds) {
    List<FrontFileEntity> frontFiles;
    if(frontFileIds == null || frontFileIds.length == 0) {
      frontFiles = frontFileSevice.findAll();
    } else {
      int count = frontFileSevice.countByIds(frontFileIds);
      Validate.isTrue(count == frontFileIds.length, "指定导出的编码规则信息与数据库存储的信息不符，请检查");
      Set<FrontFileEntity> detailsByIds = frontFileSevice.findByIds(frontFileIds);
      frontFiles = Lists.newArrayList(detailsByIds);
    }
    if(CollectionUtils.isEmpty(frontFiles)) {
      return new byte[0];
    }

    try (ByteArrayOutputStream fullZipBis = new ByteArrayOutputStream();
         ZipOutputStream zipf = new ZipOutputStream(fullZipBis);) {
      try(ByteArrayOutputStream bos = new ByteArrayOutputStream();
          ObjectOutputStream oos = new ObjectOutputStream(bos);) {
        oos.writeObject(frontFiles);
        for (FrontFileEntity frontFile : frontFiles) {
          // 将动态脚本关联的脚本附件写入压缩文件
          String relativePath = frontFile.getFilePath();
          String fileName = frontFile.getFileName();
          byte[] bytes = venusFileService.readFileContent(relativePath, fileName);
          Validate.isTrue(bytes != null && bytes.length > 0, "未找到指定文件的内容，请检查！！");
          ZipFileUtils.writeZipFile(zipf, relativePath, fileName, bytes);
        }
        ZipFileUtils.writeZipFile(zipf, FRONT_FILE_FILENAME, bos.toByteArray());
      }
      zipf.finish();
      this.save(MigrateDataTypeEnum.FRONT_FILE.getType(), StringUtils.join(frontFileIds, ","));
      return fullZipBis.toByteArray();
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }
}
