package com.bizunited.platform.kuiper.starter.service.migrate.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.FileUtils;
import com.bizunited.platform.kuiper.entity.MigrateImportDetailEntity;
import com.bizunited.platform.kuiper.entity.MigrateImportEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.migrate.MigrateImportDetailRepository;
import com.bizunited.platform.kuiper.starter.repository.migrate.MigrateImportRepository;
import com.bizunited.platform.kuiper.starter.service.FrontFileSevice;
import com.bizunited.platform.kuiper.starter.service.ListTemplateService;
import com.bizunited.platform.kuiper.starter.service.PageFlowService;
import com.bizunited.platform.kuiper.starter.service.migrate.MigrateImportService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.bizunited.platform.venus.common.service.image.FileUpdateService;
import com.bizunited.platform.venus.common.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.MigrateDataConstants.ZIP_FILE_SUBFIX;

/**
 * 数据信息迁入的查询，分析，执行。包括基础配置信息与表单信息
 * @author: yanwe
 * @date: 03/Sep/2019 15:47
 */
@Service("kuiperMigrateImportService")
public class MigrateImportServiceImpl implements MigrateImportService {

  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateImportServiceImpl.class);

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Autowired
  private MigrateImportRepository migrateImportRepository;
  @Autowired
  private MigrateImportDetailRepository migrateImportDetailRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private PageFlowService pageFlowService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private ListTemplateService listTemplateService;
  @Autowired
  private FrontFileSevice frontFileSevice;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public MigrateImportEntity upload(MultipartFile file, Integer dataType) {
    Validate.notNull(file, "上传文件不能为空！");
    Validate.notNull(dataType, "数据类型不能为空");
    MigrateDataTypeEnum dataTypeEnum = MigrateDataTypeEnum.valueOfType(dataType);
    Validate.notNull(dataTypeEnum, "不支持的数据类型：%d", dataType);
    UserVo userVo = SecurityUtils.getCurrentUser();
    MigrateImportEntity migrateImport = new MigrateImportEntity();
    migrateImport.setCreateTime(new Date());
    migrateImport.setExecuted(false);
    migrateImport.setDataType(dataType);
    migrateImport.setCreator(userVo.getAccount());
    migrateImport.setOriginalFileName(file.getOriginalFilename());
    MultipartFile[] files = new MultipartFile[] {file};
    List<OrdinaryFileVo> ordinaryFileEntities = fileUpdateService.fileUpload("migrate", userVo.getAccount(), null, files);
    Validate.notEmpty(files, "保存上传文件失败，请检查！");
    OrdinaryFileVo fileEntity = ordinaryFileEntities.get(0);
    Validate.notNull(fileEntity,"上次文件保存返回结果不能为空！");
    migrateImport.setFileName(fileEntity.getFileName());
    migrateImport.setRelativeLocal(fileEntity.getRelativeLocal());
    migrateImport.setProjectName(platformContext.getAppName());
    return this.create(migrateImport);
  }

  @Override
  @Transactional
  public MigrateImportEntity create(MigrateImportEntity migrateImport) {
    Validate.notNull(migrateImport.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(migrateImport.getRelativeLocal(), "迁入文件在本地路径不能为空！");
    Validate.notBlank(migrateImport.getFileName(), "迁入文件重命名后的文件名字不能为空！");
    Validate.notNull(migrateImport.getCreator(), "迁入文件上传人不能为空！");
    Validate.notNull(migrateImport.getCreateTime(), "迁入文件上传时间不能为空！");
    Validate.notNull(migrateImport.getExecuted(), "迁入文件任务是否被执行不能为空！");
    migrateImport.setProjectName(platformContext.getAppName());
    return migrateImportRepository.saveAndFlush(migrateImport);
  }

  @Override
  public List<MigrateImportEntity> findDetailsByDataType(Integer dataType) {
    if(dataType == null) {
      return Lists.newArrayList();
    }
    List<MigrateImportEntity> imports = null;
    String projectName = platformContext.getAppName();
    if(StringUtils.isNotBlank(projectName)){
      imports = migrateImportRepository.findByDataTypeAndProjectName(dataType, projectName);
    }else {
      imports = migrateImportRepository.findByDataTypeAndBlankProjectName(dataType);
    }
    if(CollectionUtils.isEmpty(imports)) {
      return Lists.newArrayList();
    }
    return this.loadUsers(imports);
  }

  /**
   * 加载用户信息
   * @param imports
   * @return
   */
  private List<MigrateImportEntity> loadUsers(List<MigrateImportEntity> imports) {
    Map<String, UserVo> tmp = new HashMap<>();
    for (MigrateImportEntity anImport : imports) {
      String creator = anImport.getCreator();
      UserVo createUser = tmp.get(creator);
      if(createUser == null) {
        createUser = userService.findByAccount(creator);
        tmp.put(creator, createUser);
      }
      anImport.setCreateUser(createUser);
      String executor = anImport.getExecutor();
      if(StringUtils.isNotBlank(executor)) {
        UserVo executeUser = tmp.get(executor);
        if(executeUser == null) {
          executeUser = userService.findByAccount(executor);
          tmp.put(executor, executeUser);
        }
        anImport.setExecuteUser(executeUser);
      }
    }
    return imports;
  }

  @Override
  public MigrateImportEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    MigrateImportEntity migrateImportEntity = migrateImportRepository.findById(id).orElse(null);
    if(migrateImportEntity == null){
      return null;
    }
    if(migrateImportEntity.getDetail() == null){
      return null;
    }
    return migrateImportEntity;
  }

  @Override
  @Transactional
  public MigrateImportEntity execute(String id, Integer mode) {
    /*
     * 1、验证入参
     * 2、获取导入的文件内容
     * 3、将文件内容存到本地的临时文件中，并加载压缩文件
     * 4、根据数据类型调用对应的导入方法
     * 5、删除临时文件
     * 6、保存明细信息并返回
     *
     * */
    // 1、
    Validate.notBlank(id, "导入记录ID不能为空");
    ImportExecuteModeEnum executeMode;
    if(mode == null) {
      executeMode = ImportExecuteModeEnum.SKIP;
    } else {
      executeMode = ImportExecuteModeEnum.valueOfMode(mode);
      Validate.notNull(executeMode, "不支持的执行模式：%s", mode);
    }
    UserVo userVo = SecurityUtils.getCurrentUser();
    // 2
    //注意：此处不能用findDetailsById，可能会引起null结果
    MigrateImportEntity migrateImport = migrateImportRepository.findById(id).orElse(null);
    Validate.notNull(migrateImport, "未找到导入的记录");
    //Validate.isTrue(!migrateImport.getExecuted(), "该导入已经执行过了！！");
    MigrateDataTypeEnum dataType = MigrateDataTypeEnum.valueOfType(migrateImport.getDataType());
    Validate.notNull(dataType, "不支持导入的数据类型:%s", migrateImport.getDataType());
    byte[] bytes = venusFileService.readFileContent(migrateImport.getRelativeLocal(), migrateImport.getFileName());
    MigrateImportDetailEntity importDetail = migrateImport.getDetail();
    if(importDetail == null) {
      importDetail = new MigrateImportDetailEntity();
    }
    Validate.isTrue(bytes != null && bytes.length > 0, "导入的文件内容为空");
    // 3
    File tmpFile = FileUtils.writeLocalFile(bytes, fileRoot, ZIP_FILE_SUBFIX);
    int executeResult = NormalStatusEnum.ENABLE.getStatus();
    try (ZipFile zipFile = new ZipFile(tmpFile)) {
      MigrateImportModel importModel = new MigrateImportModel(zipFile, executeMode);
      // 4
      try {
        switch (dataType) {
          case LIST_TEMPLATE:
            listTemplateService.importData(importModel);
            break;
          case PAGE_FLOW:
            pageFlowService.importData(importModel);
            break;
          case FORM_TEMPLATE:
            templateService.importData(importModel);
            break;
          case FRONT_FILE:
            frontFileSevice.importData(importModel);
            break;
          default:
            importModel.setError(true);
            importModel.appendLine("导入错误：不支持的数据类型【%s】", dataType.name());
            break;
        }
        importModel.appendLine("导入完成！");
      } catch (RuntimeException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.setError(true);
        importModel.append("导入错误：").appendLine(e.getMessage());
      } finally {
        importDetail.setCreateCount(importModel.getCreateCount());
        importDetail.setIgnoreCount(importModel.getSkipCount());
        importDetail.setTotalCount(importModel.getTotalCount());
        importDetail.setUpdateCount(importModel.getUpdateCount());
        importDetail.setExecuteLog(importModel.getExecuteLog().toString());
        if(importModel.isError()) {
          executeResult = NormalStatusEnum.DISABLE.getStatus();
        }
      }
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("读取压缩文件出错：%s", e.getMessage()));
    } finally {
      // 5
      tmpFile.delete();
    }
    // 6
    importDetail.setProjectName(platformContext.getAppName());
    migrateImportDetailRepository.saveAndFlush(importDetail);
    migrateImport.setDetail(importDetail);
    migrateImport.setExecuted(true);
    migrateImport.setExecutor(userVo.getAccount());
    migrateImport.setExecuteTime(new Date());
    migrateImport.setExecuteResult(executeResult);
    migrateImport.setProjectName(platformContext.getAppName());
    migrateImportRepository.saveAndFlush(migrateImport);
    return migrateImport;
  }

}
