package com.bizunited.platform.kuiper.starter.common.excel;

import com.bizunited.platform.kuiper.starter.common.excel.reader.IExcelReader;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.lang3.Validate;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

/**
 * excel导入操作基础封装
 *
 * @author Keller
 * @create 2020/8/24
 */
public class ExcelImportWrapper {

  public static final Logger LOGGER = LoggerFactory.getLogger(ExcelImportWrapper.class);

  /**
   * 对象转化工具
   */
  public static final ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();

  static {
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          return (T) DateUtil.getJavaDate(Double.parseDouble(value.toString()));
        }
        return null;
      }
    }, Date.class);
    convertUtilsBean.register(new Converter() {

      @SuppressWarnings("unchecked")
      @Override
      public <T> T convert(Class<T> type, Object value) {
        if (value != null) {
          String src = value.toString();
          if (src.matches("\\d+(\\.\\d+)?")) {
            BigDecimal bigDecimal = new BigDecimal(value.toString());
            return (T) new Integer(bigDecimal.intValue());
          }
        }
        return null;
      }
    }, Integer.class);
  }

  /**
   * POI Excel Workbook
   */
  private Workbook wb;
  /**
   * 用于读取excel
   */
  private IExcelReader excelReader;

  /**
   * @param excelReader excel读取器
   */
  public ExcelImportWrapper(IExcelReader excelReader) {
    this.excelReader = excelReader;
    if (excelReader instanceof IObgainWorkbook) {
      wb = ((IObgainWorkbook) excelReader).getWorkbook();
    }
  }

  /***
   * 获取Workbook 主要用于写入
   *
   * @return
   */
  public Workbook getWorkbook() {
    if (wb == null) {
      throw new UnsupportedOperationException("不支持该操作");
    }
    return wb;
  }

  /**
   * 获取已经存在的sheet
   *
   * @param name sheet名称
   * @return
   */
  public Sheet getSheet(String name) {
    Validate.notBlank(name, "sheet名称不能为空");
    return getWorkbook().getSheet(name);
  }

  /***
   * 获取已经存在的sheet
   *
   * @param sheetIndex sheet索引
   * @return
   */
  public Sheet getSheet(int sheetIndex) {
    Validate.isTrue(sheetIndex > -1, "sheet索引小于0");
    return getWorkbook().getSheetAt(sheetIndex);
  }

  /**
   * 从某一行开始读取所有数据，起始索引位置为0
   *
   * @param sheetIndex     sheet索引
   * @param columnEndIndex 哪一列索引结束
   * @param interceptor    每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int columnEndIndex, IRecordInterceptor<Object[]> interceptor) {
    excelReader.readSheet(sheetIndex, 0, 0, columnEndIndex, null, interceptor);
  }

  /**
   * 从某一行开始读取所有数据，起始索引位置为0
   *
   * @param sheetIndex       sheet索引
   * @param rowBeginIndex    哪一行索引开始
   * @param columnBeginIndex 哪一列索引开始
   * @param columnEndIndex   哪一列索引结束
   * @param classs           每一列数据的类型，根据所填入的类型自动进行数据类型转换
   * @param interceptor      每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    excelReader.readSheet(sheetIndex, rowBeginIndex, columnBeginIndex, columnEndIndex, classs, interceptor);
  }

  /**
   * 从某一行开始读取所有数据，起始索引位置为0
   *
   * @param sheetIndex       sheet索引
   * @param rowBeginIndex    哪一行索引开始
   * @param columnBeginIndex 哪一列索引开始
   * @param columnEndIndex   哪一列索引结束
   * @param interceptor      每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, IRecordInterceptor<Object[]> interceptor) {
    excelReader.readSheet(sheetIndex, rowBeginIndex, columnBeginIndex, columnEndIndex, null, interceptor);
  }

  /**
   * 从某一行开始读取所有数据，起始索引位置为0
   *
   * @param sheetIndex       sheet索引
   * @param rowBeginIndex    哪一行索引开始
   * @param columnBeginIndex 哪一列索引开始
   * @param columnEndIndex   哪一列索引结束
   * @param interceptor      每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int rowBeginIndex, int columnBeginIndex, int columnEndIndex, List<Class<?>> classList, IRecordInterceptor<Object[]> interceptor) {
    Class<?>[] classs = new Class<?>[classList.size()];
    classList.toArray(classs);
    excelReader.readSheet(sheetIndex, rowBeginIndex, columnBeginIndex, columnEndIndex, classs, interceptor);
  }

  /**
   * 读取某些行数据
   *
   * @param sheetIndex     sheet索引
   * @param columnEndIndex 哪一列索引结束
   * @param rowIndexs      需要读取的行索引
   * @param interceptor    每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int columnEndIndex, int[] rowIndexs, IRecordInterceptor<Object[]> interceptor) {
    excelReader.readSheet(sheetIndex, 0, columnEndIndex, rowIndexs, null, interceptor);
  }

  /**
   * 读取某些行数据
   *
   * @param sheetIndex       sheet索引
   * @param columnBeginIndex 哪一列索引开始
   * @param columnEndIndex   哪一列索引结束
   * @param rowIndexs        需要读取的行索引
   * @param classs           每一列数据的类型，根据所填入的类型自动进行数据类型转换
   * @param interceptor      每读取一行数据，调用一次
   */
  public void readSheet(int sheetIndex, int columnBeginIndex, int columnEndIndex, int[] rowIndexs, Class<?>[] classs, IRecordInterceptor<Object[]> interceptor) {
    excelReader.readSheet(sheetIndex, columnBeginIndex, columnEndIndex, rowIndexs, classs, interceptor);
  }

  /**
   * 读取某一行数据
   *
   * @param sheetIndex     sheet索引
   * @param rowIndex       需要读取的行索引
   * @param columnEndIndex 哪一列索引结束
   * @return
   */
  public Object[] readRow(int sheetIndex, int rowIndex, int columnEndIndex) {
    return excelReader.readRow(sheetIndex, rowIndex, 0, columnEndIndex, null);
  }

  /**
   * 读取行数据
   *
   * @param sheetIndex       sheet索引
   * @param rowIndex         需要读取的行索引
   * @param columnBeginIndex 哪一列索引开始
   * @param columnEndIndex   哪一列索引结束
   * @param classs           每一列数据的类型，根据所填入的类型自动进行数据类型转换
   * @return
   */
  public Object[] readRow(int sheetIndex, int rowIndex, int columnBeginIndex, int columnEndIndex, Class<?>[] classs) {
    return excelReader.readRow(sheetIndex, rowIndex, columnBeginIndex, columnEndIndex, classs);
  }
}
