package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.service.DynamicTemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.dao.InvalidDataAccessApiUsageException;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;

@Api(value = "DynamicTemplateController")
@RestController
@RequestMapping("/v1/kuiper/dynamicTemplates")
public class DynamicTemplateController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicTemplateController.class);

  @Autowired
  private DynamicTemplateService dynamicTemplateService;

  @ApiOperation(value = "初始化动态模板，包括动态模板中的一般属性、关联属性、明细编辑属性和分组属性，包括建立个主要数据表和明细、分组关联数据表"  , notes="")
  @PostMapping("/initDynamicTemplate")
  @CacheEvict(cacheNames="template" , allEntries=true)
  public ResponseModel initDynamicTemplate(@RequestBody @ApiParam(name = "template", value = "新的表单模板信息，至少说明版本信息、模板类型、已经使用的静态类全名，其它不需要指定了") TemplateEntity template) {
    try {
      Principal principal = this.getPrincipal();
      TemplateEntity result = this.dynamicTemplateService.initDynamicTemplate(template, principal, false);
      return this.buildHttpResultW(result, "properties", "relations","creator",
              "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
              "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
              "itemRelations","itemRelations.properties","itemRelations.relations",
              "event","visibility","visibility.attributes","visibility.buttons");
    } catch (InvalidDataAccessApiUsageException e) {
      LOGGER.error(e.getMessage(), e);
      Throwable rootException = e.getRootCause();
      return this.buildHttpResultForException((Exception)rootException);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对指定的动态表单模板进行升级处理，通过这个接口服务系统将自动保留templateId对应的模板的全部主要设定（包括模型、数据源、事件、可见性）")
  @PutMapping("")
  @CacheEvict(cacheNames="template" , allEntries=true)
  public ResponseModel upgrade(@RequestParam(name="templateId") @ApiParam(name = "templateId", value = "指定的表单模板升级时使用的参考模板") String templateId ,
                               @RequestBody @ApiParam(name = "newTemplate", value = "新的完成的动态表单模板结构，id、formInstanceId实例字段不需要传入") TemplateEntity newTemplate ,
                               @RequestParam(name="newVersion") @ApiParam(name = "newVersion", value = "新的版本号") String newVersion,
                               @ApiParam(value = "更新模版实例使用当前模版版本") @RequestParam(name="updateInstance", required = false, defaultValue = "0") Boolean updateInstance) {
    try {
      Principal principal = this.getPrincipal();
      TemplateEntity result = this.dynamicTemplateService.upgrade(templateId, newTemplate, newVersion, updateInstance, principal);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对指定的动态表单模板进行升级处理时，需要获取原有模板的结构信息，同时过滤掉id，formInstanceId以及回溯关系等")
  @GetMapping("/findStructByTemplateId")
  public ResponseModel findStructByTemplateId(@RequestParam(name="templateId") @ApiParam(name = "templateId", value = "指定的表单模板升级时使用的参考模板") String templateId) {
    try {
      TemplateEntity result = this.dynamicTemplateService.findStructByTemplateId(templateId);
      return this.buildHttpResultW(result, "properties", "relations",
              "groupRelations","groupRelations.properties","groupRelations.relations","groupRelations.itemRelations",
              "groupRelations.itemRelations.properties","groupRelations.itemRelations.relations",
              "itemRelations","itemRelations.properties","itemRelations.relations");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}