package com.bizunited.platform.kuiper.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxDetailsEntity;
import com.bizunited.platform.kuiper.entity.FormDetailsImportBoxEntity;
import com.bizunited.platform.kuiper.starter.service.FormDetailsImportBoxService;
import com.bizunited.platform.kuiper.starter.vo.FormDetailsImportBoxVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import static com.bizunited.platform.common.constant.MigrateDataConstants.EXPORT_FILENAME_DATE_FORMAT;

/**
 * 基于表单引擎，列表引擎，对业务数据通过xls文件的方式进行导入，并关联相关表单实例的服务的http restful接口层实现
 * 实现工具箱方式导入文件。1、上传文件。2、异步导入文件
 *
 * @author Keller
 * @create 2020/8/23
 */
@Api(value = "FormDetailsImportBoxController")
@RestController
@RequestMapping("/v1/kuiper/formDetailsImportBox")
public class FormDetailsImportBoxController extends BaseController {

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FormDetailsImportBoxController.class);

  private FormDetailsImportBoxService formDetailsImportBoxService;

  public FormDetailsImportBoxController(FormDetailsImportBoxService formDetailsImportBoxService) {
    this.formDetailsImportBoxService = formDetailsImportBoxService;
  }

  /**
   * 执行导入任务
   *
   * @param formDetailsImportBoxVo
   * @param request
   * @return
   */
  @PostMapping("/execute")
  @ApiOperation("执行导入")
  public ResponseModel execute(@Valid FormDetailsImportBoxVo formDetailsImportBoxVo,
                               HttpServletRequest request) {
    // 构建可能的params信息
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }
    try {
      FormDetailsImportBoxEntity entity = formDetailsImportBoxService.execute(formDetailsImportBoxVo, params);
      return this.buildHttpResultW(entity, "creator", "executor");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取所有导入记录信息（分页）
   *
   * @return
   */
  @ApiOperation(value = "查询上传工具箱列表")
  @GetMapping("/queryAll")
  public ResponseModel queryAll(@ApiParam("分页信息") @PageableDefault(value = 5) Pageable pageable) {
    try {
      Page<FormDetailsImportBoxEntity> entitys = formDetailsImportBoxService.queryPage(pageable);
      return this.buildHttpResultW(entitys, "creator", "executor");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 清除导入记录
   *
   * @return
   */
  @ApiOperation(value = "清除导入记录")
  @DeleteMapping("/{id}")
  public ResponseModel deleteById(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      formDetailsImportBoxService.deleteById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消当前导入任务
   *
   * @return
   */
  @ApiOperation(value = "取消当前导入任务")
  @PutMapping("/{id}")
  public ResponseModel cancel(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      formDetailsImportBoxService.cancel(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查看导入单条信息
   *
   * @return
   */
  @ApiOperation(value = "查看导入单条信息")
  @GetMapping("/{id}")
  public ResponseModel query(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      FormDetailsImportBoxEntity entity = formDetailsImportBoxService.findById(id);
      return this.buildHttpResultW(entity, "creator", "executor");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查看导入详情信息
   *
   * @return
   */
  @ApiOperation(value = "查看导入详情信息")
  @GetMapping("/details/{id}")
  public ResponseModel queryPageById(@PathVariable("id") @ApiParam(value = "主键", required = true) String id) {
    try {
      Page<FormDetailsImportBoxDetailsEntity> page = formDetailsImportBoxService.queryPageById(id);
      return this.buildHttpResultW(page.getContent(), "creator", "executor");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 下载错误文件
   *
   * @return
   */
  @ApiOperation(value = "下载错误文件")
  @GetMapping("/errorFile/{id}")
  public void errorFile(@PathVariable("id") @ApiParam(value = "主键", required = true) String id,
                        HttpServletRequest request,
                        HttpServletResponse response) {
    try {
      byte[] bytes = formDetailsImportBoxService.exportErrorFile(id);
      FormDetailsImportBoxEntity entity = formDetailsImportBoxService.findById(id);

      Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "没有任何下载信息");
      String fileName = StringUtils.join(entity.getOriginalFileName());
      String time = new SimpleDateFormat(EXPORT_FILENAME_DATE_FORMAT).format(new Date());
      //重新组装下载文件名
      fileName = fileName.substring(0, fileName.lastIndexOf(".")) + "-" + time + "." + fileName.substring(fileName.lastIndexOf(".") + 1);
      writeResponseFile(request, response, bytes, fileName);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      printResponseModel(buildHttpResultForException(e), response);
    }
  }

  @ApiOperation(value = "通过该接口，系统将可以返回该进程中已经被扫描到的所有业务数据导入处理器的类名信息")
  @GetMapping(value = "/findProcessClassNames")
  public ResponseModel findProcessClassNames() {
    try {
      Set<String> results = this.formDetailsImportBoxService.findProcessClassNames();
      return this.buildHttpResultW(results, new String[]{});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "清除导入列表所有数据")
  @DeleteMapping(value = "/deleteAll")
  public ResponseModel clearAll() {
    try {
      formDetailsImportBoxService.deleteAll();
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
