package com.bizunited.platform.kuiper.starter.controller;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseCode;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.kuiper.entity.InstanceActivityLogEntity;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Date;
import java.util.Set;

@RestController
@RequestMapping("/v1/kuiper/activityLogs")
public class InstanceActivityLogController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(InstanceActivityLogController.class);
  @Autowired
  private InstanceActivityLogService instanceActivityLogService;
  
  /**
   * 查询指定活动实例下的数据变化日志，并按照时间顺序进行返回
   * @param instanceActivityId 活动实例编号
   * @return
   */
  @GetMapping("/findDetailsByInstanceActivity")
  @ApiOperation(value = "按照活动编号，查询指定的活动信息详情——包括关联信息，并按照时间顺序进行返回")
  public ResponseModel findDetailsByInstanceActivity(@RequestParam("instanceActivityId") String instanceActivityId) {
    try {
      Set<InstanceActivityLogEntity> logs = this.instanceActivityLogService.findDetailsByInstanceActivity(instanceActivityId);
      return this.buildHttpResultW(logs, "instanceActivity", "creator");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 查询指定活动变化日志的详情，注意，这里只是查询数据库中存储的信息，并不包括变化日志的json结构信息
   * @param instanceActivityLogId 指定的实例活动变化日志编号
   * @return
   */
  @GetMapping("/findDetailsById")
  @ApiOperation(value = "按照活动编号，查询指定的活动信息详情——包括关联信息")
  public ResponseModel findDetailsById(@RequestParam("instanceActivityLogId") String instanceActivityLogId) {
    try {
      InstanceActivityLogEntity log = this.instanceActivityLogService.findDetailsById(instanceActivityLogId);
      return this.buildHttpResultW(log, "instanceActivity" , "creator");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 查询指定活动变化日志的详细内容，注意，这里查询的是变化详细内容的json信息。
   * @param instanceActivityLogId 指定的实例活动变化日志编号
   * @return
   */
  @GetMapping("/findContentById")
  @ApiOperation(value = "查询指定活动变化日志的详细内容" , notes="注意，这里查询的是变化详细内容的json信息。")
  public ResponseModel findContentById(@RequestParam("instanceActivityLogId") String instanceActivityLogId) {
    try {
      JSONObject jsonObject = this.instanceActivityLogService.findContentById(instanceActivityLogId);
      return this.buildHttpResultW(jsonObject);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照指定的实例编号，查询这个实例编号下最后一次发生的数据变化日志
   * @param instanceId 指定的实例编号
   */
  @GetMapping("/findTop/{instanceId}")
  @ApiOperation(value = "按照指定的实例编号，查询这个实例编号下最后一次发生的数据变化日志" , notes="返回的数据中有一个字符串，说明当前最后一次操作日志的编号")
  public ResponseModel findTopByInstanceId(@PathVariable("instanceId") String instanceId) {
    try {
      String logid = this.instanceActivityLogService.findTopByInstanceId(instanceId);
      ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
      result.setResponseCode(ResponseCode.E0);
      result.setSuccess(true);
      result.setData(logid);
      return result;
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照指定的数据变更日志对表单数据进行恢复。注意：这个日志必须是当前表单实例最后一次变动的日志。
   * 另外，当前方法只能恢复当前日志项，并不提供数据日志依据时间线的批量恢复
   * @param instanceActivityLogId 指定的实例活动编号
   * @return 如果恢复成功，数据的回复情况将在数据库中体现出来，且恢复后的数据json情况也将被返回
   */
  @PostMapping("/recovery/{instanceActivityLogId}")
  @ApiOperation(value = "按照指定的数据变更日志对表单数据进行恢复。注意：这个日志必须是当前表单实例最后一次变动的日志" , notes="当前方法只能恢复当前日志项，并不提供数据日志依据时间线的批量恢复")
  public ResponseModel recovery(@PathVariable("instanceActivityLogId") String instanceActivityLogId) {
    try {
      JSONObject jsonObject = this.instanceActivityLogService.recoveryByInstanceActivityLogId(instanceActivityLogId);
      return this.buildHttpResultW(jsonObject);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 基于指定的实例活动编号，将对应的表单实例从最近的一次变化日志开始进行数据恢复，
   * 直到数据恢复到这个指定的实例活动编号对应的最后一次数据变化日志为止。<p>
   * @param targetInstanceActivityId 指定的实例活动编号
   * @return 如果恢复成功，数据的回复情况将在数据库中体现出来，且恢复后的数据json情况也将被返回
   */
  @PostMapping("/recoverys/{targetInstanceActivityId}")
  @ApiOperation(value = "基于指定的实例活动编号，将对应的表单实例从最近的一次变化日志开始进行数据恢复，直到数据恢复到这个指定的实例活动编号对应的最后一次数据变化日志为止。")
  public ResponseModel recoveryByInstanceActivityId(@PathVariable("targetInstanceActivityId") String targetInstanceActivityId) {
    try {
      JSONObject jsonObject = this.instanceActivityLogService.recoveryByInstanceActivityId(targetInstanceActivityId);
      return this.buildHttpResultW(jsonObject);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @PostMapping("/validateRecoveryByInstanceActivityLogId/{instanceActivityLogId}")
  @ApiOperation(value = "按照指定的数据变更日志对表单数据进行恢复验证。" , notes = "注意，该方法肯定会抛出异常，如果抛出的异常时运行时异常（RuntimeException异常或者子级异常）那么说明整个恢复过程确实出现了问题")
  public ResponseModel validateRecoveryByInstanceActivityLogId(@PathVariable("instanceActivityLogId") String instanceActivityLogId) throws Exception {
    try {
      this.instanceActivityLogService.validateRecoveryByInstanceActivityLogId(instanceActivityLogId);
    } catch (IllegalStateException e) {
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
    
    return this.buildHttpResult();
  }
  
  @PostMapping("/validateRecoveryByInstanceActivityId/{targetInstanceActivityId}")
  @ApiOperation(value = "基于指定的实例活动编号，将对应的表单实例从最近的一次变化日志开始进行数据恢复。" , notes = "注意，该方法肯定会抛出异常，如果抛出的异常时运行时异常（RuntimeException异常或者子级异常）那么说明整个恢复过程确实出现了问题")
  public ResponseModel validateRecoveryByInstanceActivityId(@PathVariable("targetInstanceActivityId") String targetInstanceActivityId) throws Exception {
    try {
      this.instanceActivityLogService.validateRecoveryByInstanceActivityId(targetInstanceActivityId);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    } catch (Exception e) {
      return this.buildHttpResult();
    }
    
    return this.buildHttpResult();
  }
}