package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.service.invoke.handle.request.PrincipalHandle;
import com.bizunited.platform.core.service.invoke.handle.request.TransactionalHandle;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.service.DynamicInstanceService;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.RelationsTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.DynamicInstanceCustomRepository;
import com.bizunited.platform.kuiper.starter.repository.table.TableOperateRepositoryCustom;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import com.bizunited.platform.kuiper.starter.service.instances.handle.DynamicFormDetailsRequestHandle;
import com.bizunited.platform.kuiper.starter.service.instances.handle.DynamicFormDetailsResponseHandle;
import com.bizunited.platform.kuiper.starter.service.instances.handle.DynamicInstanceDetailsLogUpdateHandle;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 动态实例相关的处理方法
 */
@Service("DynamicInstanceServiceImpl")
public class DynamicInstanceServiceImpl implements DynamicInstanceService {
  private static final String DYNAMIC = "dynamic";
  private static final String MESS_FORMINSTANCE_ID = "formInstanceId";
  private static final String MAPPING = "%s_%s_mapping";
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicInstanceServiceImpl.class);

  @Autowired
  private InstanceService instanceService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private DynamicInstanceCustomRepository dynamicInstanceCustomRepository;
  @Autowired
  private TableOperateRepositoryCustom tableOperateRepositoryCustom;
  @Autowired
  private InstanceActivityService instanceActivityService;

  /**
   * 该方法获取实例基本信息，不对任何关联信息进行查询
   */
  @Override
  public JSONObject findByFormInstanceId(String formInstanceId) {
    //1. 先进行简单的验证并获取模板信息
    TemplateEntity template = this.validateInstanceAndTemplateInfo(formInstanceId);
    
    //2. 筛选需要的一般属性，只对一般属性进行查询
    Set<TemplatePropertyEntity> properties = template.getProperties();
    Validate.isTrue(!CollectionUtils.isEmpty(properties) , "未发现模板的一般属性信息，请检查!!");
    
    //3 拼接一般属性的查询SQL   调用执行SQL
    JSONArray jsonArr = this.dynamicInstanceCustomRepository.findProperties(properties , template.getTableName() , formInstanceId);
    
    //4 拼装返回的json
    return this.fetchOne(jsonArr);
  }

  /**
   * 根据实例ID信息，获取实例的详情，包含所有可能的关联信息
   */
  @Override
  public JSONObject findDetailsByFormInstanceId(String formInstanceId) {
    //1. 先进行简单的验证并获取模板信息
    TemplateEntity template = this.validateInstanceAndTemplateInfo(formInstanceId);
    
    //2.1 筛选需要的一般属性，只对一般属性进行查询
    Set<TemplatePropertyEntity> properties = template.getProperties();
    Validate.isTrue(!CollectionUtils.isEmpty(properties) , "未发现模板的一般属性信息，请检查!!");
    //2.2 拼接一般属性的查询SQL
    //调用执行SQL
    JSONArray jsonArr = this.dynamicInstanceCustomRepository.findProperties(properties , template.getTableName() , formInstanceId);
    
    if(!this.existJSON(jsonArr)) {
      return null;
    }
    JSONObject json = this.fetchOne(jsonArr);
    if(json == null) {
      return null;
    }
    String mainId = json.getString("id");
    Validate.notBlank(mainId, "实例数据ID值不能为空，请检查!!");
    
    //2.3 获取主表下可能的关联属性(ManyToMany  ManyToOne)
    Set<TemplateRelationEntity> mainRelations = template.getRelations();
    if(!CollectionUtils.isEmpty(mainRelations)) {
      this.processManyToMany(mainRelations , template.getTableName() , mainId , json);
      this.processManyToOne(mainRelations , template.getTableName() , mainId , json);
    }
    
    //2.4 获取主表下可能的分组关系
    Set<TemplateGroupEntity> oneToOnes =  template.getGroupRelations();
    if(!CollectionUtils.isEmpty(oneToOnes)) {
      this.processOneToOne(oneToOnes, mainId , json);
    }
    
    //2.5 获取主表下可能的明细关系
    Set<TemplateItemEntity> items = template.getItemRelations();
    if(!CollectionUtils.isEmpty(items)) {
      this.processOneToMany(items, mainId , json);
    }
    
    //2.6 返回最终数据
    return json;
  }

  @Override
  @NebulaServiceMethod(name = "dynamicInstanceService.save" , desc = "动态模板业务数据的保存，括动态模板中的一般属性、关联属性、明细编辑属性和分组属性" 
  , requestHandleTypes = {PrincipalHandle.class , TransactionalHandle.class , DynamicFormDetailsRequestHandle.class} 
  , responseHandleTypes = {DynamicFormDetailsResponseHandle.class , DynamicInstanceDetailsLogUpdateHandle.class})
  public void create(JSONObject jsonObject) {
    /*
     * 保存流程如下：
     * 1.首先验证传入数据信息的正确性
     * a.传入实例ID有效，并且有相对模板
     * b.验证必填项、可能的唯一约束的值、长度
     * 2.入库：拼接sql,存入数据
     * a.先存入无关联的一般属性与manyToOne关联属性
     *    备注：因为可能manyToOne属性设置为不能为空，所以需要同时insert
     * b.存入manyToMany关联属性
     * c.存入明细信息
     * d.存入分组属性
     *   i.存入分组属性的一般属性
     *   ii.存入分组属性的关联属性
     *   iii.存入分组属性的明细信息
     */
    // 1.======验证
    this.validDynamicTemplateData(jsonObject, true);
    // 新增时赋予id值
    String parentId = UUID.randomUUID().toString();
    jsonObject.put("id", parentId);
    //从实例ID中查询得到模板
    String instanceActivityId = jsonObject.getString("instanceActivityId");
    InstanceEntity instance = null;
    if(StringUtils.isBlank(instanceActivityId)){
      String formInstanceId = jsonObject.getString(MESS_FORMINSTANCE_ID);
      instance = instanceService.findDetailsById(formInstanceId);
    }else {
      InstanceActivityEntity instanceActivity = this.instanceActivityService.findDetailsById(instanceActivityId);
      instance = instanceActivity.getInstance();
    }

    TemplateEntity template = instance.getTemplate();
    // 2.======保存
    // a.===保存一般属性
    // 调用保存一般属性与manyToOne的方法
    this.saveTable(jsonObject, template.getProperties(),template.getRelations(), template.getTableName(), false, null, null);

    // b.===保存manyToMany属性
    this.saveMappingTable(template.getRelations(),jsonObject,template.getTableName());
    // c.===保存明细属性
    this.saveItems(template.getItemRelations(),jsonObject,parentId);
    // d.===保存分组属性
    if(!CollectionUtils.isEmpty(template.getGroupRelations())){
      Set<TemplateGroupEntity> groups = template.getGroupRelations().stream().filter(o -> null != jsonObject.get(o.getPropertyName())).collect(Collectors.toSet());
      //对每一个分组属性做遍历操作
      for(TemplateGroupEntity group : groups){
        JSONObject itemJson = jsonObject.getJSONObject(group.getPropertyName());
        if(itemJson!=null){
          String groupId = UUID.randomUUID().toString();
          itemJson.put("id",groupId);
          //保存一般属性数据
          this.saveTable(itemJson,group.getProperties(),group.getRelations(),group.getTableName(),true,group.getParentTableName(),parentId);
          //保存关联
          this.saveMappingTable(this.filterOneToOneRelation(group.getRelations(),group.getParentTableName()),itemJson,group.getTableName());
          //保存明细
          this.saveItems(group.getItemRelations(),itemJson,groupId);
        }
      }
    }
  }

  /**
   * 将oneToOne自动建立在分组中与主表的manyToOne关系过滤掉，方便数据操作。
   * @param relations
   * @param parentTable
   * @return
   */
  private Set<TemplateRelationEntity> filterOneToOneRelation(Set<TemplateRelationEntity> relations,String parentTable){
    if(!CollectionUtils.isEmpty(relations)){
      return relations.stream().filter(o -> (!o.getPropertyDbName().equals(String.format("%s_id",parentTable)))).collect(Collectors.toSet());
    }
    return new HashSet<>();
  }

  /**
   * 保存主业务一般属性与关联属性中的ManyToOne属性，存入数据库
   * 注意，该方法仅用于主业务属性
   * @param jsonObject 带有业务数据的JSON
   * @param properties 模板中保存的一般属性集合
   * @param relations 模板中保存的关联信息，用来保存manyToOne类型的数据
   * @param tableName 存入的表单数据库名
   * @param isItem 是否保存主表的明细项中的数据
   * @param isItem 如果保存的明细，本次保存的明细实例
   * @param parentId 如果保存的明细，传入父类ID
   */
  private void saveTable(JSONObject jsonObject, Set<TemplatePropertyEntity> properties,Set<TemplateRelationEntity> relations,
                         String tableName, Boolean isItem, String parentTableName, String parentId) {
    Map<String, String> kv = new HashMap<>();
    //添加一般属性数据
    if(!CollectionUtils.isEmpty(properties)){
      Map<String,String> propertyKV = new HashMap<>();
      for(TemplatePropertyEntity p : properties){
        String key = String.format("`%s`",p.getPropertyDbName());
        String res = String.valueOf(jsonObject.get(p.getPropertyName()));
        //如果是字符串，需要添加单引号
        if (StringUtils.equals(String.class.getName(), p.getPropertyClassName())) {
          res = res == null ? null : String.format("'%s'", res);
        } else if (StringUtils.equals("java.util.Date", p.getPropertyClassName())) {
          res = res == null ? null : String.format("'%s'", this.formatDate(res));
        }
        propertyKV.put(key,res);
      }
      //因为ID并在 properties 中，所以此处需要添加ID, ywt,2019.10.08
      kv.put(String.format("`%s`","id"),String.format("'%s'",jsonObject.getString("id")));
      kv.putAll(propertyKV);
    }
    //添加ManyToOne数据
    if((null!=relations)&&(!CollectionUtils.isEmpty(relations))){
      Set<TemplateRelationEntity> manyToOnes = relations.stream()
          .filter(o -> RelationsTypeEnum.MANY_TO_ONE.getRelation().equals(o.getRelationType()) && !o.getBackProperty()).collect(Collectors.toSet());
      if(!CollectionUtils.isEmpty(manyToOnes)){
        Map<String,String> map = new HashMap<>();
        for(TemplateRelationEntity p : manyToOnes){
          String key = String.format("`%s`",p.getPropertyDbName());
          JSONObject json = jsonObject.getJSONObject(p.getPropertyName());
          Validate.isTrue(json != null, "传入manyToOne对象不存在");
          String manyToOneId = json.getString("id");
          String value = manyToOneId == null ? null : String.format("'%s'",manyToOneId);
          map.put(key,value);
        }
        kv.putAll(map);
      }
    }
    // 如果是子项的话，需要加上之前添加的父项ID
    if (Boolean.TRUE.equals(isItem)) {
      String parentIdKey = String.format("`%s_id`",parentTableName);
      String parentIdValue = String.format("'%s'", parentId);
      kv.put(parentIdKey, parentIdValue);
    }
    StringBuilder vals = new StringBuilder();
    for(String v : kv.values()){
      if(v == null){
        vals.append("null,");
      }else if(StringUtils.isEmpty(v)){
        vals.append(",");
      }else{
        vals.append(v + ",");
      }
    }
    dynamicInstanceCustomRepository.executeNativeSQL("INSERT INTO `%s` (%s) VALUES (%s)", tableName,StringUtils.join(kv.keySet(), ","), StringUtils.left(vals.toString(),vals.length() - 1));
  }


  /**
   * 保存关联信息
   * 注意，可能是保存主业务表的关系信息，也可能是分组表的关联信息
   * @param relations
   * @param jsonObject
   * @param tableName
   */
  @SuppressWarnings("unchecked")
  private void saveMappingTable(Set<TemplateRelationEntity> relations,JSONObject jsonObject,String tableName){
    if(CollectionUtils.isEmpty(relations)){
      return;
    }
    // 找出manyToMany 类型的插入数据
    Set<TemplateRelationEntity> manyToManys = relations.stream()
        .filter(o -> RelationsTypeEnum.MANY_TO_MANY.getRelation().equals(o.getRelationType())).collect(Collectors.toSet());
    // 处理manyToMany关系
    if(CollectionUtils.isEmpty(manyToManys)){
      return;
    }
    for (TemplateRelationEntity relation : manyToManys) {
      if(Boolean.TRUE.equals(relation.getBackProperty())) {
        continue;
      }
      // 为两张表名排序，从而得到中间表表名
      ArrayList<String> tableNames = new ArrayList<>(Lists.newArrayList(tableName, relation.getTargetTableName()));
      List<String> sortedName = tableNames.stream().sorted().collect(Collectors.toList());
      String dbMappingTable = String.format(MAPPING, sortedName.get(0), sortedName.get(1));
      // 为插入的表命后拼接_id 为中间表字段名
      List<String> nameWithId = tableNames.stream().map(o -> "`" + o.concat("_id") + "`").collect(Collectors.toList());
      // 拼接插入sql
      String keys = StringUtils.join(nameWithId, ",");
      JSONArray targetIds = jsonObject.getJSONArray(relation.getPropertyName());
      // 可能同时绑定多个关系
      for (Object targetIdObject : targetIds) {
        String targetId = this.analysisTargetId((Map<String , Object>)targetIdObject);
        String values = String.format("'%s','%s'", jsonObject.getString("id"), targetId);
        dynamicInstanceCustomRepository.executeNativeSQL("INSERT INTO `%s` (%s) VALUES (%s)", dbMappingTable, keys, values);
      }
    }
  }
  
  private String analysisTargetId(Map<String , Object> targetIdMap) {
    Object targetIdValue = targetIdMap.get("id");
    Validate.notNull(targetIdValue , "错误的id编号信息!!");
    String targetId = targetIdValue.toString();
    Validate.notBlank(targetId , "错误的id编号信息!!");
    return targetId;
  }

  /**
   * 保存明细信息
   * 注意，可能是保存主业务表的明细信息，也可能是分组表的明细信息
   * @param items
   * @param jsonObject
   * @param parentId
   */
  private void saveItems(Set<TemplateItemEntity> items,JSONObject jsonObject,String parentId){
    if(CollectionUtils.isEmpty(items)){
      return;
    }
    // 得到所有在数据存在的明细信息
    Set<TemplateItemEntity> templateItems = items.stream()
        .filter(o -> null != jsonObject.get(o.getPropertyName())).collect(Collectors.toSet());
    // 对每一个明细项做遍历操作
    for (TemplateItemEntity item : templateItems) {
      // 得到明细项的字段编辑json
      JSONArray jsonArray = jsonObject.getJSONArray(item.getPropertyName());
      if(jsonArray == null || jsonArray.isEmpty()) {
        continue;
      }
      // 遍历保存每一条明细Object obj : jsonArray
      for (int i = 0 ; i < jsonArray.size() ; i++) {
        JSONObject itemJson = jsonArray.getJSONObject(i);
        // 为新增明细添加ID
        itemJson.put("id", UUID.randomUUID().toString());
        // 得到明细项在明细表中有编辑的一般属性信息
        Set<TemplatePropertyEntity> itemProperties = item.getProperties().stream()
            .filter(o -> null != itemJson.get(o.getPropertyName())).collect(Collectors.toSet());
        //得到该明细项在明细表中的关联属性信息
        Set<TemplateRelationEntity> itemRelations = item.getRelations().stream().filter(o ->
            !(Boolean.TRUE.equals(o.getNullable()) && null == itemJson.get(o.getPropertyName()))).collect(Collectors.toSet());
        Set<TemplateRelationEntity> itemManyToOnes ;
        itemManyToOnes = itemRelations.stream().filter(o -> RelationsTypeEnum.MANY_TO_ONE.getRelation().equals(o.getRelationType())).collect(Collectors.toSet());
        //保存明细中的一般属性与manyToOne
        this.saveTable(
            itemJson, itemProperties, itemManyToOnes,item.getTableName(),true, item.getParentTableName(), parentId);
        //保存明细中的manyToMany
        this.saveMappingTable(item.getRelations(),itemJson,item.getTableName());

      }
    }
  }

  @Override
  @NebulaServiceMethod(name = "dynamicInstanceService.update" , desc = "动态模板业务数据的更新，括动态模板中的一般属性、关联属性、明细编辑属性和分组属性" 
  , requestHandleTypes = {PrincipalHandle.class , TransactionalHandle.class , DynamicFormDetailsRequestHandle.class} 
  , responseHandleTypes = {DynamicFormDetailsResponseHandle.class , DynamicInstanceDetailsLogUpdateHandle.class})
  public void update(JSONObject jsonObject) {
    /*
     * 更新流程如下：
     * 1.首先验证传入数据信息的正确性
     * a.传入实例ID有效，并且有相对模板
     * b.验证必填项、可能的唯一约束的值、长度
     * 2.更新入库：拼接sql,存入数据
     * a.先更新无关联的一般属性
     * b.更新关联属性
     * c.更新明细信息
     * d.更新分组属性
     *   i.更新分组属性的一般属性
     *   ii.更新分组属性的关联属性
     *   iii.更新分组属性的明细信息
     */
    // 1.======
    this.validDynamicTemplateData(jsonObject, false);
    String formInstanceId = jsonObject.getString(MESS_FORMINSTANCE_ID);
    InstanceEntity instance = instanceService.findDetailsById(formInstanceId);
    TemplateEntity template = instance.getTemplate();
    Set<TemplatePropertyEntity> properties = template.getProperties();
    // 2.======
    // a.===
    // 调用保存一般属性的方法，因为在明细项保存一般属性中会调用相同方法，所以将此方法抽出。
    this.updateProperties(jsonObject, properties, template.getTableName());
    // b.===
    this.updateRelations(jsonObject,template.getRelations(),template.getTableName());
    // c.===
    this.updateItems(jsonObject,template.getItemRelations());
    // d.===
    if(!CollectionUtils.isEmpty(template.getGroupRelations())){
      Set<TemplateGroupEntity> groups = template.getGroupRelations().stream().filter(o -> null != jsonObject.get(o.getPropertyName())).collect(Collectors.toSet());
      for(TemplateGroupEntity group : groups){
        JSONObject itemJson = jsonObject.getJSONObject(group.getPropertyName());
        if(itemJson!=null){
          //保存一般属性数据
          this.updateProperties(itemJson,group.getProperties(),group.getTableName());
          //保存关联
          this.updateRelations(itemJson,this.filterOneToOneRelation(group.getRelations(),group.getParentTableName()),group.getTableName());
          //保存明细
          this.updateItems(itemJson,group.getItemRelations());
        }
      }
    }

  }

  /**
   * 更新主业务表业务数据
   * 注意，该方法不仅仅用于主业务属性，也用于明细属性，与分组属性
   * @param jsonObject 传入业务数据的JSON
   * @param properties 模板中保存的一般属性集合
   * @param tableName 存入的表单数据库名
   */
  private void updateProperties(JSONObject jsonObject, Set<TemplatePropertyEntity> properties, String tableName) {
    if(CollectionUtils.isEmpty(properties)){
      return;
    }

    List<String> kv = properties.stream().filter(o -> null != jsonObject.get(o.getPropertyName()) &&
        !StringUtils.equalsIgnoreCase("id", o.getPropertyName()) && !StringUtils.equalsIgnoreCase(MESS_FORMINSTANCE_ID, o.getPropertyName())).map(
        o -> {
          String value = String.valueOf(jsonObject.get(o.getPropertyName()));
          String valueStr = value;
          if (StringUtils.equals(String.class.getName(), o.getPropertyClassName())) {
            valueStr = String.format("'%s'", value);
          } else if ((StringUtils.equals("java.util.Date", o.getPropertyClassName()))) {
            valueStr = String.format("'%s'", this.formatDate(value));
          }
          return String.format("`%s`=%s", o.getPropertyDbName(), valueStr);
        }).collect(Collectors.toList());

    if(CollectionUtils.isEmpty(kv)) {
      return;
    }
    dynamicInstanceCustomRepository.executeNativeSQL("UPDATE `%s` SET %s WHERE `id` = '%s'",tableName, StringUtils.join(kv, ","), jsonObject.getString("id"));
  }

  /**
   * 更新关联信息
   * 注意，可能是更新主业务表的关系信息，也可能是分组属性的关联信息
   * @param jsonObject
   * @param relations
   * @param tableName
   */
  @SuppressWarnings("unchecked")
  private void updateRelations(JSONObject jsonObject,Set<TemplateRelationEntity> relations,String tableName){
    if(CollectionUtils.isEmpty(relations)){
      return;
    }
    // 分别找出 manyToOne 与 manyToMany 两种类型的插入数据
    Set<TemplateRelationEntity> manyToOnes = relations.stream()
        .filter(o -> RelationsTypeEnum.MANY_TO_ONE.getRelation().equals(o.getRelationType()) && null != jsonObject.get(o.getPropertyName()) ).collect(Collectors.toSet());
    Set<TemplateRelationEntity> manyToManys = relations.stream()
        .filter(o -> RelationsTypeEnum.MANY_TO_MANY.getRelation().equals(o.getRelationType()) && null != jsonObject.get(o.getPropertyName()) ).collect(Collectors.toSet());
    // 更新manyToOne的关系
    // manyToOne关系的，逻辑与新增相同，UPDATE即可
    if(!CollectionUtils.isEmpty(manyToOnes)){
      List<String> kvs = new LinkedList<>();
      for(TemplateRelationEntity p : manyToOnes){
        JSONObject json = jsonObject.getJSONObject(p.getPropertyName());
        Validate.isTrue(json != null, "传入manyToOne对象不存在");
        String manyToOneId = json.getString("id");
        if(manyToOneId == null){
          kvs.add(String.format("`%s`=", p.getPropertyDbName()) + "null");
        }else if(StringUtils.isEmpty(manyToOneId)){
          kvs.add(String.format("`%s`=''", p.getPropertyDbName()));
        }else{
          kvs.add(String.format("`%s`='%s'", p.getPropertyDbName(), manyToOneId));
        }
      }
      // 拼接更新关联语句
      dynamicInstanceCustomRepository.executeNativeSQL("UPDATE `%s` SET %s WHERE `id` = '%s'",tableName, StringUtils.join(kvs, ","), jsonObject.getString("id"));
    }
    // 更新manyToMany关系
    // 先移除所有当前实例绑定的多对多关系，再重复绑定上
    if(!CollectionUtils.isEmpty(manyToManys)){
      for (TemplateRelationEntity relation : manyToManys) {
        // 为两张表名排序，从而得到中间表表名
        ArrayList<String> tableNames = new ArrayList<>(Lists.newArrayList(tableName, relation.getTargetTableName()));
        List<String> sortedName = tableNames.stream().sorted().collect(Collectors.toList());
        String dbMappingTable = String.format(MAPPING, sortedName.get(0), sortedName.get(1));
        //删除之前绑定关系
        this.deleteManyToMany(tableName,relation.getTargetTableName(),jsonObject.getString("id"));
        //重新插入新的绑定关系
        // 为插入的表命后拼接_id 为中间表字段名
        List<String> nameWithId = tableNames.stream().map(o -> String.format("`%s_id`",o)).collect(Collectors.toList());
        String keys = StringUtils.join(nameWithId, ",");
        JSONArray targetIds = jsonObject.getJSONArray(relation.getPropertyName());
        // 可能同时绑定多个关系
        for (Object targetIdObject : targetIds) {
          String targetId = this.analysisTargetId((Map<String , Object>)targetIdObject);
          String values = String.format("'%s','%s'", jsonObject.getString("id"), targetId);
          dynamicInstanceCustomRepository.executeNativeSQL("INSERT INTO `%s` (%s) VALUES (%s)", dbMappingTable, keys, values);
        }
      }
    }
  }

  /**
   * 更新明细信息
   * 注意，可能是更新主业务表的明细信息，也可能是分组表的明细信息
   * @param jsonObject
   * @param items
   */
  private void updateItems(JSONObject jsonObject,Set<TemplateItemEntity> items){
    if(CollectionUtils.isEmpty(items)) {
      return;
    }
    /*
     * 更新明细逻辑：
     * 查询原来有的明细信息，得到已存在集合A，对比新传入的数据集合B
     * B中没有ID项的--新增
     * B中有ID，且能在A中找到对应的ID，更新
     * B中没有，A中有的，删除
     */
    // 得到所有在数据存在的明细信息
    Set<TemplateItemEntity> templateItems = items.stream()
        .filter(o -> null != jsonObject.get(o.getPropertyName())).collect(Collectors.toSet());
    // 对每一个明细项做遍历操作
    for (TemplateItemEntity item : templateItems) {
      //查询现有明细集
      List<?> existIds = dynamicInstanceCustomRepository.executeQuerySql("SELECT id FROM %s WHERE %s_id = '%s'",item.getTableName(),item.getParentTableName(),jsonObject.get("id"));
      ArrayList<?> unProcessedIds = new ArrayList<>(existIds);
      JSONArray jsonArray = jsonObject.getJSONArray(item.getPropertyName());
      //找出明细中每一行的数据
      List<JSONObject> itemJsonObjects = jsonArray.stream().map(o -> JSON.parseObject(JSON.toJSONString(o))).collect(Collectors.toList());
      for(JSONObject itemJson : itemJsonObjects){
        //查询出明细中待修改的一般属性
        Set<TemplatePropertyEntity> itemProperties = item.getProperties().stream().filter(o -> null != itemJson.get(o.getPropertyName())).collect(Collectors.toSet());
        //传入没有ID的明细项，新增
        if(StringUtils.isBlank(itemJson.getString("id"))) {
          // 新增明细项在一般属性与manyToOne关联属性
          itemJson.put("id",UUID.randomUUID().toString());
          this.saveTable(itemJson,itemProperties,item.getRelations(),item.getTableName(),true,item.getParentTableName(),jsonObject.getString("id"));
          //新增明细项中的manyToMany属性
          this.saveMappingTable(item.getRelations(),itemJson,item.getTableName());
        }
        //传入明细项有ID，该ID也存在于数据库，修改
        if(!StringUtils.isBlank(itemJson.getString("id"))){
          String itemId = itemJson.getString("id");
          if(existIds.contains(itemId)){
            //更新明细项中的一般属性
            this.updateProperties(itemJson,itemProperties,item.getTableName());
            //更新明细项中的关联属性
            this.updateRelations(itemJson,item.getRelations(),item.getTableName());
            //从待删除列表中移除该行
            unProcessedIds.remove(itemJson.getString("id"));
          }
        }
      }
      //移除数据库中仍存在的明细项,注意：如果该明细有多对多关联，则需要先移除中间表中数据
      if(!CollectionUtils.isEmpty(unProcessedIds)){
        for(Object id : unProcessedIds){
          Set<TemplateRelationEntity> manyToManys = item.getRelations().stream()
              .filter(o -> RelationsTypeEnum.MANY_TO_MANY.getRelation().equals(o.getRelationType())).collect(Collectors.toSet());
          for(TemplateRelationEntity relation : manyToManys){
            this.deleteManyToMany(item.getTableName(),relation.getTargetTableName(),id.toString());
          }
          //移除manyToMany中间表后，才可以移除明细表
          dynamicInstanceCustomRepository.executeNativeSQL("DELETE FROM `%s` WHERE `id` = '%s'",item.getTableName(),id);
        }
      }
    }
  }


  /**
   * 移除中间表中数据
   * @param tableName
   * @param targetTableName
   * @param itemId
   */
  private void deleteManyToMany(String tableName,String targetTableName,String itemId){
    Validate.notBlank(tableName,"移除中间表中，数据表名不能为空！");
    Validate.notBlank(targetTableName,"移除中间表中，数据表名不能为空！");
    Validate.notBlank(itemId,"移除中间表中，明细ID不能为空！");
    ArrayList<String> tableNames = new ArrayList<>(Lists.newArrayList(tableName, targetTableName));
    List<String> sortedName = tableNames.stream().sorted().collect(Collectors.toList());
    String dbMappingTable = String.format(MAPPING, sortedName.get(0), sortedName.get(1));
    dynamicInstanceCustomRepository.executeNativeSQL("DELETE FROM `%s` WHERE `%s_id` = '%s'",dbMappingTable,tableName,itemId);
  }

  /**
   * 检查新建/更新参数
   *
   * @param jsonObject 数据
   * @param isCreate 是否为新建，true新建，false更新
   */
  private void validDynamicTemplateData(JSONObject jsonObject, Boolean isCreate) {
    /*
     * 保存流程如下：
     * 1.首先验证传入数据信息的正确性
     * a.传入实例ID有效，并且有相对模板
     * b.验证必填项、可能的唯一约束的值、长度
     */
    // 1.======
    Validate.notNull(jsonObject, "前端传入业务数据为空，请检查！");
    // a.===
    String id = jsonObject.getString("id");
    if (Boolean.TRUE.equals(isCreate)) {
      Validate.isTrue(StringUtils.isBlank(id), "新增时，不能传入ID！");
    } else {
      Validate.notBlank(id, "更新时候，传入模板ID不能为空！");
    }
    String instanceActivityId = jsonObject.getString("instanceActivityId");
    InstanceEntity instance = null;
    if(StringUtils.isBlank(instanceActivityId)){
      String formInstanceId = jsonObject.getString(MESS_FORMINSTANCE_ID);
      instance = instanceService.findDetailsById(formInstanceId);
    }else {
      InstanceActivityEntity instanceActivity = this.instanceActivityService.findDetailsById(instanceActivityId);
      instance = instanceActivity.getInstance();
    }
    Validate.notNull(instance, "未找到该模板实例，请检查！");
    TemplateEntity template = instance.getTemplate();
    Validate.notNull(template, "未找到该实例绑定模板，请检查！");
    // b.===
    //验证主表单中的一般属性
    this.checkProperties(template.getProperties(),jsonObject,isCreate);
    //验证关联属性
    this.checkRelations(template.getRelations(),jsonObject,isCreate);
    //验证明细属性
    this.checkItems(template.getItemRelations(),jsonObject,isCreate);
    //验证分组属性
    Set<TemplateGroupEntity> groups = template.getGroupRelations();
    if(!CollectionUtils.isEmpty(groups)){
      for(TemplateGroupEntity group : groups){
        JSONObject groupJSON = jsonObject.getJSONObject(group.getPropertyName());
        Validate.notNull(group,String.format("未写入分组%s信息，请检查！",group.getPropertyName()));
        if(groupJSON!=null){
          //验证分组属性中的一般属性
          this.checkProperties(group.getProperties(),groupJSON,isCreate);
          //验证分组属性中的关联属性，注意：需要先剔除自动生成的与主表manyToOne关联
          this.checkRelations(this.filterOneToOneRelation(group.getRelations(),group.getParentTableName()),groupJSON,isCreate);
          //验证分组属性中的明细属性
          this.checkItems(group.getItemRelations(),groupJSON,isCreate);
        }

      }
    }

  }

  /**
   * 验证一般属性
   * @param properties
   * @param jsonObject
   */
  private void checkProperties(Set<TemplatePropertyEntity> properties,JSONObject jsonObject,Boolean isCreate){
    if(CollectionUtils.isEmpty(properties)){
      return;
    }
    for (TemplatePropertyEntity o : properties) {
      // 检查是否必填
      if (!Boolean.TRUE.equals(o.getNullable())) {
        Boolean isNull = null==jsonObject.get(o.getPropertyName()) || StringUtils.isBlank(o.getPropertyName());
        Validate.notNull(isNull, "业务数据中必填项" + o.getPropertyName() + "为空，请检查！");
      }
      // 检查长度,条件：设定做大值大于0，且有传入数据,且只对字符串做检查
      if (null != o.getMaxLen() && o.getMaxLen() > 0
          && null != jsonObject.getString(o.getPropertyName())
          && String.class.getName().equals(o.getPropertyClassName())) {
        Validate.isTrue(
            o.getMaxLen() > jsonObject.getString(o.getPropertyName()).length(),
            "数据中" + o.getPropertyName() + "超出限定长度" + o.getMaxLen() + "，请检查！");
      }
      // 检查唯一属性，查出数据库中该字段，并进行验证
      boolean b = o.getUnique() && null != jsonObject.get(o.getPropertyName()) && (isCreate || (!isCreate && !MESS_FORMINSTANCE_ID.equals(o.getPropertyName())));
      if (b) {
        // 因为该一般属性父级可能为模板，或分组，或明细，所以需要得到确切的父级表名称
        String tabelName = "";
        if(o.getCurrentTemplate()!=null){
          tabelName = o.getCurrentTemplate().getTableName();
        }else if(o.getCurrentGroup()!=null){
          tabelName = o.getCurrentGroup().getTableName();
        }else if(o.getCurrentItem()!=null){
          tabelName = o.getCurrentItem().getTableName();
        }
        // 拼接查询sql
        List<?> rows = dynamicInstanceCustomRepository.executeQuerySql("select %s from `%s` where %s = '%s'",o.getPropertyDbName(),tabelName,o.getPropertyDbName(),jsonObject.get(o.getPropertyName()));
        if(Boolean.TRUE.equals(isCreate)){
          Validate.isTrue(!rows.contains(jsonObject.get(o.getPropertyName())), "业务数据项" + o.getPropertyDesc() + "重复，请检查！");
        }
      }
    }
  }

  /**
   * 检查关联属性
   * @param relations
   * @param jsonObject
   * @param isCreate
   */
  @SuppressWarnings("unchecked")
  private void checkRelations(Set<TemplateRelationEntity> relations,JSONObject jsonObject,Boolean isCreate){
    if(CollectionUtils.isEmpty(relations)){
      return;
    }
    //验证其必填性,对于必填项是否有数据
    relations.stream().filter(o -> !o.getNullable()).forEach(
        p -> Validate.isTrue(null != jsonObject.get(p.getPropertyName())
            ,String.format("关联关系中必填项%s没有数据，请检查！",p.getPropertyName())));
    //验证其在目标数据表的存在性
    relations.stream().forEach(
        p -> Validate.isTrue(tableOperateRepositoryCustom.existTableName(p.getTargetTableName())
            ,String.format("关联表%s不存在，请检查！",p.getTargetTableName())));

    //验证manyToMany关系中，中间表是否存在
    relations.stream().filter(o -> RelationsTypeEnum.MANY_TO_MANY.getRelation().equals(o.getRelationType())).forEach(p -> {
      //获取中间表名
      String tableName = null;
      if(p.getCurrentTemplate() != null){
        tableName = p.getCurrentTemplate().getTableName();
      } else if(p.getCurrentItem() != null){
        tableName = p.getCurrentItem().getTableName();
      } else if(p.getCurrentGroup() != null){
        tableName = p.getCurrentGroup().getTableName();
      }
      ArrayList<String> tableNames = new ArrayList<>(Lists.newArrayList(
              tableName, p.getTargetTableName()));
      List<String> sortedName = tableNames.stream().sorted().collect(Collectors.toList());
      String dbMappingTable = String.format(MAPPING, sortedName.get(0), sortedName.get(1));
      Validate.isTrue(tableOperateRepositoryCustom.existTableName(dbMappingTable)
          ,String.format("关联表%s不存在，请检查！",p.getTargetTableName()));
    });

    //验证其目标数据项ID是否在目标数据表中存在
    relations.stream().filter(o -> null != jsonObject.get(o.getPropertyName())).forEach(p -> {
      if(p.getRelationType().equals("ManyToOne")) {
        JSONObject json = jsonObject.getJSONObject(p.getPropertyName());
        Object value = json.getString("id");
        Validate.notNull(value , "未发现指定数据的id项目");
        Validate.isTrue(this.checkRelationDataExist(p.getTargetTableName(),value.toString()),"传入manyToOne对象ID不存在");
      } else if (p.getRelationType().equals(RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
        JSONArray manyToManyIds = jsonObject.getJSONArray(p.getPropertyName());
        manyToManyIds.stream().forEach(o -> {
          Map<String , Object> kvs = (Map<String , Object>)o;
          Object value = kvs.get("id");
          Validate.notNull(value , "未发现指定数据的id项目");
          Validate.isTrue(this.checkRelationDataExist(p.getTargetTableName() , value.toString()),"传入manyToMany对象ID不存在");
        });
      }
    });
  }

  /**
   * 验证其目标数据项ID是否在目标数据表中存在
   * @param targetTableName 目标数据表名
   * @param kv
   * @return
   */
  private Boolean checkRelationDataExist(String targetTableName,String id) {
    Validate.notBlank(targetTableName,"目标数据表不能为空，请检查！");
    Validate.notBlank(id,"目标数据表数据ID不能为空，请检查！");
    List<?> result = dynamicInstanceCustomRepository.executeQuerySql("SELECT id FROM `%s` where id = '%s'",targetTableName,id);
    return !CollectionUtils.isEmpty(result);
  }

  /**
   * 检查明细属性
   * @param items
   * @param jsonObject
   * @param isCreate
   */
  private void checkItems(Set<TemplateItemEntity> items, JSONObject jsonObject,Boolean isCreate){
    if(CollectionUtils.isEmpty(items)){
      return;
    }
    for(TemplateItemEntity item : items){
      if(null!=jsonObject.get(item.getPropertyName())){
        JSONArray itemJsonArray = jsonObject.getJSONArray(item.getPropertyName());
        if(itemJsonArray == null || itemJsonArray.isEmpty()) {
          continue;
        }

        List<JSONObject> itemJsonObjects = new ArrayList<>();
        for(int i = 0 ; i < itemJsonArray.size() ; i++){
          itemJsonObjects.add(itemJsonArray.getJSONObject(i));
        }


        //List<JSONObject> itemJsonObjects = itemJsonArray.stream().map(o -> JSONObject.parseObject(o.toString())).collect(Collectors.toList());
        if(!CollectionUtils.isEmpty(itemJsonObjects)){
          for(JSONObject itemjson : itemJsonObjects){
            //检查明细项中的一般属性
            this.checkProperties(item.getProperties(),itemjson,isCreate);
            //检查明细中的关联属性 注意：先过滤oneToOne造成的反向关联
            this.checkRelations(filterOneToOneRelation(item.getRelations(),item.getParentTableName()),itemjson,isCreate);
          }
        }
      }
    }
  }



  /**
   * 根据实例ID信息获取相应的模板信息，并验证模板信息是否符合动态模板的信息
   * @param formInstanceId
   */
  private TemplateEntity validateInstanceAndTemplateInfo(String formInstanceId) {
    InstanceEntity instanceEntity = this.instanceService.findDetailsById(formInstanceId);
    Validate.notNull(instanceEntity, "根据传入的实例ID未能获取到实例信息，请检查!!");
    Validate.isTrue(instanceEntity.getTemplate() != null, "在实例信息中，未能获取到对应的模板信息，请检查!!");
    Validate.notBlank(instanceEntity.getTemplate().getId(), "在实例信息中，模板ID不能为空，请检查!!");
    TemplateEntity template = this.templateService.findDetailsById(instanceEntity.getTemplate().getId());
    Validate.notBlank( template.getId(), "根据传入的实例ID信息，未能获取到对应的模板ID信息，请检查!!");
    Validate.isTrue( StringUtils.equals(template.getType(), DYNAMIC), "该模板不是动态模板信息，请检查!!");
    Validate.notBlank(template.getTableName(), "未发现该动态模板的设置的数据库表名，请检查!!");
    Validate.isTrue(template.getRepositoryEntity() , "该动态表单模板没有数据层信息进行支撑，请检查!!");
    Validate.isTrue(template.getTstatus() == 1 , "该动态表单模板当前状态为禁用，请检查!!");
    return template;
  }



  /**
   * ManyToMany的查询逻辑   筛选数据、拼接SQL、执行SQL、组装返回数据
   * @param relations 筛选并过滤回溯的ManyToMany
   * @param tableName 指ManyToMany的上层表名
   * @param id 指ManyToMany的上层表的id
   * @param json 指ManyToMany的上层表的json数据
   */
  private void processManyToMany(Set<TemplateRelationEntity> relations , String tableName , String id , JSONObject json) {
    //筛选出ManyToMany，并过滤回嗍属性
    Set<TemplateRelationEntity> manyToManys = relations.stream().filter(e -> this.isManyToMany(e.getRelationType()) && !e.getBackProperty()).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(manyToManys)) {
      return;
    }
    Map<String,Object> map = new HashMap<>();
    for(TemplateRelationEntity e : manyToManys) {
      JSONArray jsonArr = this.dynamicInstanceCustomRepository.findManyToMany(tableName , e.getTargetTableName() , id);
      map.put(e.getPropertyName(), jsonArr);
    }
    this.buildJSON(map,json);
  }


  /**
   * ManyToOne的查询逻辑   筛选数据、拼接SQL、执行SQL、组装返回数据
   * @param relations 筛选并过滤回溯的ManyToOne
   * @param tableName 指ManyToOne的上层表名
   * @param id 指ManyToOne的上层表的id
   * @param json 指ManyToOne的上层表的json数据
   */
  private void processManyToOne(Set<TemplateRelationEntity> relations , String tableName , String id , JSONObject json){
    //筛选ManyToOne数据，并过滤回嗍
    Set<TemplateRelationEntity> manyToOnes = relations.stream().filter(e -> this.isManyToOne(e.getRelationType()) && !e.getBackProperty()).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(manyToOnes)) {
      return;
    }
    Map<String, Object> map = new HashMap<>();
    for(TemplateRelationEntity e : manyToOnes) {
      JSONArray jsonArr = this.dynamicInstanceCustomRepository.findManyToOne(tableName, e.getTargetTableName(), e.getPropertyDbName(), id);
      map.put(e.getPropertyName(), this.fetchOne(jsonArr));
    }
    this.buildJSON(map,json);
  }



  /**
   * 处理分组的查询逻辑
   * @param groups
   * @param id
   * @param resultJson
   */
  private void processOneToOne(Set<TemplateGroupEntity> groups , String id , JSONObject resultJson){
    Map<String, Object> otoMap = new HashMap<>();
    for(TemplateGroupEntity group : groups) {
      //获取OneToOne下的一般属性
      Set<TemplatePropertyEntity> props = group.getProperties();
      Validate.notNull(props, "%s下的一般属性不能为空，请检查!!",group.getPropertyDesc());
      Validate.notBlank(group.getTableName(), "%s下的表名不能为空，请检查!!",group.getPropertyDesc());
      Validate.notBlank(group.getParentTableName(), "%s下的上层表名不能为空，请检查!!",group.getPropertyDesc());

      JSONArray jsonArr = this.dynamicInstanceCustomRepository.findProperties(props,group.getTableName(),group.getParentTableName(),id,group.getPropertyDesc());

      JSONObject json = this.fetchOne(jsonArr);
      if(json == null) {
        continue;
      }
      String groupId = json.getString("id");

      //获取OneToOne下的关联信息
      Set<TemplateRelationEntity> otoRelations = group.getRelations();
      if(!CollectionUtils.isEmpty(otoRelations)) {
        this.processManyToMany(otoRelations , group.getTableName() , groupId , json);
        this.processManyToOne(otoRelations , group.getTableName() , groupId , json);
      }

      //获取OneToOne下的OneToMany关系
      Set<TemplateItemEntity> otmRelations = group.getItemRelations();
      if(!CollectionUtils.isEmpty(otmRelations)) {
        this.processOneToMany(otmRelations ,  groupId , json);
      }
      otoMap.put(group.getPropertyName(), json);
    }
    this.buildJSON(otoMap, resultJson);
  }


  /**
   * 处理明细的查询逻辑
   * @param oneToManys
   * @param id
   * @param json
   */
  private void processOneToMany(Set<TemplateItemEntity> oneToManys , String id , JSONObject json){
    Map<String, Object> otmMap = new HashMap<>();
    for(TemplateItemEntity item : oneToManys) {
      //获取OneToMany下的一般属性
      Set<TemplatePropertyEntity> itemProps = item.getProperties();
      Validate.notNull(itemProps, "%s下的一般属性不能为空，请检查!!",item.getPropertyDesc());
      Validate.notBlank(item.getTableName(), "%s下的表名不能为空，请检查!!",item.getPropertyDesc());
      Validate.notBlank(item.getParentTableName(), "%s下的上层表名不能为空，请检查!!",item.getPropertyDesc());

      JSONArray jsonArr = this.dynamicInstanceCustomRepository.findProperties(itemProps,item.getTableName(),item.getParentTableName(),id,item.getPropertyDesc());

      //获取OneToMany下的关联关系
      Set<TemplateRelationEntity> itemRelations = item.getRelations();
      if(!CollectionUtils.isEmpty(itemRelations) && this.existJSON(jsonArr)) {
        for(int i = 0 ; i < jsonArr.size() ; i++) {
          JSONObject itemJson = jsonArr.getJSONObject(i);
          String itemId = itemJson.getString("id");
          this.processManyToMany(itemRelations, item.getTableName(),itemId,itemJson);
          this.processManyToOne(itemRelations, item.getTableName(),itemId,itemJson);
        }
      }
      otmMap.put(item.getPropertyName(), jsonArr);
    }
    this.buildJSON(otmMap, json);
  }




  /**
   * 判断传入的JSONArray是否有值
   * @param jsonArr
   * @return
   */
  private boolean existJSON(JSONArray jsonArr) {
    return jsonArr != null && !jsonArr.isEmpty();
  }

  /**
   * 从给定的JSONArray中，提取一条数据，如果没有数据则返回null
   * 注意：使用该方法时，需要与existJSON私有方法配合使用
   * @param jsonArr
   * @return JSONObject
   */
  private JSONObject fetchOne(JSONArray jsonArr) {
    return this.existJSON(jsonArr) ? jsonArr.getJSONObject(0) : null;
  }


  /**
   * 根据传入的结果数据，拼装json
   * @param map
   * @param json
   */
  private void buildJSON(Map<String, Object> map , JSONObject json) {
    if(!CollectionUtils.isEmpty(map)) {
      map.forEach( (key , value) ->
        json.put(key, value)
      );
    }
  }


  /**
   * 判断关联关系是否MANY_TO_MANY
   * @param relationType
   * @return
   */
  private boolean isManyToMany(String relationType) {
    return StringUtils.equalsAnyIgnoreCase(relationType, RelationsTypeEnum.MANY_TO_MANY.getRelation());
  }
  /**
   * 判断关联关系是否MANY_TO_ONE
   * @param relationType
   * @return
   */
  private boolean isManyToOne(String relationType) {
    return StringUtils.equalsAnyIgnoreCase(relationType, RelationsTypeEnum.MANY_TO_ONE.getRelation());
  }

  private String formatDate(String value) {
    try {
      return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(Long.parseLong(value));
    } catch (IllegalArgumentException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("时间类型参数输入非法, 请检查 input value: [%s]", value));
    }
  }
}
