package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplateMaintainerEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.entity.TemplateVisibilityEntity;
import com.bizunited.platform.kuiper.service.DynamicTemplateService;
import com.bizunited.platform.kuiper.service.InstanceService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum;
import com.bizunited.platform.kuiper.starter.common.enums.RelationsTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.TemplatePropertyRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRelationRepository;
import com.bizunited.platform.kuiper.starter.repository.TemplateRepository;
import com.bizunited.platform.kuiper.starter.repository.table.TableOperateRepositoryCustom;
import com.bizunited.platform.kuiper.starter.service.DynamicTemplateDraftService;
import com.bizunited.platform.kuiper.starter.service.TemplateGroupService;
import com.bizunited.platform.kuiper.starter.service.TemplateItemService;
import com.bizunited.platform.kuiper.starter.service.TemplateMaintainerService;
import com.bizunited.platform.kuiper.starter.service.TemplateVisibilityService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

@Service("DynamicTemplateServiceImpl")
public class DynamicTemplateServiceImpl extends AbstractTemplateService implements DynamicTemplateService {
  /**
   * 目前动态数据表只支持的数据类型
   */
  private static Map<String, String> classTypeMapping = new HashMap<>();
  private static final String MAPPING = "%s_%s_mapping";
  private static final String DOUBLE = "double";
  private static final String FORMINSTANCE_ID = "formInstanceId";

  static {
    classTypeMapping.put(String.class.getName(), "varchar");
    classTypeMapping.put(Integer.class.getName(), "int");
    classTypeMapping.put(Long.class.getName(), "bigint");
    classTypeMapping.put(Float.class.getName(), "float");
    classTypeMapping.put(BigDecimal.class.getName(), DOUBLE);
    classTypeMapping.put(Double.class.getName(), DOUBLE);
    classTypeMapping.put(Boolean.class.getName(), "bit");
    classTypeMapping.put(Date.class.getName(), "timestamp");
  }
  /**
   * 保留字段名，这些字段名不能出现在动态模型中。
   * 要进行验证报错
   */
  private static final Set<String> RETAINFIELDNAMES = Sets.newHashSet("add" , "all" , "alter" , "analyze" , "and" , "as" ,
      "asc" , "asensitive" , "before" , "between" , "bigint" , "binary" , "blob" , "both" , "by" , "call" , "cascade" , "case" , "create" , 
      "change" , "char" , "character" , "check" , "collate" , "column" , "condition" , "connection" , "constraint" , "continue" , "convert" , 
      "cross" , "current_date" , "current_time" , "current_timestamp" , "current_user" , "cursor" , "database" , "databases" , "day_hour" , 
      "day_microsecond" , "day_minute" , "day_second" , "dec" , "decimal" , "declare" , "default" , "delayed" , "delete" , "desc" , 
      "describe" , "deterministic" , "distinct" , "distinctrow" , "div" , DOUBLE , "drop" , "dual" , "each" , "else" , "elseif" , "enclosed" ,
      "escaped" , "exists" , "exit" , "explain" , "false" , "fetch" , "float" , "float4" , "float8" , "for" , "force" , "foreign" , "from" , 
      "fulltext" , "goto" , "grant" , "group" , "having" , "high_priority" , "hour_microsecond" , "hour_minute" , "hour_second" , "if" , 
      "ignore" , "in" , "index" , "infile" , "inner" , "inout" , "insensitive" , "insert" , "int" , "int1" , "int2" , "int3" , "int4" , "int8" , 
      "integer" , "interval" , "into" , "is" , "iterate" , "join" , "key" , "keys" , "kill" , "label" , "leading" , "leave" , "left" , "like" , 
      "limit" , "linear" , "lines" , "load" , "localtime" , "localtimestamp" , "lock" , "long" , "longblob" , "longtext" , "loop" , "low_priority" , 
      "match" , "mediumblob" , "mediumint" , "mediumtext" , "middleint" , "minute_microsecond" , "minute_second" , "mod" , "modifies" , 
      "natural" , "not" , "no_write_to_binlog" , "null" , "numeric" , "on" , "optimize" , "option" , "optionally" , "or" , "order" , "out" , 
      "outer" , "outfile" , "precision" , "primary" , "procedure" , "purge" , "raid0" , "range" , "read" , "reads" , "real" , "references" , 
      "regexp" , "release" , "rename" , "repeat" , "replace" , "require" , "restrict" , "return" , "revoke" , "right" , "rlike" , "schema" , 
      "second_microsecond" , "select" , "sensitive" , "separator" , "set" , "show" , "smallint" , "spatial" , "specific" , "sql" , 
      "sqlexception" , "sqlstate" , "sqlwarning" , "sql_big_result" , "sql_calc_found_rows" , "sql_small_result" , "ssl" , "starting" , 
      "straight_join" , "table" , "terminated" , "then" , "tinyblob" , "tinyint" , "tinytext" , "to" , "trailing" , "trigger" , "true" , 
      "undo" , "union" , "unique" , "unlock" , "unsigned" , "update" , "usage" , "use" , "using" , "utc_date" , "utc_time" , "utc_timestamp" , 
      "values" , "varbinary" , "varchar" , "varcharacter" , "varying" , "when" , "where" , "while" , "with" , "write" , "x509" , "xor" , "year_month" , "zerofill");

  /**
   * 该表达式字符串描述符合的表数据结构
   */
  private static final String TABLEFORMAT = "^[0-9|a-z|A-Z]{1}[0-9|a-z|A-Z|_]*$";

  @Autowired
  private TableOperateRepositoryCustom tableOperateRepositoryCustom;
  @Autowired
  private TemplateRepository templateRepository;
  @Autowired
  private TemplatePropertyRepository templatePropertyRepository;
  @Autowired
  private TemplateRelationRepository templateRelationRepository;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private InstanceService instanceService;
  @Autowired
  private TemplateGroupService templateGroupService;
  @Autowired
  private TemplateItemService templateItemService;
  @Autowired
  private UserService userService;
  @Autowired
  private TemplateVisibilityService templateVisibilityService;
  @Autowired
  private TemplateMaintainerService templateMaintainerService;
  @Autowired
  private DynamicTemplateDraftService dynamicTemplateDraftService;

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.DynamicTemplateService#initDynamicTemplate(com.bizunited.platform.formengine.entity.TemplateEntity, java.security.Principal)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public TemplateEntity initDynamicTemplate(TemplateEntity template , Principal principal, boolean isImport) {
    /*
     * 操作过程如下：
     * 1、首先验证过template中信息的正确性
     * 注意：由于客户创建主表信息、明细表信息、分组信息时，只会关注那些业务字段，
     * 所以在验证过程中，我们要自行增加那些非业务字段（例如各表的主键信息、）
     * a、首先是主表信息的验证和自动完善（包括检查一般信息和关联信息）
     * b、然后是明细表的验证和完善（包括检查一般信息和关联信息）
     * c、然后是分组表信息的验证和完善（包括检查一般信息和关联信息）
     * d、检测主表、各明细表、各分组表等的表名是否存在重复的问题
     * 2、将当前模板的基本信息、关联信息、明细编辑信息和分组信息进行保存
     * 3、为当前主模板属性和分组下的属性建立初始化可见性（调用可见性Service完成）
     * 4、建立动态数据表
     * */
    // 1、======
    Validate.notNull(template , "基于动态模板创建数据表时，动态模板信息必须传入!!");
    String tableName = template.getTableName();
    Validate.notBlank(tableName , "进行动态模型创建，必须指定动态模型对应的数据表!!");
    String id = template.getId();
    Validate.isTrue(StringUtils.isBlank(id) , "动态表单模板初始化时，不能传入表单模板编号!!");
    template.setId(null);
    String cversion = template.getCversion();
    Validate.notBlank(cversion , "新的动态表单模板的版本号信息必须指定!!");
    String name = template.getName();
    Validate.notBlank(name , "新的动态表单模板的名称信息必须传入!!");
    String type = template.getType();
    Validate.notBlank(type , "动态表单模板类型必须传入");
    String code = template.getCode();
    Validate.notBlank(code , "动态模板必须填写业务编号!!");
    Validate.isTrue(StringUtils.equals(FormTemplateTypeEnum.DYNAMIC.getType(), template.getType()) , "该操作只支持动态模板，请检查!!");
    // 1.a、======
    Set<TemplatePropertyEntity> properties = template.getProperties();
    this.perfectingProperty(tableName , properties, false, isImport);
    // 检查其下的关联关系，是否都设定了目标数据表，以及目标数据表是否都存在
    Set<TemplateRelationEntity> relations = template.getRelations();
    if(relations != null && !relations.isEmpty()) {
      for (TemplateRelationEntity relation : relations) {
        perfectingRelation(tableName , relation, false);
      }
    }
    // 1.b、======
    Set<TemplateItemEntity> templateItems = template.getItemRelations();
    this.perfectingOneToManyRelation(template, null, tableName, templateItems, false, isImport);
    // 1.c、========
    Set<TemplateGroupEntity> templateGroups = template.getGroupRelations();
    this.perfectingOneToOneRelation(templateGroups, tableName, false, isImport);
    this.validate(template);
    // 1.d、=========
    Set<String> existTables = new TreeSet<>();
    existTables.add(tableName);
    if(templateItems != null && !templateItems.isEmpty()) {
      for (TemplateItemEntity templateItem : templateItems) {
        String itemTableName = templateItem.getTableName();
        Validate.isTrue(!existTables.contains(itemTableName) , "在检测明细表时，发现设定重复的数据表名，请检查!!");
        existTables.add(itemTableName);
      }
    }
    if(templateGroups != null && !templateGroups.isEmpty()) {
      for (TemplateGroupEntity templateGroupItem : templateGroups) {
        String groupTableName = templateGroupItem.getTableName();
        Validate.isTrue(!existTables.contains(groupTableName) , "在检测分组明细表时，发现设定重复的数据表名，请检查!!");
        existTables.add(groupTableName);
        // 还要检测分组表下的明细表信息
        Set<TemplateItemEntity> groupTemplateItems = templateGroupItem.getItemRelations();
        if(groupTemplateItems != null && !groupTemplateItems.isEmpty()) {
          for (TemplateItemEntity groupTemplateItem : groupTemplateItems) {
            String groupItemTableName = groupTemplateItem.getTableName();
            Validate.isTrue(!existTables.contains(groupItemTableName) , "在检测分组下的明细表时，发现设定重复的数据表名，请检查!!");
            existTables.add(groupItemTableName);
          }
        }
      }
    }

    // 2、======
    // 首先是动态模板的基本信息保存
    TemplateEntity currentTemplate = this.templateRepository.findByCodeAndCversion(code, cversion);
    Validate.isTrue(currentTemplate == null , "当前指定的表单模板编号和版本已经存在，请检查!!");
    super.saveTemplate(template, principal);
    // 然后是对主表的一般属性进行初始化
    this.initDynamicProperties(template, template.getProperties());
    // 然后是对主表的关联属性进行初始化
    this.initDynamicRelations(template, template.getRelations());
    // 然后对主表的OneToOne属性进行初始化
    templateGroups = template.getGroupRelations();
    if(templateGroups != null && !templateGroups.isEmpty()) {
      for (TemplateGroupEntity templateGroup : templateGroups) {
        this.templateGroupService.initDynamicGroups(templateGroup, template);
      }
    }
    // 然后是对主表的OneToMany属性进行初始化
    templateItems = template.getItemRelations();
    if(templateItems != null && !templateItems.isEmpty()) {
      for (TemplateItemEntity templateItem : templateItems) {
        this.templateItemService.initDynamicItems(templateItem , template, null, tableName);
      }
    }
    this.templateRepository.flush();

    // 3、=====
    Set<TemplateVisibilityEntity> templateVisibilities = this.templateVisibilityService.initDynamicTemplateVisibilities(template);
    template.setVisibility(templateVisibilities);

    // 4、======
    Boolean existTableName = this.tableOperateRepositoryCustom.existTableName(tableName);
    if(!existTableName) {
      // TODO: 2020-04-12 这里为了兼容导入的动态模版，暂时修改为判断主表是否存在，如果主表不存在就创建表，后续可以对这个兼容性做优化 a:陈加平
      this.tableOperateRepositoryCustom.createTable(template);
    }
    this.dynamicTemplateDraftService.deleteByCode(template.getCode());
    return template;
  }

  /**
   * 初始化一般模型
   * @param currentTemplate
   * @param properties
   */
  private void initDynamicProperties(TemplateEntity currentTemplate , Set<TemplatePropertyEntity> properties) {
    for (TemplatePropertyEntity templateProperty : properties) {
      templateProperty.setId(null);
      templateProperty.setCurrentTemplate(currentTemplate);
      templateProperty.setCurrentGroup(null);
      templateProperty.setCurrentItem(null);
      this.templatePropertyRepository.save(templateProperty);
    }
  }

  /**
   * 初始化关联模型，包括ManyToOne关联和ManyToMany关联
   * @param currentTemplate
   * @param relations
   */
  private void initDynamicRelations(TemplateEntity currentTemplate , Set<TemplateRelationEntity> relations) {
    if(relations == null || relations.isEmpty()) {
      return;
    }
    for (TemplateRelationEntity templateRelation : relations) {
      templateRelation.setId(null);
      templateRelation.setCurrentGroup(null);
      templateRelation.setCurrentItem(null);
      templateRelation.setCurrentTemplate(currentTemplate);
      // 将和静态模型有关的属性设置为空
      templateRelation.setQueryService("");
      this.templateRelationRepository.save(templateRelation);
    }
  }

  /**
   * 进行信息完善
   * @param parentTableName
   * @param properties
   * @param force 是否强制建表
   */
  private void perfectingProperty(String parentTableName , Set<TemplatePropertyEntity> properties , boolean force, boolean isImport) {
    /*
     * 过程包括：
     * 1、完善主表的id主键、以及forminstanceid信息（如果用户以传入，则需要报错）
     * 2、完善明细表的主键信息、反向外键关联信息（如果用户已传入，则需要报错）
     * 3、完善分组表的主键信息、反向外键关联信息（如果用户已传入，则需要报错）
     * */
    // 1、======
    //Validate.isTrue(properties != null && !properties.isEmpty() , "请至少设定一个在主表信息上的一般属性") ;
    // 查询主表是否存在
    if(!force && !isImport) {
      Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(parentTableName) , "检测到数据表'%s'已存在，请更换数据表名，或者使用强制更新(不推荐)!" , parentTableName);
    }
    // 注意，这里要自动新增PK和表单实例信息，因为外部可能没有传入
    Validate.isTrue(properties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0  , "进行动态模板初始化时，不可设定主表的id主键属性");
    properties.add(this.autoAppendPk());
    Validate.isTrue(properties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), FORMINSTANCE_ID)).count() == 0  , "进行动态模板初始化时，不可设定主表的formInstanceId表单实例属性");
    properties.add(this.autoAppendFormInstanceId());
    // 检查其下的一般属性信息
    for (TemplatePropertyEntity templateProperty : properties) {
      String propertyName = templateProperty.getPropertyName();
      Validate.notBlank(propertyName , "一般属性名称必须填写!!");
      Validate.isTrue(propertyName.length() <= 64 , "在初始化时发现字段[%s]的长度超过了64个字符，请检查!!" , propertyName);
      String propertyDesc = templateProperty.getPropertyDesc();
      Validate.notBlank(propertyDesc , "一般属性说明信息必须填写!!");
      String propertyDbName = templateProperty.getPropertyDbName();
      Validate.notBlank(propertyDbName , "一般属性'%s'的数据库字段名必须填写!!" , propertyDbName);
      Validate.isTrue(propertyDbName.length() <= 64 , "在初始化时发现字段的数据库属性[%s]的长度超过了64个字符，请检查!!" , propertyName);
      Validate.matchesPattern(propertyDbName, TABLEFORMAT , "字段名只能使用数字、字母（支持大小写）和“_”符号[%s]" , propertyDbName);
    }
  }

  /**
   * 验证指定模型下的关联属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param relation
   * @param parentTableName
   * @param force
   */
  private void perfectingRelation(String parentTableName , TemplateRelationEntity relation , boolean force) {
    // 注意不用判定关联关系的目标数据表存不存在，因为这只是一个辅助信息。
    String targetTableName = relation.getTargetTableName();
    Validate.notBlank(targetTableName , "关联信息必须设定数据表!!");
    String propertyName = relation.getPropertyName();
    Validate.notBlank(propertyName , "关联属性名称必须填写!!");
    String propertyDesc = relation.getPropertyDesc();
    Validate.notBlank(propertyDesc , "关联属性说明信息必须填写!!");
    String relationType = relation.getRelationType();
    Validate.notBlank(relationType , "关联属性类型必须填写!!");
    Validate.isTrue(StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_ONE.getRelation()) || StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_MANY.getRelation()) , "关联属性类型只能是ManyToOne或者ManyToMany");
    if(StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_ONE.getRelation())) {
      String propertyDbName = relation.getPropertyDbName();
      Validate.notBlank(propertyDbName , "关联属性'%s'的数据库字段名必须填写!!" , propertyName);
      Validate.matchesPattern(propertyDbName, TABLEFORMAT , "字段名只能使用数字、字母（支持大小写）和“_”符号[%s]" , propertyDbName);
    }
    String propertyClassName = relation.getPropertyClassName();
    if(StringUtils.isBlank(propertyClassName)) {
      relation.setPropertyClassName(String.class.getName());
    }
    // ManyToOne和ManyToMany关系的目标数据表都必须已经存在
    Validate.isTrue(this.tableOperateRepositoryCustom.existTableName(targetTableName) , "检测到数据表%s不存在，请检查!!" , targetTableName);
    // 如果是ManyToMany关联，还需要继续检查映射表
    if(StringUtils.equals(relation.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation()) && !force) {
      // 为了保证一张表和另一张表的映射表不会出现 两个，
      // 基于resouceTableName和targetTableName创建的映射表名，需要进行排序
      List<String> sortedNames = Arrays.stream(new String[]{parentTableName , targetTableName}).sorted().collect(Collectors.toList());
      String mappingTable = String.format(MAPPING, sortedNames.get(0) , sortedNames.get(1));
      Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(mappingTable) , "检测到数据表'%s'已存在,请更换数据表名，或者使用强制更新(不推荐)!!" , mappingTable);
    }
  }

  /**
   * 验证指定模型下的明细属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param parentTemplate
   * @param parentGroup
   * @param parentTableName
   * @param templateOneToManyItems
   * @param force
   */
  private void perfectingOneToManyRelation(TemplateEntity parentTemplate , TemplateGroupEntity parentGroup , String parentTableName , Set<TemplateItemEntity> templateOneToManyItems , boolean force, boolean isImport) {
    if(templateOneToManyItems == null || templateOneToManyItems.isEmpty()) {
      return;
    }

    for (TemplateItemEntity templateOneToManyItem : templateOneToManyItems) {
      String tableName = templateOneToManyItem.getTableName();
      Validate.notBlank(tableName , "发现至少一个明细编辑表没有设定数据表名!!");
      Validate.matchesPattern(tableName, TABLEFORMAT , "数据表只能使用数字、字母（支持大小写）和“_”符号[%s]！" , tableName);
      templateOneToManyItem.setParentTableName(parentTableName);
      // 查询数据表是否存在
      if(!force && !isImport) {
        Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(tableName) , "检测到数据表'%s'已存在，请更换数据表名，或者使用强制更新(不推荐)!!" , tableName);
      }
      Set<TemplatePropertyEntity> oneToManyProperties = templateOneToManyItem.getProperties();
      Validate.notNull(oneToManyProperties , "请至少为明细表（%s）设定一个一般属性!" , tableName);
      Validate.isTrue(!oneToManyProperties.isEmpty() , "请至少为明细表（%s）设定一个一般属性！" , tableName) ;
      // 完善明细表的主键信息
      Validate.isTrue(oneToManyProperties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0  , "进行动态模板初始化时，不可设定明细编辑表（%s）的id主键属性" , tableName);
      oneToManyProperties.add(this.autoAppendPk());
      Set<TemplateRelationEntity> omRelations = templateOneToManyItem.getRelations();
      if(omRelations != null && !omRelations.isEmpty()) {
        for (TemplateRelationEntity relation : omRelations) {
          this.perfectingRelation(parentTableName , relation, force);
        }
      }

      if(omRelations == null) {
        omRelations = new LinkedHashSet<>();
        templateOneToManyItem.setRelations(omRelations);
      }
      // 完善明细表和主表的反向关联信息
      TemplateRelationEntity backRelation = this.autoAppendBackRelation(parentTableName, parentTemplate, parentGroup, templateOneToManyItem);
      String backRelationDbName = backRelation.getPropertyDbName();
      Validate.isTrue(omRelations.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), backRelationDbName)).count() == 0
          , "进行动态模板初始化时，不可设定明细编辑表（%s）和主表的关联属性（%s）!" , tableName , backRelationDbName);
      omRelations.add(backRelation);
    }
  }

  /**
   * 验证指定模型下的分组属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param templateGroups
   * @param parentTableName
   * @param force
   */
  private void perfectingOneToOneRelation(Set<TemplateGroupEntity> templateGroups , String parentTableName , boolean force, boolean isImport) {
    if(templateGroups == null || templateGroups.isEmpty()) {
      return;
    }

    for (TemplateGroupEntity templateGroup : templateGroups) {
      String tableName = templateGroup.getTableName();
      Validate.notBlank(tableName , "发现至少一个分组信息表没有设定数据表名!!");
      Validate.matchesPattern(tableName, TABLEFORMAT , "数据表只能使用数字、字母（支持大小写）和 “_”符号[%s]" , tableName);
      templateGroup.setParentTableName(parentTableName);
      if(!force && !isImport) {
        Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(tableName) , "检测到数据表'%s'已存在，请更换数据表名，或者使用强制更新(不推荐)!!" , tableName);
      }
      Set<TemplatePropertyEntity> properties = templateGroup.getProperties();
      Validate.notNull(properties, "请至少为分组表（%s）设定一个一般属性!" , tableName);
      Validate.isTrue(!properties.isEmpty() , "请至少为分组表（%s） 设定一个一般属性" , tableName) ;
      // 注意，这里要自动新增PK
      Validate.isTrue(properties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0 , "进行动态模板初始化时，不可设定分组表的id主键属性");
      properties.add(this.autoAppendPk());

      // 检测其下的关联信息
      Set<TemplateRelationEntity> ooRelations = templateGroup.getRelations();
      if(ooRelations != null && !ooRelations.isEmpty()) {
        for (TemplateRelationEntity relation : ooRelations) {
          perfectingRelation(tableName , relation, force);
        }
      }
      if(ooRelations == null) {
        ooRelations = new LinkedHashSet<>();
        templateGroup.setRelations(ooRelations);
      }
      // 建立分组的反向关联信息
      TemplateRelationEntity backRelation = this.autoAppendBackRelation(parentTableName, null, templateGroup, null);
      String backRelationDbName = backRelation.getPropertyDbName();
      Validate.isTrue(ooRelations.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), backRelationDbName)).count() == 0
          , "进行动态模板初始化时，不可设定明细编辑表 （%s）和主表的关联属性（%s）" , tableName , backRelationDbName);
      ooRelations.add(backRelation);

      // 检测分组表下可能的明细编辑信息
      Set<TemplateItemEntity> templateItems = templateGroup.getItemRelations();
      this.perfectingOneToManyRelation(null, templateGroup, tableName, templateItems, force, isImport);
    }
  }

  /**
   * 该私有方法用于在生成和执行数据库建库脚本前，检查模板中的各个数据项是否合法。
   * @param template
   */
  private void validate(TemplateEntity template) {
    /*
     * 检查字段有没有重复，类型是否符合规定、字段名是否存在保留字
     * 1、检查主表下的一般字段和关联字段
     * 2、分组下的一般字段、关联字段
     * 3、分组下明细字段下的，一般字段、关联字段
     * 4、以及明细下的一般字段、关联字段
     * */
    // 1、===========
    Set<TemplatePropertyEntity> templateProperties = template.getProperties();
    templateProperties.forEach(item ->
      Validate.notBlank(item.getPropertyName() , "参数名必须填写")
    );
    List<String> propertyDbNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).sorted().collect(Collectors.toList());
    List<String> propertyClassNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyClassName).sorted().collect(Collectors.toList());
    List<String> propertyNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyName).sorted().collect(Collectors.toList());
    // 然后合并ManyToOne和ManyToMany性质的关联属性
    Set<TemplateRelationEntity> templateRelations = template.getRelations();
    if(templateRelations != null && !templateRelations.isEmpty()) {
      templateRelations.forEach(item ->
        Validate.notBlank(item.getPropertyName() , "关联参数名必须填写")
      );
      propertyDbNames.addAll(templateRelations.stream().filter(item -> StringUtils.equals(item.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())).map(TemplateRelationEntity::getPropertyDbName).sorted().collect(Collectors.toList()));
      propertyNames.addAll(templateRelations.stream().map(TemplateRelationEntity::getPropertyName).sorted().collect(Collectors.toList()));
    }
    // 开始检查
    this.validateCollections(propertyDbNames, propertyClassNames, propertyNames);

    // 2、===========
    Set<TemplateGroupEntity> templateGroups = template.getGroupRelations();
    if(templateGroups != null && !templateGroups.isEmpty()) {
      for (TemplateGroupEntity templateGroupItem : templateGroups) {
        Set<TemplatePropertyEntity> groupProperties = templateGroupItem.getProperties();
        groupProperties.forEach(item ->
          Validate.notBlank(item.getPropertyName() , "检验分组信息时，发现至少一个参数名没有填写!!")
        );

        List<String> groupPropertyDbNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).sorted().collect(Collectors.toList());
        List<String> groupPropertyClassNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyClassName).sorted().collect(Collectors.toList());
        List<String> groupPropertyNames = templateProperties.stream().map(TemplatePropertyEntity::getPropertyName).sorted().collect(Collectors.toList());
        // 然后合并这个分组下的ManyToOne和ManyToMany性质的关联属性
        Set<TemplateRelationEntity> groupRelations = templateGroupItem.getRelations();
        if(groupRelations != null && !groupRelations.isEmpty()) {
          groupRelations.forEach(item ->
            Validate.notBlank(item.getPropertyName() , "检验分组关联信息时，发现至少一个参数名没有填写!!")
          );
          groupPropertyDbNames.addAll(groupRelations.stream().filter(item -> StringUtils.equals(item.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())).map(TemplateRelationEntity::getPropertyDbName).sorted().collect(Collectors.toList()));
          groupPropertyNames.addAll(groupRelations.stream().map(TemplateRelationEntity::getPropertyName).sorted().collect(Collectors.toList()));
        }
        // 开始检查
        this.validateCollections(groupPropertyDbNames, groupPropertyClassNames, groupPropertyNames);

        // 3、==========
        Set<TemplateItemEntity> templateItems = templateGroupItem.getItemRelations();
        if(templateItems != null && !templateItems.isEmpty()) {
          for (TemplateItemEntity templateItem : templateItems) {
            Set<TemplatePropertyEntity> itemProperties = templateItem.getProperties();
            itemProperties.forEach(item ->
              Validate.notBlank(item.getPropertyName() , "分析明细编辑信息时，发现至少一个参数名没有填写!!")
            );
            // 单个明细项中的一般字段、关联字段，也需要判定
            List<String> itemPropertyDbNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).sorted().collect(Collectors.toList());
            List<String> itemPropertyClassNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyClassName).sorted().collect(Collectors.toList());
            List<String> itemPropertyNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyName).sorted().collect(Collectors.toList());
            // 然后合并这个明细下的ManyToOne和ManyToMany性质的关联属性
            Set<TemplateRelationEntity> itemRelations = templateItem.getRelations();
            if(itemRelations != null && !itemRelations.isEmpty()) {
              itemRelations.forEach(item ->
                Validate.notBlank(item.getPropertyName() , "分析明细编辑关联信息时，发现至少一个参数名没有填写!!")
              );
              itemPropertyDbNames.addAll(itemRelations.stream().filter(item -> StringUtils.equals(item.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())).map(TemplateRelationEntity::getPropertyDbName).sorted().collect(Collectors.toList()));
              itemPropertyNames.addAll(itemRelations.stream().map(TemplateRelationEntity::getPropertyName).sorted().collect(Collectors.toList()));
            }
            // 开始检查
            this.validateCollections(itemPropertyDbNames, itemPropertyClassNames, itemPropertyNames);
          }
        }
      }
    }

    // 4、=============
    Set<TemplateItemEntity> itemsRelations = template.getItemRelations();
    if(itemsRelations != null && !itemsRelations.isEmpty()) {
      for (TemplateItemEntity templateItem : itemsRelations) {
        Set<TemplatePropertyEntity> itemProperties = templateItem.getProperties();
        itemProperties.forEach(item ->
          Validate.notBlank(item.getPropertyName() , "分析明细编辑信息时，发现至少一个参数名没有填写!!")
        );

        // 单个明细项中的一般字段、关联字段，也需要判定
        List<String> itemPropertyDbNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyDbName).sorted().collect(Collectors.toList());
        List<String> itemPropertyClassNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyClassName).sorted().collect(Collectors.toList());
        List<String> itemPropertyNames = itemProperties.stream().map(TemplatePropertyEntity::getPropertyName).sorted().collect(Collectors.toList());
        // 然后合并这个明细下的ManyToOne和ManyToMany性质的关联属性
        Set<TemplateRelationEntity> itemRelations = templateItem.getRelations();
        if(itemRelations != null && !itemRelations.isEmpty()) {
          itemRelations.forEach(item ->
            Validate.notBlank(item.getPropertyName() , "分析明细编辑关联信息时，发现至少一个参数名没有填写!!")
          );
          itemPropertyDbNames.addAll(itemRelations.stream().filter(item -> StringUtils.equals(item.getRelationType(), RelationsTypeEnum.MANY_TO_ONE.getRelation())).map(TemplateRelationEntity::getPropertyDbName).sorted().collect(Collectors.toList()));
          itemPropertyNames.addAll(itemRelations.stream().map(TemplateRelationEntity::getPropertyName).sorted().collect(Collectors.toList()));
        }
        // 开始检查
        this.validateCollections(itemPropertyDbNames, itemPropertyClassNames, itemPropertyNames);
      }
    }
  }

  /**
   * 对一般属性、关联属性进行检查
   * @param propertyDbNames
   * @param propertyClassNames
   * @param propertyNames
   */
  private void validateCollections(List<String> propertyDbNames , List<String> propertyClassNames , List<String> propertyNames) {

    /*
     * 1、首先检查propertyDbNames、propertyNames集合中是否有值重复的元素
     * 2、然后检查propertyDbNames、propertyNames集合中是否有保留字
     * 3、检查propertyClassNames中的类型是否都是可处理的
     * */
    // 1、=======
    this.validateDuplicate(propertyDbNames);
    this.validateDuplicate(propertyNames);

    // 2、======
    SetView<String> intersections = Sets.intersection(DynamicTemplateServiceImpl.RETAINFIELDNAMES, new HashSet<>(propertyDbNames));
    if(!CollectionUtils.isEmpty(intersections)) {
      String intersectionValue = Arrays.toString(intersections.toArray(new String[]{}));
      throw new IllegalArgumentException(String.format("发现保留字被使用在数据库字段名中：%s", intersectionValue));
    }
    intersections = Sets.intersection(DynamicTemplateServiceImpl.RETAINFIELDNAMES, new HashSet<>(propertyNames));
    if(!CollectionUtils.isEmpty(intersections)) {
      String intersectionValue = Arrays.toString(intersections.toArray(new String[]{}));
      throw new IllegalArgumentException(String.format("发现保留字被使用在字段名中：%s", intersectionValue));
    }

    // 3、======
    propertyClassNames.forEach(item -> {
      if(DynamicTemplateServiceImpl.classTypeMapping.get(item) == null) {
        throw new IllegalArgumentException(String.format("不支持的类型: %s (如果是基础类型，请使用对应的主类型)", item));
      }
    });
  }

  /**
   * 该私有方法检查指定的集合中是否有重复的值
   * @param properties
   */
  private void validateDuplicate(List<String> properties) {
    List<String> destProperties = new ArrayList<>(properties);
    List<String> duplicateProperties = new ArrayList<>();
    for(int index = 0 ; index < destProperties.size() ;) {
      String currentProperty = destProperties.get(index);
      if(duplicateProperties.contains(currentProperty)) {
        throw new IllegalArgumentException(String.format("在一组不能重复的字段定义中，发现重复的字段名%s，请检查!!" , currentProperty));
      }

      duplicateProperties.add(destProperties.remove(index));
    }
  }
  /**
   * 为当前正在处理的过程，增加“主键”信息
   * @return
   */
  private TemplatePropertyEntity autoAppendPk() {
    TemplatePropertyEntity pk = new TemplatePropertyEntity();
    pk.setIndex(0);
    pk.setNullable(false);
    pk.setPropertyClassName(String.class.getName());
    pk.setPropertyDbName("id");
    pk.setPropertyDesc("主键");
    pk.setPropertyName("id");
    pk.setPrimaryKey(true);
    pk.setMaxLen(255);
    pk.setUnique(false);
    return pk;
  }

  /**
   * 为当前正在处理主动太模型，增加“表单实例编号”信息
   * @return
   */
  private TemplatePropertyEntity autoAppendFormInstanceId() {
    TemplatePropertyEntity formInstanceId = new TemplatePropertyEntity();
    formInstanceId.setIndex(0);
    formInstanceId.setNullable(false);
    formInstanceId.setPropertyClassName(String.class.getName());
    formInstanceId.setPropertyDbName("form_instance_id");
    formInstanceId.setPropertyDesc("表单实例编号");
    formInstanceId.setPropertyName(FORMINSTANCE_ID);
    formInstanceId.setUnique(true);
    formInstanceId.setMaxLen(255);
    return formInstanceId;
  }

  /**
   * 该私有方法在指定的resouceTableName的基础上，建立一个反向的ManyToOne关联。
   * @param parentTableName
   * @param currentTemplate
   * @param currentGroup
   * @param currentItem
   * @return
   */
  private TemplateRelationEntity autoAppendBackRelation(String parentTableName , TemplateEntity currentTemplate
      , TemplateGroupEntity currentGroup , TemplateItemEntity currentItem) {
    TemplateRelationEntity relation = new TemplateRelationEntity();
    relation.setBackProperty(true);
    relation.setCurrentGroup(currentGroup);
    relation.setCurrentItem(currentItem);
    relation.setCurrentTemplate(currentTemplate);
    relation.setIndex(0);
    relation.setNullable(false);
    relation.setPropertyClassName(String.class.getName());
    relation.setPropertyDbName(String.format("%s_id", parentTableName));
    relation.setPropertyDesc("明细关联");
    relation.setPropertyName(String.format("%s_id", parentTableName));
    relation.setRelationType(RelationsTypeEnum.MANY_TO_ONE.getRelation());
    relation.setTargetTableName(parentTableName);
    return relation;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.formengine.starter.service.DynamicTemplateService#upgrade(java.lang.String, com.bizunited.platform.formengine.entity.TemplateEntity, java.lang.String, java.security.Principal)
   */
  @Override
  @Transactional
  @CacheEvict(cacheNames="template" , allEntries=true)
  public TemplateEntity upgrade(String templateId, TemplateEntity newTemplate, String newVersion, boolean updateInstance, Principal principal) {
    /*
     * 1、查询之前参考的动态模板，并和当前最新的template进行验证
     *  1.1、验证主表单中的一般属性和关联属性
     *  1.2、验证各分组信息，包括新增的分组、分组下的一般属性和关联属性
     *  1.3、验证各明细编辑信息，包括新增的明细编辑、明细下的一般属性和关联属性
     * 2、验证成功后，对主表、各分组、各明细进行保存
     *  2.1、首先保存主表信息
     *  2.2、然后是主表下的分组信息
     *  2.3、最后是主表下的明细编辑信息
     * 3、进行布局信息的拷贝（注意布局的信息可能包括PC布局、移动端布局和打印端布局）
     * 4、进行事件信息的拷贝
     * 5、进行可见性信息的拷贝
     * */
    // 1、=======
    Validate.notNull(principal , "必须指定操作者信息!!");
    // 验证和查询旧模板
    Validate.notBlank(templateId , "参照动态模板的编号必须进行传入!!");
    Validate.notBlank(newVersion , "升级时,新的版本信息必须传入");
    TemplateEntity oldTemplate = this.templateService.findDetailsById(templateId);
    Validate.notNull(oldTemplate , "未找到指定的原始动态模板参照信息，请检查!!");
    Validate.isTrue(StringUtils.equals(oldTemplate.getType(), FormTemplateTypeEnum.DYNAMIC.getType()) , "参照模板【%s】必须是“动态”模板，请检查" , templateId);
    String tableName = oldTemplate.getTableName();
    Validate.notBlank(tableName , "错误的业务数据表信息，请检查!!");
    // 动态模板升级时，只能从最新的模板进行升级
    List<TemplateEntity> oldTemplates = this.templateRepository.findByCode(oldTemplate.getCode());
    Validate.isTrue(oldTemplates != null && !oldTemplates.isEmpty() , "没有找到原有模板中的编号对应的任何历史编号信息");
    TemplateEntity lastTemplate = oldTemplates.get(0);
    Validate.isTrue(StringUtils.equals(templateId, lastTemplate.getId()) , "动态模板升级时，只能参照最新的模板进行升级!");
    // 验证新模板
    Validate.notNull(newTemplate , "新模板信息必须传入");
    Validate.isTrue(StringUtils.isBlank(newTemplate.getId()) , "新模板的id编号不能传入");
    TemplateEntity exsitTemplate = this.templateRepository.findByCodeAndCversion(oldTemplate.getCode(), newVersion);
    Validate.isTrue(exsitTemplate == null , "指定的版本编号信息已经在指定的code[%s]中存在，请重新指定版本编号信息!!" , newTemplate.getCode());
    Validate.isTrue(StringUtils.equals(newTemplate.getType(), FormTemplateTypeEnum.DYNAMIC.getType()) , "该升级方法只适用于“动态表单模板”!!");
    // 1.1、======
    Set<TemplatePropertyEntity> oldProperties = oldTemplate.getProperties();
    if(oldProperties == null) {
      oldProperties = Sets.newHashSet();
      oldTemplate.setProperties(oldProperties);
    }
    Set<TemplatePropertyEntity> newProperties = newTemplate.getProperties();
    if(newProperties == null) {
      newProperties = Sets.newHashSet();
      newTemplate.setProperties(newProperties);
    }
    Set<TemplateRelationEntity> oldRelations = oldTemplate.getRelations();
    if(oldRelations == null) {
      oldRelations = Sets.newHashSet();
      oldTemplate.setRelations(oldRelations);
    }
    Set<TemplateRelationEntity> newRelations = newTemplate.getRelations();
    if(newRelations == null) {
      newRelations = Sets.newHashSet();
      newTemplate.setRelations(newRelations);
    }
    // 新版的基本属性和关联属性
    this.perfectingUpgradeProperty(tableName, newProperties);
    this.validatePropertiesAndRelations(tableName, oldProperties , newProperties , oldRelations , newRelations);
    // 1.2、======
    Set<TemplateGroupEntity> newGroups = newTemplate.getGroupRelations();
    if(newGroups == null) {
      newGroups = Sets.newHashSet();
      newTemplate.setGroupRelations(newGroups);
    }
    Set<TemplateGroupEntity> oldGroups = oldTemplate.getGroupRelations();
    if(oldGroups == null) {
      oldGroups = Sets.newHashSet();
      oldTemplate.setGroupRelations(oldGroups);
    }
    // 补全可能存在的OneToOne关联信息
    this.perfectingUpgradeOneToOneRelation(newGroups, oldGroups, tableName);
    this.validateGroups(oldGroups, newGroups);
    // 1.3、======
    Set<TemplateItemEntity> newItems = newTemplate.getItemRelations();
    if(newItems == null) {
      newItems = Sets.newHashSet();
      newTemplate.setItemRelations(newItems);
    }
    Set<TemplateItemEntity> oldItems = oldTemplate.getItemRelations();
    if(oldItems == null) {
      oldItems = Sets.newHashSet();
      oldTemplate.setItemRelations(oldItems);
    }
    // 补全可能存在的OneToMany关联信息
    this.perfectingUpgradeOneToManyRelation(newTemplate, null, tableName, newItems, oldItems);
    this.validateItems(newItems, oldItems);

    // 2、========
    // 首先更新模板的基本信息
    String account = principal.getName();
    UserVo user = this.userService.findByAccount(account);
    Validate.notNull(user , "未找到任何创建者信息，请检查!!");
    // TODO: 2020-03-24  需要保存用户的账户名
    newTemplate.setCreator(account);
    newTemplate.setCreateTime(new Date());
    newTemplate.setCversion(newVersion);
    newTemplate.setTableName(tableName);
    newTemplate.setModifyTime(new Date());
    newTemplate.setProjectName(oldTemplate.getProjectName());
    this.templateRepository.save(newTemplate);
    this.initDynamicProperties(newTemplate, newProperties);
    Set<TemplateRelationEntity> relations = newTemplate.getRelations();
    if(relations != null && !relations.isEmpty()) {
      this.initDynamicRelations(newTemplate, relations);
    }
    // 然后是可能存在的分组信息
    Set<TemplateGroupEntity> templateGroups = newTemplate.getGroupRelations();
    if(templateGroups != null && !templateGroups.isEmpty()) {
      for (TemplateGroupEntity templateGroup : templateGroups) {
        this.templateGroupService.initDynamicGroups(templateGroup, newTemplate);
      }
    }
    // 最后是可能存在的明细编辑信息
    Set<TemplateItemEntity> templateItems = newTemplate.getItemRelations();
    if(templateItems != null && !templateItems.isEmpty()) {
      for (TemplateItemEntity templateItem : templateItems) {
        this.templateItemService.initDynamicItems(templateItem, newTemplate, null, tableName);
      }
    }

    // 3、========再是复制布局信息
    String newTemplateId = newTemplate.getId();
    super.upgradeTemplateLayout(templateId, newTemplateId, newTemplate);
    // 4、======== 再是复制事件信息
    super.upgradeTemplateEvent(templateId, newTemplateId, newTemplate);
    // 5、======== 然后是复制可见性信息
    super.upgradeTemplateVisibility(templateId, newTemplateId, newTemplate);
    // 6、======== 最后是构建升级的数据表脚本，并执行
    this.tableOperateRepositoryCustom.upgradeTable(oldTemplate, newTemplate);
    // 7、======== 修改为当前最新升级的template为默认的version
    this.templateService.updateDefaultVersion(newTemplateId);

    //8、======== 查询现有的维护人员，并绑定关系
    Set<TemplateMaintainerEntity> maintainers = templateMaintainerService.findByTemplateId(templateId);
    if(!CollectionUtils.isEmpty(maintainers)) {
      String[] accounts = maintainers.stream().map(TemplateMaintainerEntity::getUserAccount).collect(Collectors.toList()).toArray(new String[maintainers.size()]);
      templateMaintainerService.binding(templateId, accounts);
    }
    if(updateInstance) {
      instanceService.updateTemplate(newTemplate);
    }
    return newTemplate;
  }

  /**
   * 验证分组信息
   * @param oldTemplateGroups
   * @param newTemplateGroups
   */
  private void validateGroups(Set<TemplateGroupEntity> oldTemplateGroups , Set<TemplateGroupEntity> newTemplateGroups) {
    // 不能重复
    List<String> oldGroupPropertyListNames = newTemplateGroups.stream().map(TemplateGroupEntity::getPropertyName).collect(Collectors.toList());
    this.validateDuplicate(new LinkedList<>(oldGroupPropertyListNames));
    Set<String> newGroupPropertyNames = newTemplateGroups.stream().map(TemplateGroupEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> oldGroupPropertyNames = oldTemplateGroups.stream().map(TemplateGroupEntity::getPropertyName).collect(Collectors.toSet());
    Validate.isTrue(Sets.difference(oldGroupPropertyNames, newGroupPropertyNames).isEmpty() , "在进行模板升级时，分组信息只允许新增不允许删除!!");
    // 做成mapping，以便后续进行调用
    Map<String, TemplateGroupEntity> oldMappingGroups = oldTemplateGroups.stream().collect(Collectors.toMap(TemplateGroupEntity::getPropertyName, item -> item));
    for (TemplateGroupEntity newTemplateGroup : newTemplateGroups) {
      TemplateGroupEntity oldTemplateGroup = oldMappingGroups.get(newTemplateGroup.getPropertyName());
      String groupTableName = newTemplateGroup.getTableName();
      // 如果条件成立，说明是新增分组操作，所以不需要进行验证了
      if(oldTemplateGroup == null) {
        continue;
      }
      // 否则就是调整操作，那么就验证其下的一般属性、关联属性和可能的明细编辑信息
      Set<TemplatePropertyEntity> newGroupProperties = newTemplateGroup.getProperties();
      Set<TemplateRelationEntity> newGroupRelations = newTemplateGroup.getRelations();
      Set<TemplatePropertyEntity> oldGroupProperties = oldTemplateGroup.getProperties();
      Set<TemplateRelationEntity> oldGroupRelations = oldTemplateGroup.getRelations();
      this.validatePropertiesAndRelations(groupTableName, oldGroupProperties, newGroupProperties, oldGroupRelations, newGroupRelations);
      // 1.3、======= 验证其下的明细信息
      Set<TemplateItemEntity> newItems = newTemplateGroup.getItemRelations();
      if(newItems == null) {
        newItems = Sets.newHashSet();
      }
      Set<TemplateItemEntity> oldItems = oldTemplateGroup.getItemRelations();
      if(oldItems == null) {
        oldItems = Sets.newHashSet();
      }
      this.validateItems(newItems, oldItems);
    }
  }

  /**
   * 验证明细编辑
   * @param newItems
   * @param oldItems
   */
  private void validateItems(Set<TemplateItemEntity> newItems , Set<TemplateItemEntity> oldItems) {
    List<String> newItemNames = newItems.stream().map(TemplateItemEntity::getPropertyName).collect(Collectors.toList());
    this.validateDuplicate(newItemNames);
    // 做成mapping，以便后续进行调用
    Map<String, TemplateItemEntity> oldMappingItems = oldItems.stream().collect(Collectors.toMap(TemplateItemEntity::getPropertyName, item -> item));
    for (TemplateItemEntity newItem : newItems) {
      // 如果条件成立，说明是新增分组操作，所以不需要进行验证了
      String propertyName = newItem.getPropertyName();
      String itemTableName = newItem.getTableName();
      Validate.notBlank(itemTableName , "明细表必须指定，请检查!!");
      TemplateItemEntity oldItem = oldMappingItems.get(propertyName);
      if(oldItem == null) {
        continue;
      }

      Set<TemplatePropertyEntity> oldItemProperties = oldItem.getProperties();
      Set<TemplatePropertyEntity> newItemProperties = newItem.getProperties();
      Set<TemplateRelationEntity> oldItemRelations = oldItem.getRelations();
      Set<TemplateRelationEntity> newItemRelations = newItem.getRelations();
      this.validatePropertiesAndRelations(itemTableName, oldItemProperties, newItemProperties, oldItemRelations, newItemRelations);
    }
  }

  /**
   * 在升级前验证指定表单下的一般属性和关联属性
   * @param tableName
   * @param oldProperties
   * @param newProperties
   * @param oldRelations
   * @param newRelations
   */
  private void validatePropertiesAndRelations(String tableName , Set<TemplatePropertyEntity> oldProperties , Set<TemplatePropertyEntity> newProperties , Set<TemplateRelationEntity> oldRelations , Set<TemplateRelationEntity> newRelations) {
    /*
     * 验证主表单中的一般属性和关联属性，
     * a、只能新增，不能进行修改；
     * b、只允许已有的not null改为nullable，反之则不行；
     * c、String类型的属性，长度只能一样或者新增
     * */
    // 字段信息中的getPropertyName不能重复
    List<String> newPropertyListNames = newProperties.stream().map(TemplatePropertyEntity::getPropertyName).collect(Collectors.toList());
    List<String> newRelationListNames = newRelations.stream().map(TemplateRelationEntity::getPropertyName).collect(Collectors.toList());
    this.validateDuplicate(newPropertyListNames);
    this.validateDuplicate(newRelationListNames);

    Set<String> oldPropertyNames = oldProperties.stream().map(TemplatePropertyEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> newPropertyNames = newProperties.stream().map(TemplatePropertyEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> oldRelationNames = oldRelations.stream().map(TemplateRelationEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> newRelationNames = newRelations.stream().map(TemplateRelationEntity::getPropertyName).collect(Collectors.toSet());
    // id不能传入
    Validate.isTrue(newProperties.stream().filter(item -> !StringUtils.isBlank(item.getId())).count() == 0 , "进行升级时，所有一般属性的id编号都不能传入");
    Validate.isTrue(newRelations.stream().filter(item -> !StringUtils.isBlank(item.getId())).count() == 0 , "进行升级时，所有关联属性的id编号都不能传入");
    // 设置成Mapping，便于后续使用
    Map<String, TemplatePropertyEntity> newMappingProperties = newProperties.stream().collect(Collectors.toMap(TemplatePropertyEntity::getPropertyName, item -> item));
    Map<String, TemplateRelationEntity> newMappingRelations = newRelations.stream().collect(Collectors.toMap(TemplateRelationEntity::getPropertyName, item -> item));
    // 属性只能进行新增
    Validate.isTrue(Sets.difference(oldPropertyNames, newPropertyNames).isEmpty() , "一般属性只能进行新增，不能进行删除。请检查【%s】" , tableName);
    Validate.isTrue(Sets.difference(oldRelationNames, newRelationNames).isEmpty() , "关联属性只能进行新增，不能进行删除。请检查【%s】" , tableName);
    // 只允许已有的not null改为nullable，反之则不行
    Set<String> oldNullProperties = oldProperties.stream().filter(TemplatePropertyEntity::getNullable).map(TemplatePropertyEntity::getPropertyName).collect(Collectors.toSet());
    for (String oldNullPropertyName : oldNullProperties) {
      TemplatePropertyEntity newTemplateProperty = newMappingProperties.get(oldNullPropertyName);
      Validate.notNull(newTemplateProperty , "未发现指定的一般属性信息，请检查!!");
      Validate.isTrue(newTemplateProperty.getNullable() , "指定的一般属性在升级时，不能被从 not null able设置为null able，请检查【%s】" , oldNullPropertyName);
    }
    Set<String> oldNullRelations = oldRelations.stream().filter(TemplateRelationEntity::getNullable).map(TemplateRelationEntity::getPropertyName).collect(Collectors.toSet());
    for (String oldNullRelationName : oldNullRelations) {
      TemplateRelationEntity newTemplateRelation = newMappingRelations.get(oldNullRelationName);
      Validate.notNull(newTemplateRelation , "未发现指定的关联属性信息，请检查!!");
      Validate.isTrue(newTemplateRelation.getNullable() , "指定的关联属性在升级时，不能被从 not null able设置为null able，请检查【%s】" , oldNullRelationName);
    }
    // 新增的一般字段和关联字段都不能是必填
    SetView<String> appendedPropertyNames = Sets.difference(newPropertyNames , oldPropertyNames);
    if(!appendedPropertyNames.isEmpty()) {
      for (String appendedPropertyName : appendedPropertyNames) {
        TemplatePropertyEntity item = newMappingProperties.get(appendedPropertyName);
        Validate.isTrue(item.getNullable() , "新增一般字段%s，不能设置为 not nullable，请检查" , appendedPropertyName);
      }
    }
    SetView<String> appendedRelationNames = Sets.difference(newRelationNames , oldRelationNames);
    if(!appendedRelationNames.isEmpty()) {
      for (String appendedRelationName : appendedRelationNames) {
        TemplateRelationEntity item = newMappingRelations.get(appendedRelationName);
        Validate.isTrue(item.getNullable() , "新增的关联字段%s，不能设置为 not nullable，请检查" , appendedRelationName);
      }
    }
    // String类型的属性，长度只能一样或者新增
    Set<TemplatePropertyEntity> oldStringProperties = oldProperties.stream().filter(item -> StringUtils.equals(String.class.getName(), item.getPropertyClassName())).collect(Collectors.toSet());
    for(TemplatePropertyEntity oldStringProperty : oldStringProperties) {
      Integer oldMaxLen = oldStringProperty.getMaxLen();
      if(oldMaxLen == null) {
        oldMaxLen = 255;
      }
      String propertyName = oldStringProperty.getPropertyName();
      TemplatePropertyEntity newTemplateProperty = newMappingProperties.get(propertyName);
      Validate.notNull(newTemplateProperty , "没有找到指定的管理属性，请检查【%s】" , propertyName);
      Integer newMaxLen = newTemplateProperty.getMaxLen();
      Validate.isTrue(newMaxLen >= oldMaxLen , "在进行模板升级时，字符串属性字段的长度只能增加不能减少，请检查%s【%s】" , propertyName , tableName);
    }
  }

  /**
   * 进行一般属性信息完善（在进行主表升级的使用）
   * @param parentTableName
   * @param newProperties
   */
  private void perfectingUpgradeProperty(String parentTableName , Set<TemplatePropertyEntity> newProperties) {
    /*
     * 过程包括：
     * 1、完善主表的id主键、以及forminstanceid信息（如果用户以传入，则需要报错）
     * 2、完善明细表的主键信息、反向外键关联信息（如果用户已传入，则需要报错）
     * 3、完善分组表的主键信息、反向外键关联信息（如果用户已传入，则需要报错）
     * */
    // 1、======
    //Validate.isTrue(newProperties != null && !newProperties.isEmpty() , "请至少设定一个在主表信息上的一般属性") ;
    // 查询主表是否存在
    Validate.isTrue(this.tableOperateRepositoryCustom.existTableName(parentTableName) , "检测到主业务数据表'%s'不存在，请检查!!" , parentTableName);
    // 注意，这里要自动新增PK和表单实例信息，因为外部可能没有传入
    Validate.isTrue(newProperties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0  , "进行动态模板升级时，不可设定主表的id主键属性");
    newProperties.add(this.autoAppendPk());
    Validate.isTrue(newProperties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), FORMINSTANCE_ID)).count() == 0  , "进行动态模板升级时，不可设定主表的formInstanceId表单实例属性");
    newProperties.add(this.autoAppendFormInstanceId());
    // 检查其下的一般属性信息
    for (TemplatePropertyEntity newProperty : newProperties) {
      String propertyName = newProperty.getPropertyName();
      Validate.notBlank(propertyName , "一般属性名称必须填写!!");
      String propertyDesc = newProperty.getPropertyDesc();
      Validate.notBlank(propertyDesc , "一般属性说明信息必须填写!!");
      String propertyDbName = newProperty.getPropertyDbName();
      Validate.notBlank(propertyDbName , "一般属性'%s'的数据库字段名必须填写!!" , propertyDbName);
    }
  }

  /**
   * 验证指定模型下的关联属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param parentTableName
   * @param newRelation
   * @param oldRelation
   */
  private void perfectingUpgradeRelation(String parentTableName , TemplateRelationEntity newRelation , TemplateRelationEntity oldRelation) {
    // 注意不用判定关联关系的目标数据表存不存在，因为这只是一个辅助信息。
    String targetTableName = newRelation.getTargetTableName();
    Validate.notBlank(targetTableName , "关联信息必须设定数据表!!");
    String propertyName = newRelation.getPropertyName();
    Validate.notBlank(propertyName , "关联属性名称必须填写!!");
    String propertyDesc = newRelation.getPropertyDesc();
    Validate.notBlank(propertyDesc , "关联属性说明信息必须填写!!");
    String relationType = newRelation.getRelationType();
    Validate.notBlank(relationType , "关联属性类型必须填写!!");
    Validate.isTrue(StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_ONE.getRelation()) || StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_MANY.getRelation()) , "关联属性类型只能是ManyToOne或者ManyToMany");
    if(StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
      String propertyDbName = newRelation.getPropertyDbName();
      Validate.notBlank(propertyDbName , "关联属性'%s'的数据库字段名必须填写!!" , propertyName);
    }
    String propertyClassName = newRelation.getPropertyClassName();
    if(StringUtils.isBlank(propertyClassName)) {
      newRelation.setPropertyClassName(String.class.getName());
    }
    // ManyToOne和ManyToMany关系的目标数据表都必须已经存在
    Validate.isTrue(this.tableOperateRepositoryCustom.existTableName(targetTableName) , "检测到数据表%s不存在，请检查!!" , targetTableName);

    /*
     * 检测映射表，有两种情况：
     * 1、如果当前newRelation是新增的，则这个映射表不能存在
     * 2、如果当前newRelation已经存在的，则这个映射表应该存在
     * */
    // 如果条件成立，则说明是情况1
    if(oldRelation == null && StringUtils.equals(newRelation.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
      List<String> sortedNames = Arrays.stream(new String[]{parentTableName , targetTableName}).sorted().collect(Collectors.toList());
      String mappingTable = String.format(MAPPING, sortedNames.get(0) , sortedNames.get(1));
      Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(mappingTable) , "检测到数据表 '%s'已存在，请更换数据表名!!" , mappingTable);
    }
    // 否则是情况2
    else if(oldRelation != null && StringUtils.equals(newRelation.getRelationType(), RelationsTypeEnum.MANY_TO_MANY.getRelation())){
      List<String> sortedNames = Arrays.stream(new String[]{parentTableName , targetTableName}).sorted().collect(Collectors.toList());
      String mappingTable = String.format(MAPPING, sortedNames.get(0) , sortedNames.get(1));
      Validate.isTrue(this.tableOperateRepositoryCustom.existTableName(mappingTable) , "检测到数据表'%s'不存在，请检查!!" , mappingTable);
    }
  }

  /**
   * 验证指定模型下的明细属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param parentTemplate
   * @param parentGroup
   * @param parentTableName
   * @param newItems
   * @param oldItems
   */
  private void perfectingUpgradeOneToManyRelation(TemplateEntity parentTemplate , TemplateGroupEntity parentGroup , String parentTableName , Set<TemplateItemEntity> newItems , Set<TemplateItemEntity> oldItems) {
    /*
     * 由于前端不会传入任何除了业务字段以外的其它字段（例如回溯字段），所以：
     * 1、任何明细信息都需要补全，主要就是补全回溯字段、ID字段
     * 2、如果存在新增的明细信息，则还需要进行数据表存在性的验证
     * */
    // 做成Mapping，方便后续调用
    Map<String, TemplateItemEntity> oldItemsMapping;
    if(CollectionUtils.isEmpty(oldItems)) {
      return;
    }
    oldItemsMapping = oldItems.stream().collect(Collectors.toMap(TemplateItemEntity::getTableName, item -> item));
    // 1、======
    for (TemplateItemEntity newItem : newItems) {
      String tableName = newItem.getTableName();
      Validate.notBlank(tableName , "发现至少一个明细编辑表没有设定数据表名!!");
      Validate.matchesPattern(tableName, TABLEFORMAT , "数据表只能使用数字、字母（支持大小写）和“_”符号[%s]" , tableName);
      newItem.setParentTableName(parentTableName);
      Set<TemplatePropertyEntity> oneToManyProperties = newItem.getProperties();
      Validate.notNull(oneToManyProperties , "请至少为明细表（%s）设定一个一般属性" , tableName);
      Validate.isTrue(!oneToManyProperties.isEmpty() , "请至少为明细表（%s）设定一个一般属性" , tableName) ;
      // 完善明细表的主键信息
      Validate.isTrue(oneToManyProperties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0  , "进行动态模板初始化时，不可设定明细编辑表（%s）的id主键属性" , tableName);
      oneToManyProperties.add(this.autoAppendPk());

      // 完成明细表下的关联信息
      Set<TemplateRelationEntity> omRelations = newItem.getRelations();
      TemplateItemEntity oldItem = oldItemsMapping.get(tableName);
      if(omRelations != null && !omRelations.isEmpty()) {
        Map<String , TemplateRelationEntity> oldRelationMappings = Maps.newHashMap();
        if(oldItem != null) {
          Set<TemplateRelationEntity> oldRelations = oldItem.getRelations();
          if(oldRelations != null && !oldRelations.isEmpty()) {
            oldRelationMappings = oldRelations.stream().collect(Collectors.toMap(TemplateRelationEntity::getPropertyName, item -> item));
          }
        }

        for (TemplateRelationEntity relation : omRelations) {
          String propertyName = relation.getPropertyName();
          this.perfectingUpgradeRelation(tableName, relation, oldRelationMappings.get(propertyName));
        }
      }

      if(omRelations == null) {
        omRelations = new LinkedHashSet<>();
        newItem.setRelations(omRelations);
      }
      // 完善明细表和主表的反向关联信息
      TemplateRelationEntity backRelation = this.autoAppendBackRelation(parentTableName, parentTemplate, parentGroup, newItem);
      String backRelationDbName = backRelation.getPropertyDbName();
      Validate.isTrue(omRelations.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), backRelationDbName)).count() == 0
          , "进行动态模板初始化时，不可设定明细编辑表（%s）和主表的关联属性（%s）" , tableName , backRelationDbName);
      omRelations.add(backRelation);
    }

    // 2、======
    Set<String> oldItemNames = oldItems.stream().map(TemplateItemEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> newItemNames = newItems.stream().map(TemplateItemEntity::getPropertyName).collect(Collectors.toSet());
    SetView<String> appendItemNames = Sets.difference(newItemNames, oldItemNames);
    Set<TemplateItemEntity> appendItems = newItems.stream().filter(item -> appendItemNames.contains(item.getPropertyName())).collect(Collectors.toSet());
    for (TemplateItemEntity appendItem : appendItems) {
      String tableName = appendItem.getTableName();
      // 查询数据表是否存在
      Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(tableName) , "检测到数据表'%s'已存在，请更换数据表名!!" , tableName);
    }
  }

  /**
   * 验证指定模型下的分组属性，保证其关联信息、数据表信息的正确性，并补齐相关关联信息
   * @param newGroups
   * @param oldGroups
   * @param parentTableName
   */
  private void perfectingUpgradeOneToOneRelation(Set<TemplateGroupEntity> newGroups , Set<TemplateGroupEntity> oldGroups , String parentTableName) {
    /*
     * 由于前端不会传入任何除了业务字段以外的其它字段（例如回溯字段），所以：
     * 1、任何分组信息都需要补全，主要就是补全回溯字段
     * 2、如果存在新增的分组信息，则还需要进行数据表存在性的验证
     * */
    Map<String, TemplateGroupEntity> oldGroupMappings = Maps.newHashMap();
    if(oldGroups != null) {
      oldGroupMappings = oldGroups.stream().collect(Collectors.toMap(TemplateGroupEntity::getTableName, item -> item));
    }
    // 1、======
    for (TemplateGroupEntity newGroup : newGroups) {
      String tableName = newGroup.getTableName();
      Validate.notBlank(tableName , "发现至少一个分组信息表没有设定数据表名!!");
      Validate.matchesPattern(tableName, TABLEFORMAT , "数据表只能使用数字、字母（支持大小写）和“_”符号[%s]" , tableName);
      newGroup.setParentTableName(parentTableName);
      Set<TemplatePropertyEntity> properties = newGroup.getProperties();
      Validate.notNull(properties , "请至少为分组表（%s）设定一个一般属性" , tableName);
      Validate.isTrue(!properties.isEmpty() , "请至少为分组表（%s）设定一个一般属性" , tableName);
      // 注意，这里要自动新增PK
      Validate.isTrue(properties.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), "id")).count() == 0 , "进行动态模板升级时，不可设定分组表的id主键属性");
      properties.add(this.autoAppendPk());

      // 检测其下的关联信息
      Set<TemplateRelationEntity> newRelations = newGroup.getRelations();
      TemplateGroupEntity oldGroup = oldGroupMappings.get(tableName);
      if(newRelations == null) {
        newRelations = new LinkedHashSet<>();
        newGroup.setRelations(newRelations);
      }
      Map<String, TemplateRelationEntity> oldGroupRelationMappings = Maps.newHashMap();
      if(oldGroup != null) {
        Set<TemplateRelationEntity> oldGroupRelations = oldGroup.getRelations();
        if(oldGroupRelations != null) {
          oldGroupRelationMappings = oldGroupRelations.stream().collect(Collectors.toMap(TemplateRelationEntity::getPropertyName, item -> item));
        }
      }

      for (TemplateRelationEntity relation : newRelations) {
        TemplateRelationEntity oldGroupRelation = oldGroupRelationMappings.get(relation.getPropertyName());
        this.perfectingUpgradeRelation(tableName, relation, oldGroupRelation);
      }
      // 建立分组的反向关联信息
      TemplateRelationEntity backRelation = this.autoAppendBackRelation(parentTableName, null, newGroup, null);
      String backRelationDbName = backRelation.getPropertyDbName();
      Validate.isTrue(newRelations.stream().filter(item -> StringUtils.equals(item.getPropertyDbName(), backRelationDbName)).count() == 0
          , "进行动态模板初始化时，不可设定明细编辑表（%s）和主表的关联属性（%s）" , tableName , backRelationDbName);
      newRelations.add(backRelation);

      // 检测分组表下可能的明细编辑信息
      Set<TemplateItemEntity> templateItems = newGroup.getItemRelations();
      Set<TemplateItemEntity> oldItems = null;
      if(oldGroup != null) {
        oldItems = oldGroup.getItemRelations();
      }
      this.perfectingUpgradeOneToManyRelation(null, newGroup, tableName, templateItems, oldItems);
    }

    // 2、==========
    Set<String> oldGroupNames = oldGroups.stream().map(TemplateGroupEntity::getPropertyName).collect(Collectors.toSet());
    Set<String> newGroupNames = newGroups.stream().map(TemplateGroupEntity::getPropertyName).collect(Collectors.toSet());
    SetView<String> appendGroupNames = Sets.difference(newGroupNames, oldGroupNames);
    Set<TemplateGroupEntity> appendGroups = newGroups.stream().filter(item -> appendGroupNames.contains(item.getPropertyName())).collect(Collectors.toSet());
    for (TemplateGroupEntity templateGroup : appendGroups) {
      String tableName = templateGroup.getTableName();
      Validate.isTrue(!this.tableOperateRepositoryCustom.existTableName(tableName) , "检测到数据表'%s'已存在，请更换数据表名!!" , tableName);
    }
  }

  @Override
  public TemplateEntity findStructByTemplateId(String templateId) {
    //1 基础验证
    Validate.notBlank(templateId, "传入的模板ID不能为空，请检查!!");
    TemplateEntity template = templateService.findDetailsById(templateId);
    Validate.notNull(template, "根据模板ID未能获取到对应的模板信息，请检查!!");
    template.setId(null);
    template.setCversion(null);

    //2 过滤模板中id，formInstanceId以及回溯关系等，并且设置所有的id值为null。因为已经是拷贝数据，所以可以直接操作
    this.cleanProps(template.getProperties());
    this.cleanRelations(template.getRelations());
    this.cleanItems(template.getItemRelations());
    this.cleanGroups(template.getGroupRelations());

    return template;
  }

  /**
   * 清除明细下的id，formInstanceId，并且设置所有的id值为null
   * @param props
   */
  private void cleanProps(Set<TemplatePropertyEntity> props) {
    if(!CollectionUtils.isEmpty(props)) {
      props.removeIf(e -> {
        e.setId(null);
        return StringUtils.equalsIgnoreCase(e.getPropertyName(), "id") || StringUtils.equalsIgnoreCase(e.getPropertyName(), FORMINSTANCE_ID);
      });
    }
  }

  /**
   * 清除关联关系下的回溯关系等，并且设置所有的id值为null
   * @param relations
   */
  private void cleanRelations(Set<TemplateRelationEntity> relations) {
    if(!CollectionUtils.isEmpty(relations)) {
      relations.removeIf(e -> {
        e.setId(null);
        return e.getBackProperty();
      });
    }
  }

  /**
   * 清除明细下的id，formInstanceId以及回溯关系等，并且设置所有的id值为null
   * @param items
   */
  private void cleanItems(Set<TemplateItemEntity> items) {
    if(!CollectionUtils.isEmpty(items)) {
      items.stream().forEach(item -> {
        item.setId(null);
        //明细下的一般属性
        this.cleanProps(item.getProperties());
        //明细下的关联属性
        this.cleanRelations(item.getRelations());
      });
    }
  }

  /**
   * 清除分组下的id，formInstanceId以及回溯关系等，并且设置所有的id值为null
   * @param groups
   */
  private void cleanGroups(Set<TemplateGroupEntity> groups) {
    if(!CollectionUtils.isEmpty(groups)) {
      groups.stream().forEach(group -> {
        group.setId(null);
        //分组下的一般属性
        this.cleanProps(group.getProperties());
        //分组下的关联属性
        this.cleanRelations(group.getRelations());
        //分组下的明细属性
        this.cleanItems(group.getItemRelations());
      });
    }
  }


}