package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;
import com.bizunited.platform.kuiper.entity.InstanceActivityEntity;
import com.bizunited.platform.kuiper.entity.InstanceActivityLogEntity;
import com.bizunited.platform.kuiper.entity.InstanceEntity;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateGroupEntity;
import com.bizunited.platform.kuiper.entity.TemplateItemEntity;
import com.bizunited.platform.kuiper.entity.TemplatePropertyEntity;
import com.bizunited.platform.kuiper.entity.TemplateRelationEntity;
import com.bizunited.platform.kuiper.service.DynamicInstanceService;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.common.enums.FormTemplateTypeEnum;
import com.bizunited.platform.kuiper.starter.common.enums.RelationsTypeEnum;
import com.bizunited.platform.kuiper.starter.repository.InstanceActivityLogRepository;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService;
import com.bizunited.platform.kuiper.starter.service.InstanceActivityService;
import com.bizunited.platform.kuiper.starter.service.KuiperToolkitService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.IllegalFormatFlagsException;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 表单实例活动变化日志的服务层实现</br>
 * @author yinwenjie
 */
@Service("InstanceActivityLogServiceImpl")
public class InstanceActivityLogServiceImpl implements InstanceActivityLogService {
  @Autowired
  private TemplateService templateService;
  @Autowired
  private DynamicInstanceService dynamicInstanceService;
  @Autowired
  private InstanceActivityService instanceActivityService;
  @Autowired
  private InstanceActivityLogRepository instanceActivityLogRepository;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private ServicableMethodService servicableMethodService;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  @Qualifier("KuiperToolkitService")
  private KuiperToolkitService kuiperToolkitService;
  private static final Logger LOGGER = LoggerFactory.getLogger(InstanceActivityLogServiceImpl.class);
  private static final String PROPERTIES = "$['properties']";
  private static final String PROPERTY_NAME = "propertyName";
  private static final String RELATIONS = "$['relations']";
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#create(com.bizunited.platform.kuiper.entity.InstanceActivityLogEntity, com.alibaba.fastjson.JSONObject)
   */
  @Override
  @Transactional
  public InstanceActivityLogEntity create(InstanceActivityLogEntity instanceActivityLog, JSONObject log) {
    /*
     * 实际上添加过程就是将日志内容存储到磁盘上，将日志描述对象存储到数据库中
     * */
    Validate.notNull(instanceActivityLog , "进行保存操作时，日志描述对象必须传入!!");
    String id = instanceActivityLog.getId();
    Validate.isTrue(StringUtils.isBlank(id) , "进行保存操作是，日志描述对象的id编号不能有值");
    instanceActivityLog.setId(null);
    // 验证实例活动信息
    InstanceActivityEntity instanceActivity = instanceActivityLog.getInstanceActivity();
    Validate.notNull(instanceActivity , "进行保存时，关联的实例活动信息必须传入");
    String instanceActivityId = instanceActivity.getId();
    Validate.isTrue(StringUtils.isNotBlank(instanceActivityId) , "进行保存时，关联的实例活动编号信息必须传入!!");
    InstanceActivityEntity currentInstanceActivity = this.instanceActivityService.findById(instanceActivityId);
    Validate.notNull(currentInstanceActivity , "未找到当前活动日志对应的实例活动信息，请检查传参!!");
    // 验证操作者信息
    String creator = instanceActivityLog.getCreator();
    Validate.notBlank(creator , "进行保存操作时，必须传入创建者信息!!");
    
    // 首先保存日志文件内容
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String subSystem = "activityLog";
    String relativePath = StringUtils.join("/" , subSystem , "/" , folderName , "/" , (new Random().nextInt(100) % 10));
    // 写入文件
    String fileName = StringUtils.join(UUID.randomUUID().toString() , ".json");
    this.venusFileService.saveFile(relativePath, fileName, fileName, log.toJSONString().getBytes());
    
    // 然后保存日志描述到数据库
    instanceActivityLog.setCreateTime(new Date());
    instanceActivityLog.setRelativePath(relativePath);
    instanceActivityLog.setFileName(fileName);
    this.instanceActivityLogRepository.save(instanceActivityLog);
    return instanceActivityLog;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#findDetailsByInstanceActivity(java.lang.String)
   */
  @Override
  public Set<InstanceActivityLogEntity> findDetailsByInstanceActivity(String instanceActivityId) {
    if(StringUtils.isBlank(instanceActivityId)) {
      return Sets.newHashSet();
    }
    return this.instanceActivityLogRepository.findDetailsByInstanceActivity(instanceActivityId);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#findDetailsById(java.lang.String)
   */
  @Override
  public InstanceActivityLogEntity findDetailsById(String instanceActivityLogId) {
    if(StringUtils.isBlank(instanceActivityLogId)) {
      return null;
    }
    return this.instanceActivityLogRepository.findDetailsById(instanceActivityLogId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#findContentById(java.lang.String)
   */
  @Override
  public JSONObject findContentById(String instanceActivityLogId) {
    if(StringUtils.isBlank(instanceActivityLogId)) {
      return null;
    }
    InstanceActivityLogEntity currentLog = this.instanceActivityLogRepository.findDetailsById(instanceActivityLogId);
    if(currentLog == null) {
      return null;
    }
    
    // 查询文件内容
    String relativePath = currentLog.getRelativePath(); 
    String fileName = currentLog.getFileName();
    byte[] contents = this.venusFileService.readFileContent(relativePath, fileName);
    if(contents == null || contents.length == 0) {
      return null;
    }
    return JSON.parseObject(new String(contents));
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#findTopByInstanceId(java.lang.String)
   */
  @Override
  public String findTopByInstanceId(String instanceId) {
    if(StringUtils.isBlank(instanceId)) {
      return null;
    }
    return this.instanceActivityLogRepository.findTopByInstanceId(instanceId);
  }
  @Override
  @Transactional(rollbackOn = Exception.class)
  public void validateRecoveryByInstanceActivityLogId(String instanceActivityLogId) {
    try {
      this.recoveryByInstanceActivityLogId(instanceActivityLogId);
    } catch(RuntimeException e) {
      throw e;
    }
    
    throw new IllegalStateException();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#recoveryByInstanceActivityLogId(java.lang.String)
   */
  @Override
  @Transactional
  public JSONObject recoveryByInstanceActivityLogId(String instanceActivityLogId) {
    /*
     * 实际上数据恢复过程的主要过程是交给recoveryByStep私有方法以及它关联的私有方法在处理
     * 在本方法中，主要是做数据恢复前的准备工作：
     * 1、查询instanceActivityLogId对应的日志信息，准备对应的模板、实例、变化日志内容等信息
     * 2、根据数据日志中的明细查询服务源、完成代理调用，以便准备静态/动态 业务表单数据本身
     * 3、根据模板信息，得到那些白名单过滤的属性，以便将持久化对象转换为非持久化对象
     * 4、调用单步数据恢复的方法，进行数据实质性恢复。并且在恢复完成后，比较恢复前和恢复后的数据，重置其id
     * 至于为什么要重置其id主键，参见defineDiff方法的说明
     * 5、执行指定的数据更新方法，并删除多余的日志信息
     * */
    
    // 1、=======
    InstanceActivityLogEntity instanceActivityLog = this.findDetailsById(instanceActivityLogId);
    Validate.notNull(instanceActivityLog , "未找到对应的数据变化日志!!");
    InstanceActivityEntity instanceActivity = instanceActivityLog.getInstanceActivity();
    Validate.notNull(instanceActivity , "未发现实例活动信息，请检查!!");
    String instanceActivityId = instanceActivity.getId();
    Validate.notBlank(instanceActivityId , "未找到指定实例活动编号，请检查!!");
    InstanceEntity currentInstance = instanceActivity.getInstance();
    Validate.notNull(currentInstance , "未发现实例信息，请检查!!");
    String instanceId = currentInstance.getId();
    TemplateEntity currentTemplate = currentInstance.getTemplate();
    String templateType = currentTemplate.getType();
    Validate.isTrue(StringUtils.equals(templateType, FormTemplateTypeEnum.STATIC.getType()) || StringUtils.equals(templateType, FormTemplateTypeEnum.DYNAMIC.getType())  , "目前数据恢复功能只支持静态/动态表单实例,请确认!!");
    Validate.notNull(currentTemplate , "未发现关联的模板信息，请检查!!");
    String templateId = currentTemplate.getId();
    // 获取变化日志的内容
    JSONObject logJson = this.findContentById(instanceActivityLogId);
    Validate.notNull(logJson , "未找到变化日志内容，请检查文件系统!");
    // 这个变化日志必须是“最后一次”变更日志
    String newestLogId = this.instanceActivityLogRepository.findTopByInstanceId(instanceId);
    Validate.isTrue(StringUtils.equals(newestLogId, instanceActivityLogId) , "当前数据变化日志并不是本实例[%s]最近一次数据变动，不允许基于这个日志进行数据恢复!!" , instanceId);
    
    if(StringUtils.equals(templateType, FormTemplateTypeEnum.STATIC.getType())) {
      return this.recoveryByStaticInstanceActivityLogId(instanceId, currentTemplate, instanceActivityLogId, logJson);
    } else if(StringUtils.equals(templateType, FormTemplateTypeEnum.DYNAMIC.getType())) {
      return this.recoveryByDynamicInstanceActivityLogId(instanceId, templateId, instanceActivityLogId, logJson);
    } else {
      throw new IllegalFormatFlagsException("目前数据恢复功能只支持静态/动态表单实例，请确认!!");
    }
  }
  
  /**
   * 该私有方法针对静态表单实例进行日志恢复
   * @param instanceId
   * @param currentTemplate
   * @param instanceActivityLogId
   * @param logJson
   * @return
   */
  private JSONObject recoveryByStaticInstanceActivityLogId(String  instanceId , TemplateEntity currentTemplate , String instanceActivityLogId, JSONObject logJson) {
    Validate.notBlank(instanceActivityLogId , "数据变化日志编号必须传入!!");
    Validate.notBlank(instanceId , "静态实例编号必须传入!!");
    
    // 1、=======
    String persistentClassName = currentTemplate.getPersistentClassName();
    Validate.notBlank(persistentClassName , "未说明的模型类定义，请检查设定!!");
    InstanceActivityLogEntity instanceActivityLog = this.findDetailsById(instanceActivityLogId);
    Validate.notNull(instanceActivityLog , "未找到对应的数据变化日志!!");
    String templateId = currentTemplate.getId();
    /*
     * queryDetailsService的格式类似为“sXXXservice.queryDetailsByFormInstanceId”
     * 这个默认的名字肯定能通过servicableMethodService取得一个服务源信息，且这个服务源的入参只支持一个forminstanceId参数
     * 通过servicableMethodService取得的entity信息中的interfaceName属性，就可以从当前IOC容器中拿到需要的bean
     * */
    String queryDetailsService = instanceActivityLog.getQueryDetailsService();
    String updateDetailsService = instanceActivityLog.getUpdateDetailsService();
    Validate.notBlank(queryDetailsService , "未发现数据明细查询服务，请检查!!");
    Validate.notBlank(updateDetailsService , "未发现数据更新服务，请检查!!");
    String queryMethodName = null;
    String updateMethodName = null;
    try {
      queryMethodName = queryDetailsService.split("\\.")[1];
      updateMethodName = updateDetailsService.split("\\.")[1];
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("错误的服务名，请检查服务源配置!!");
    }
    
    // 2、=======
    // 调用指定的获取表单详情的服务源，得到详情对象
    ServicableMethodInfo servicableMethod = servicableMethodService.findDetailsByName(queryDetailsService);
    Validate.notNull(servicableMethod , "未找到对应的服务方法信息，请检查");
    String interfaceName = servicableMethod.getInterfaceName();
    Object details = this.invokeQueryDetails(interfaceName, queryMethodName, instanceId);
    Validate.notNull(details , "根据服务方法[%s.%s]，未查询到任务业务数据明细（基于表单实例[%s]），请检查设定!!" , interfaceName, queryMethodName, instanceId);
    Class<?> returnType = details.getClass();
    
    // 3、======
    // 进行对象过滤，保证其不受持久化规则影响。(白名单包括了这个模型的关联属性、明细属性、明细属性下的关联属性、分组属性、分组属性下的关联属性、分组属性下的明细属性、分组属性下的明细属性下的关联属性）
    // 这里可以获取当前模板下的所有直接和间接关联
    TemplateEntity template = this.templateService.findDetailsById(templateId);
    Validate.notNull(template, "未找到指定的静态模板信息[%s]，请检查!" , templateId);
    Set<String> wWhiteLists = this.kuiperToolkitService.buildAllWhiteList(template);
    String[] propertiesFilter = wWhiteLists.toArray(new String[]{});
    Object resouceObject = this.kuiperToolkitService.copyObjectByWhiteList(details, returnType, LinkedHashSet.class, ArrayList.class, propertiesFilter);
    JSONObject resouceJSONObject = (JSONObject)JSON.toJSON(resouceObject);
    JSONObject targetSONObject = (JSONObject)JSON.toJSON(resouceObject);
    
    // 4、======
    this.recoveryByStep(targetSONObject, logJson);
    this.defineDiff(template, resouceJSONObject, targetSONObject);
    
    // 5、=====调用指定的update方法，完成数据恢复过程
    // 首先是调用指定的update方法，会写业务数据
    if(StringUtils.equals(template.getType() , FormTemplateTypeEnum.STATIC.getType())) {
      this.invokeRecoveryStaticService(persistentClassName, template, targetSONObject, interfaceName, updateMethodName);
    } else if (StringUtils.equals(template.getType() , FormTemplateTypeEnum.DYNAMIC.getType())) {
      this.invokeRecoveryDynamicService(template, instanceId, targetSONObject);
    } else {
      throw new IllegalFormatFlagsException("数据恢复功能目前只支持静态模板和动态模板类型的表单");
    }
    
    // 删除当前已成功恢复的数据变化日志编号
    this.instanceActivityLogRepository.deleteById(instanceActivityLogId);
    return targetSONObject;
  }
  
  /**
   * 该私有方法针对动态表单实例进行日志恢复
   * @param instanceId
   * @param templateId
   * @param instanceActivityLogId
   * @param logJson
   * @return
   */
  private JSONObject recoveryByDynamicInstanceActivityLogId(String instanceId , String templateId , String instanceActivityLogId , JSONObject logJson) {
    Validate.notBlank(instanceActivityLogId , "数据变化日志编号必须传入!!");    
    // 2、=======
    // 调用指定的获取表单详情的服务源，得到详情对象
    JSONObject resouceJSONObject = this.dynamicInstanceService.findDetailsByFormInstanceId(instanceId);
    Validate.notNull(resouceJSONObject , "未查询到任务业务数据明细（基于表单实例[%s]），请检查设定!!", instanceId);
    
    // 3、======
    // 进行对象过滤，保证其不受持久化规则影响。(白名单包括了这个模型的关联属性、明细属性、明细属性下的关联属性、分组属性、分组属性下的关联属性、分组属性下的明细属性、分组属性下的明细属性下的关联属性）
    TemplateEntity template = this.templateService.findDetailsById(templateId);
    Validate.notNull(template, "未找到指定的静态模板信息[%s] ，请检查!!" , templateId);
    // 做一次深考
    String jsonConent = resouceJSONObject.toJSONString();
    JSONObject targetSONObject = JSON.parseObject(jsonConent);
    
    // 4、======
    this.recoveryByStep(targetSONObject, logJson);
    this.defineDiff(template, resouceJSONObject, targetSONObject);
    
    // 5、=====调用指定的update方法，完成数据恢复过程
    // 首先是调用指定的update方法，会写业务数据
    this.invokeRecoveryDynamicService(template, instanceId, targetSONObject);
    
    // 删除当前已成功恢复的数据变化日志编号
    this.instanceActivityLogRepository.deleteById(instanceActivityLogId);
    return targetSONObject;
  }
  
  /**
   * 该私有方法用于基于spring classloader，进行业务数据恢复服务的调用</br>
   * 该私有方法只支持静态表单模板（static）
   * @param persistentClassName
   * @param template
   * @param targetSONObject
   * @param interfaceName
   * @param updateMethodName
   */
  private void invokeRecoveryStaticService(String persistentClassName , TemplateEntity template , JSONObject targetSONObject , String interfaceName ,  String updateMethodName) {
    Class<?> currentPersistentClass = null;
    ClassLoader springClassLoader = applicationContext.getClassLoader();
    try {
      currentPersistentClass = springClassLoader.loadClass(persistentClassName);
    } catch (ClassNotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("未发现指定的静态模型类[" + persistentClassName + "]，请检查!!");
    }
    Object targetObject = null;
    targetObject = this.convertJSONToObject(targetSONObject , template, currentPersistentClass);
    Validate.notNull(targetObject , "convertJSONToObject内部方法出现错误，请启动调试模式");
    
    // 取得服务源（数据更新服务）
    Object bean = null;
    try {
      Class<?> interfaceClass = springClassLoader.loadClass(interfaceName);
      bean= applicationContext.getBean(interfaceClass);
    } catch(Exception e) {
      throw new IllegalArgumentException("在数据恢复时，未找到指定的服务注册[" + interfaceName + "]");
    }
    
    // 获取更新方法，并进行调用
    try {
      Method currentUpdateMethod = bean.getClass().getMethod(updateMethodName, currentPersistentClass);
      currentUpdateMethod.invoke(bean, targetObject);
    } catch(IllegalAccessException | IllegalArgumentException | NoSuchMethodException e) {
      throw new IllegalArgumentException("在数据恢复时，查找数据更新方法发生错误[" + updateMethodName + "]，很可能设定的方法已经不存在或者参数已改变（指定的方法规定只能穿入一种类型的参数[" + persistentClassName + "]），请检查!!");
    } catch(InvocationTargetException e) {
      Throwable targetThrowable = e.getTargetException();
      if(targetThrowable != null) {
        throw new RuntimeException("数据恢复时,出现错误：" + targetThrowable.getMessage() + "!如发现当前表单实例的id主键已经没有值，那么说明当前恢复操作已经是当前表单实例的最初步骤，不能再向前恢复!", targetThrowable);
      } else {
        throw new RuntimeException("数据恢复时，出现错误：" + e.getMessage() + "!如发现当前表单实例的id主键已经没有值，那么说明当前恢复操作已经是当前表单实例的最初步骤，不能再向前恢复!!", e);
      }
    } catch(Exception e) {
      throw new RuntimeException("数据恢复时，出现错误:" + e.getMessage() + "!如发现当前表单实例的id主键已经没有值，那么说明当前恢复操作已经是当前表单实例的最初步骤，不能再向前恢复!!", e);
    }
  }
  
  /**
   * 该私有方法用于基于spring classloader，进行业务数据恢复服务的调用</br>
   * 该私有方法只支持动态表单模板（dynamic）
   * @param template
   * @param targetSONObject
   */
  private void invokeRecoveryDynamicService(TemplateEntity template , String formInstanceId , JSONObject targetSONObject) {
    /*
     * 由于动态表单保存时存在一个自由的json结构定义，
     * 所以动态表单进行数据恢复的原则，就是通过targetSONObject构造前者所需要的json结构，过程为：
     * 1、TODO 相关结构差异的步骤还要确认
     * 2、构造完成后，直接调用DynamicInstanceService中的update方法进行更新
     * */
    
    targetSONObject.put("formInstanceId", formInstanceId);
    this.dynamicInstanceService.update(targetSONObject);
  }
  
  /**
   * 该私有方法用于基于spring classloader，进行业务数据明细查询
   * @param interfaceName 由服务源设定的服务接口名
   * @param queryMethodName 由服务源设定的业务明细查询的服务名
   * @param instanceId 当前的表单实例编号
   * @return
   */
  private Object invokeQueryDetails(String interfaceName , String queryMethodName , String instanceId) {
    Object bean = null;
    try {
      ClassLoader springClassLoader = applicationContext.getClassLoader();
      Class<?> interfaceClass = springClassLoader.loadClass(interfaceName);
      bean= applicationContext.getBean(interfaceClass);
    } catch(Exception e) {
      throw new IllegalArgumentException("在数据恢复时，未找到指定的服务注册[" + interfaceName + "]");
    }
    
    // 获取方法准备调用
    Object details = null;
    Class<?> returnType = null;
    try {
      Method currentMethod = bean.getClass().getMethod(queryMethodName, String.class);
      returnType = currentMethod.getReturnType();
      Validate.isTrue(returnType != null && returnType != Void.class , "方法[%s.%s]没有设定返回值，请检查!!" , interfaceName , queryMethodName);
      details = currentMethod.invoke(bean, instanceId);
    } catch(Exception e) {
      throw new IllegalArgumentException("在数据恢复时，查找数据明细发生错误[" + queryMethodName + "]，很可能设定的方法已经不存在，请检查!!");
    }
    
    return details;
  }
  
  /**
   * 手动进行JSON到Object的转换
   * @param dataJson 当前的类型信息
   * @param templateId 
   */
  private <T> T convertJSONToObject(JSONObject dataJson , TemplateEntity template , Class<T> currentPersistentClass) {
    T targetData = null;
    String className = currentPersistentClass.getName();
    try {
      targetData = currentPersistentClass.newInstance();
    } catch (Exception e) {
      LOGGER.error("类[{}]未能完成初始化，请检查类中是否保留了默认的构造函数!!", className);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    
    /*
     * 1、首先进行一般属性的转换，支持java 7种基础类型，以及BigDecimal，以及Date
     * 2、然后进行主模型下的关联属性转换。
     *  关联属性转换时，只需要关注id信息即可
     * 3、然后进行主模型下明细项的转换
     *  明细项包括了一般属性（可重用）
     *  还包括了关联属性转换（可重用）
     * 4、最后进行分组转换
     *  包括了一般属性转换（可重用）
     *  关联属性转换（可重用）
     *  明细属性转换（可重用）
     * */
    // 1、=====
    Set<TemplatePropertyEntity> templateProperties = template.getProperties();
    if(templateProperties != null && !templateProperties.isEmpty()) {
      this.convertJSONToObjectProperties(dataJson, targetData, templateProperties, currentPersistentClass);
    }
    
    Set<TemplateRelationEntity> relations = template.getRelations();
    // 2、=====
    // 关联信息转换时，只需要关注其中的id属性即可
    if(relations != null && !relations.isEmpty()) {
      this.convertJSONToObjectRelations(dataJson, targetData, relations, currentPersistentClass);
    }
    
    // 3、======
    Set<TemplateItemEntity> templateItems = template.getItemRelations();
    if(templateItems != null && !templateItems.isEmpty()) {
      this.convertJSONToObjectItems(dataJson, targetData, templateItems, currentPersistentClass);
    }
    
    // 4、======
    Set<TemplateGroupEntity> templateGroups = template.getGroupRelations();
    if(templateGroups != null && !templateGroups.isEmpty()) {
      this.convertJSONToObjectGroups(dataJson, targetData, templateGroups, currentPersistentClass);
    }
    return targetData;
  }
  
  /**
   * 完成json到object对象的分组属性转换
   */
  private void convertJSONToObjectGroups(JSONObject dataJson , Object targetData , Set<TemplateGroupEntity> templateGroups , Class<?> currentPersistentClass) {
    for (TemplateGroupEntity templateGroup : templateGroups) {
      String propertyName = templateGroup.getPropertyName();
      String propertyClassName = templateGroup.getPropertyClassName();
      Class<?> groupClass = null;
      Field propertyField = null;
      // 这是分组对象
      Object groupObject = null;
      ClassLoader springClassLoader = applicationContext.getClassLoader();
      try {
        groupClass = springClassLoader.loadClass(propertyClassName);
        groupObject = groupClass.newInstance();
        propertyField = this.findField(currentPersistentClass, propertyName);
        if(propertyField == null) {
          LOGGER.warn("未在指定的类[{}]找到属性字段[{}]，请检查!", propertyClassName, propertyName);
          continue;
        }
      } catch(Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
      JSONObject groupJsonData = (JSONObject)JSONPath.eval(dataJson, "$['" + propertyName + "']");
      if(groupJsonData == null) {
        continue;
      }
      
      // 首先是分组中的一般属性
      Set<TemplatePropertyEntity> templateGroupProperties = templateGroup.getProperties();
      if(templateGroupProperties != null && !templateGroupProperties.isEmpty()) {
        this.convertJSONToObjectProperties(groupJsonData, groupObject, templateGroupProperties, groupClass);
      }
      // 然后是分组中的关联
      Set<TemplateRelationEntity> templateGroupRelations = templateGroup.getRelations();
      if(templateGroupRelations != null && !templateGroupRelations.isEmpty()) {
        this.convertJSONToObjectRelations(groupJsonData, groupObject, templateGroupRelations, groupClass);
      }
      // 最后是分组中的明细属性
      Set<TemplateItemEntity> templateGroupItems = templateGroup.getItemRelations();
      if(templateGroupItems != null && !templateGroupItems.isEmpty()) {
        this.convertJSONToObjectItems(groupJsonData, groupObject, templateGroupItems, groupClass);
      }
      
      // 赋值到上级对象
      try {
        propertyField.setAccessible(true);
        propertyField.set(targetData, groupObject);
      } catch (Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
  
  }

  /**
   * 完成json到object对象的明细属性转换
   */
  private void convertJSONToObjectItems(JSONObject dataJson , Object targetData , Set<TemplateItemEntity> templateItems , Class<?> currentPersistentClass) {
    for (TemplateItemEntity templateItem : templateItems) {
      String propertyName = templateItem.getPropertyName();
      String propertyClassName = templateItem.getPropertyClassName();
      Class<?> itemClass = null;
      Field propertyField = null;
      ClassLoader springClassLoader = applicationContext.getClassLoader();
      try {
        itemClass = springClassLoader.loadClass(propertyClassName);
        propertyField = this.findField(currentPersistentClass, propertyName);
        if(propertyField == null) {
          LOGGER.warn("未在指定的类[{}]找到属性字段[{}]，请检查!!", propertyClassName, propertyName);
          continue;
        }
      } catch(Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
      
      // 确定当前明细集合是Set性质还是List性质
      boolean isSet = !List.class.isAssignableFrom(propertyField.getDeclaringClass());
      Collection<Object> collection = null;
      if(isSet) {
        collection = new LinkedHashSet<>();
      } else {
        collection = new LinkedList<>();
      }
      
      // 从Json Data中提取每一条明细记录(注意，如果没有信息，就不继续处理了)
      JSONArray itemJsonArrays = (JSONArray)JSONPath.eval(dataJson, "$['" + propertyName + "']");
      if(itemJsonArrays == null || itemJsonArrays.isEmpty()) {
        continue;
      }
      // 开始逐条进行object转换
      for(int index = 0 ; index < itemJsonArrays.size() ; index++) {
        JSONObject itemJsonData = itemJsonArrays.getJSONObject(index);
        Object itemObject = null;
        try {
          itemObject = itemClass.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
          throw new IllegalArgumentException(e.getMessage(), e);
        }
         
        // 首先是一般属性
        Set<TemplatePropertyEntity> itemProperties = templateItem.getProperties();
        if(itemProperties != null && !itemProperties.isEmpty()) {
          this.convertJSONToObjectProperties(itemJsonData, itemObject, itemProperties, itemClass);
        }
        // 然后是关联属性
        Set<TemplateRelationEntity> itemRelations = templateItem.getRelations();
        if(itemRelations != null && !itemRelations.isEmpty()) {
          this.convertJSONToObjectRelations(itemJsonData, itemObject, itemRelations, itemClass);
        }
        collection.add(itemObject);
      }
      
      // 赋值到上级对象
      try {
        propertyField.setAccessible(true);
        propertyField.set(targetData, collection);
      } catch (Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
  }
  
  /**
   * 完成json到object对象的关联属性转换（包括单选和多选）
   */
  private void convertJSONToObjectRelations(JSONObject dataJson , Object targetData , Set<TemplateRelationEntity> relations, Class<?> currentPersistentClass) {
    for (TemplateRelationEntity templateRelation : relations) {
      String propertyClassName = templateRelation.getPropertyClassName();
      String propertyName = templateRelation.getPropertyName();
      String relationType = templateRelation.getRelationType();
      Class<?> relationClass = null;
      Field relationIdField = null;
      Field propertyField = null;
      Object propertyValue = null;
      ClassLoader springClassLoader = applicationContext.getClassLoader();
      try {
        relationClass = springClassLoader.loadClass(propertyClassName);
      } catch(Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
      propertyField = this.findField(currentPersistentClass, propertyName);
      if(propertyField == null) {
        LOGGER.warn("未在指定的类[{}]找到属性字段[{}]，请检查!!", propertyClassName, propertyName);
        continue;
      }
      
      // 说明是多对多关联，那么需要生成集合
      if(StringUtils.equals(relationType, RelationsTypeEnum.MANY_TO_MANY.getRelation())) {
        Object valueObjects = dataJson.get(propertyName);
        // 如果不是一个json数组，说明值的情况有误，那么就跳过处理
        if(!(valueObjects instanceof JSONArray)) {
          continue;
        }
        JSONArray values = (JSONArray)valueObjects;
        
        // 判定当前关联属性是List性质还是Set性质
        boolean isSet =!List.class.isAssignableFrom(propertyField.getDeclaringClass());
        Collection<Object> collection = null;
        if(isSet) {
          collection = new LinkedHashSet<>();
        } else {
          collection = new LinkedList<>();
        }
        
        // 开始集合赋值，只需要关注它的id属性即可
        for(int index = 0 ; index < values.size() ; index++) {
          Object relationObject = null;
          String idValue = values.getString(index);
          try {
            relationIdField = this.findField(relationClass, "id");
            if(relationIdField == null) {
              LOGGER.warn("未在指定的类[{}]找到属性字段[id]，请检查!!", propertyClassName);
              continue;
            }
            relationObject = relationClass.newInstance();
            relationIdField.setAccessible(true);
            relationIdField.set(relationObject, idValue);
          } catch (Exception e) {
            throw new IllegalArgumentException(e.getMessage(), e);
          }
          collection.add(relationObject);
        }
        propertyValue = collection;
      } 
      // 否则就是单个关联，直接进行对象的id赋值即可
      else {
        JSONObject manyToOneObject = dataJson.getJSONObject(propertyName);
        if(manyToOneObject == null) {
          continue;
        }
        String idValue = manyToOneObject.getString("id");
        
        // 开始对象赋值，只需要关注它的id属性即可
        Object relationObject = null;
        try {
          relationClass = springClassLoader.loadClass(propertyClassName);
          relationIdField = this.findField(relationClass, "id");
          if(relationIdField == null) {
            LOGGER.warn("未在指定的类[{}]找到属性字段[id]，请检查!!", propertyClassName);
            continue;
          }
          relationObject = relationClass.newInstance();
          relationIdField.setAccessible(true);
          relationIdField.set(relationObject, idValue);
        } catch(Exception e) {
          throw new IllegalArgumentException(e.getMessage(), e);
        }
        propertyValue = relationObject;
      }
      
      // 为当前targetObject对象的propertyName属性赋值
      try {
        propertyField.setAccessible(true);
        propertyField.set(targetData, propertyValue);
      } catch (Exception e) {
        throw new IllegalArgumentException(e.getMessage(), e);
      }
    }
  }
  
  /**
   * 完成json到object对象的一般属性转换
   * @param dataJson 
   * @param targetData 
   * @param properties 
   * @param currentPersistentClass 
   */
  private void convertJSONToObjectProperties(JSONObject dataJson , Object targetData , Set<TemplatePropertyEntity> properties , Class<?> currentPersistentClass) {
    if(properties == null || properties.isEmpty()) {
      return;
    }
    String className = currentPersistentClass.getName();
    
    for (TemplatePropertyEntity propertyItem : properties) {
      String propertyName = propertyItem.getPropertyName();
      String propertyClassName = propertyItem.getPropertyClassName();
      // 寻找到对象中的字段
      Field field = this.findField(currentPersistentClass, propertyName);
      if(field == null) {
        LOGGER.warn("未在类[{}]以及其父类中找到属性[{}]，请检查类定义.", className, propertyName);
        continue;
      }
      
      // 开始赋值
      try {
        field.setAccessible(true);
        if(StringUtils.equals(propertyClassName, "java.lang.String")) {
          String stringValue = dataJson.getString(propertyName);
          if(stringValue != null) {
            field.set(targetData, stringValue);
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Integer")) {
          Integer intValue = dataJson.getInteger(propertyName);
          if(intValue != null) {
            field.set(targetData, intValue);
          }
        } else if(StringUtils.equals(propertyClassName, "int")) {
          Integer intValue = dataJson.getInteger(propertyName);
          if(intValue != null) {
            field.setInt(targetData, intValue.intValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Short")) {
          Short shortValue = dataJson.getShort(propertyName);
          if(shortValue != null) {
            field.set(targetData, shortValue);
          }
        } else if(StringUtils.equals(propertyClassName, "short")) {
          Short shortValue = dataJson.getShort(propertyName);
          if(shortValue != null) {
            field.setShort(targetData, shortValue.shortValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Byte")) {
          Byte byteValue = dataJson.getByte(propertyName);
          if(byteValue != null) {
            field.set(targetData, byteValue);
          }
        }  else if(StringUtils.equals(propertyClassName, "byte")) {
          Byte byteValue = dataJson.getByte(propertyName);
          if(byteValue != null) {
            field.setByte(targetData, byteValue.byteValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Long")) {
          Long longValue = dataJson.getLong(propertyName);
          if(longValue != null) {
            field.set(targetData, longValue);
          }
        }  else if(StringUtils.equals(propertyClassName, "long")) {
          Long longValue = dataJson.getLong(propertyName);
          if(longValue != null) {
            field.setLong(targetData, longValue.longValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Float")) {
          Float floatValue = dataJson.getFloat(propertyName);
          if(floatValue != null) {
            field.set(targetData, floatValue);
          }
        } else if(StringUtils.equals(propertyClassName, "float")) {
          Float floatValue = dataJson.getFloat(propertyName);
          if(floatValue != null) {
            field.setFloat(targetData, floatValue.floatValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Double")) {
          Double doubletValue = dataJson.getDouble(propertyName);
          if(doubletValue != null) {
            field.set(targetData, doubletValue);
          }
        }  else if(StringUtils.equals(propertyClassName, "double")) {
          Double doubletValue = dataJson.getDouble(propertyName);
          if(doubletValue != null) {
            field.setDouble(targetData, doubletValue.doubleValue());
          }
        } else if(StringUtils.equals(propertyClassName, "java.util.Date")) {
          Date dateValue = dataJson.getDate(propertyName);
          if(dateValue != null) {
            field.set(targetData, dateValue);
          }
        } else if(StringUtils.equals(propertyClassName, "java.math.BigDecimal")) {
          BigDecimal bigValue = dataJson.getBigDecimal(propertyName);
          if(bigValue != null) {
            field.set(targetData, bigValue);
          }
        } else if(StringUtils.equals(propertyClassName, "java.lang.Boolean")) {
          Boolean booleanValue = dataJson.getBoolean(propertyName);
          if(booleanValue != null) {
            field.set(targetData, booleanValue);
          }
        } else if(StringUtils.equals(propertyClassName, "boolean")) {
          Boolean booleanValue = dataJson.getBoolean(propertyName);
          if(booleanValue != null) {
            field.setBoolean(targetData, booleanValue.booleanValue());
          }
        }
      } catch(IllegalAccessException e) {
        LOGGER.warn("类[{}]中属性[{}]在赋值时发现类型错误，请检查类定义.", className, propertyName);
        continue;
      }
    }
  }
  
  /**
   * 在当前类以及其父类中，寻找相关属性定义
   * @param currentClass
   * @param fieldName
   * @return
   */
  private Field findField(Class<?> currentClass , String fieldName) {
    Field currentField = null;
    try {
      currentField = currentClass.getDeclaredField(fieldName);
    } catch(Exception e) {
      // 没找到就算了
    }
    if(currentField != null) {
      return currentField;
    }
    
    // 查找父类
    Class<?> parentClass = currentClass.getSuperclass();
    if(parentClass != null) {
      return this.findField(parentClass, fieldName);
    }
    return null;
  }
  
  /**
   * 由于update方法默认的明细规则，这里我们需要判定resouceJSONObject和targetSONObject
   * 中各个明细编辑项的差异，并将targetSONObject中多处的各个明细项的ID置为null
   * 这样在update方法运行是，才会真正在数据库中新增（恢复）这些数据差异
   * */
  private void defineDiff(TemplateEntity template , JSONObject resouceJSONObject , JSONObject targetSONObject) {
    Validate.notNull(template , "进行差异清理时，模板信息必须传入！！");
    if(resouceJSONObject == null) {
      resouceJSONObject = new JSONObject();
    }
    if(targetSONObject == null) {
      targetSONObject = new JSONObject();
    }
    
    // 4.1、=====首先是主模型下的明细
    Set<TemplateItemEntity> templateItems = template.getItemRelations();
    if(templateItems != null && !templateItems.isEmpty()) {
      this.handleItemIds(templateItems, resouceJSONObject, targetSONObject);
    }
    // 4.2、=====然后是分组下的明细
    Set<TemplateGroupEntity> templateGroups = template.getGroupRelations();
    if(templateGroups != null && !templateGroups.isEmpty()) {
      for (TemplateGroupEntity templateGroup : templateGroups) {
        String groupName = templateGroup.getPropertyName();
        JSONObject resouceGroupJSONObject = resouceJSONObject.getJSONObject(groupName);
        JSONObject targetGroupJSONObject = targetSONObject.getJSONObject(groupName);
        
        Set<TemplateItemEntity> groupItems = templateGroup.getItemRelations();
        if(groupItems != null && !groupItems.isEmpty()) {
          this.handleItemIds(groupItems, resouceGroupJSONObject, targetGroupJSONObject);
        }
      }
    }
  }
  
  /**
   * 请参见调用其的主方法中的注释说明
   * @param templateItems 
   * @param resouceJSONObject 
   * @param targetSONObject 
   */
  private void handleItemIds(Set<TemplateItemEntity> templateItems , JSONObject resouceJSONObject , JSONObject targetSONObject) {
    for (TemplateItemEntity templateItem : templateItems) {
      String templateItemName = templateItem.getPropertyName();
      JSONArray resouceItem = resouceJSONObject.getJSONArray(templateItemName);
      if(resouceItem == null) {
        resouceItem = new JSONArray();
      }
      JSONArray targetItem = targetSONObject.getJSONArray(templateItemName);
      if(targetItem == null) {
        targetItem = new JSONArray();
      }
      
      // 基于targetItem进行查找
      JSONArray existIds = (JSONArray)JSONPath.eval(resouceItem, "$['id']");
      EP:for(int index = 0 ; index < targetItem.size() ; index++) {
        JSONObject targetItemObject = targetItem.getJSONObject(index);
        String targetId = targetItemObject.getString("id");
        for(int idIndex = 0 ; idIndex < existIds.size() ; idIndex++) {
          String existId = existIds.getString(idIndex);
          // 如果条件成立，说明这条明细项无需新增
          if(StringUtils.equals(targetId, existId)) {
            continue EP;
          }
        }
        targetItemObject.remove("id");
      }
    }
  }
  
  /**
   * 数据单步恢复，也就是说将dataJson中的数据json描述形式，根据logJson中的变化情况，进行恢复。<br>
   * 其中logJson的格式定义请参见技术文档中关于“表单数据变化追踪结构.json”的描述定义
   * @param dataJson 数据的json格式表达
   * @param logJson 变化内容的json格式表达
   * @return 恢复后数据的json格式将被返回
   */
  private void recoveryByStep(JSONObject dataJson , JSONObject logJson) {
    Validate.notNull(dataJson , "进行回复时，数据的json格式表达不能为null");
    Validate.notNull(logJson , "进行回复时，变化日志的json格式表达不能为null");
    
    /*
     * 单步恢复过程为：
     * 1、首先根据logJson中对一般属性变化的描述，进行一般属性信息的恢复
     * 2、然后根据logJson中对关联属性变化的描述，进行关联属性信息的恢复（只需要恢复关联的主键id信息）
     *   其中又包括对单选和多选信息的恢复
     * 3、接着，根据logJson中对明细信息的描述，进行明细信息的恢复
     *   其中新增的要进行删除
     *   原删除的要进行新增
     *   有修改操作的要原值恢复，其中包括了一般属性的恢复、关联属性的恢复（可重用）
     * 4、最后根据logJson中对分组信息的描述，进行分组信息的恢复
     *   4.1、对分组中基本信息进行恢复（可重用）
     *   4.2、对分组中关联信息进行恢复（可重用）
     *   4.3、对分组中明细信息进行恢复（可重用）
     * */
    
    // 1、======
    JSONArray propertiesLogJson = (JSONArray)JSONPath.eval(logJson, PROPERTIES);
    if(propertiesLogJson != null && !propertiesLogJson.isEmpty()) {
      this.recoveryProperties(dataJson, propertiesLogJson);
    }
    
    // 2、=====
    JSONArray relationsLogJson = (JSONArray)JSONPath.eval(logJson, RELATIONS);
    if(relationsLogJson != null && !relationsLogJson.isEmpty()) {
      this.recoveryRelations(dataJson, relationsLogJson);
    }
    
    // 3、=====
    JSONArray itemsLogJsons = (JSONArray)JSONPath.eval(logJson, "$['items']");
    for(int index = 0 ; itemsLogJsons != null && index < itemsLogJsons.size() ; index++) {
      // 取得明细变化的日志情况
      JSONObject itemsLogJson = itemsLogJsons.getJSONObject(index);
      String propertyName = itemsLogJson.getString(PROPERTY_NAME);
      JSONArray itemLogs = itemsLogJson.getJSONArray("datas");
      // 取得当前的数据情况
      JSONArray dataArrays = dataJson.getJSONArray(propertyName);
      if(dataArrays == null) {
        dataArrays = new JSONArray();
      }
      JSONArray results = this.recoveryItems(propertyName, dataArrays, itemLogs);
      dataJson.put(propertyName, results);
    }
    
    // 4、=====
    JSONArray groupsLogJsons = (JSONArray)JSONPath.eval(logJson, "$['groups']");
    for(int index = 0 ; groupsLogJsons != null && index < groupsLogJsons.size() ; index++) {
      JSONObject groupsLogJson = groupsLogJsons.getJSONObject(index);
      String propertyName = groupsLogJson.getString(PROPERTY_NAME);
      JSONObject propertyDataJson = dataJson.getJSONObject(propertyName);
      Validate.notNull(propertyDataJson , "在数据恢复时，未发现分组[%s]的原始数据，请检查!!" , propertyName);
      this.recoveryByStep(propertyDataJson, groupsLogJson);
    }
  }
  
  /**
   * 该私有方法对一个明细属性中的若干数据明细项，基于变化日志进行恢复
   * @param datas 当前
   * @param itemLogs
   * @param 恢复后的当前明细属性的新的数据明细项，将被返回
   */
  private JSONArray recoveryItems(String propertyName , JSONArray datas , JSONArray itemLogs) {
    /*
     * 每一条明细项都要进行处理，处理过程为：
     * 1、首先找到日志中新增的那些数据项，这些数据项将基于目前的id属性，从datas中进行删除
     * 2、然后找到日志中删除的那些数据项，这些数据项将基于其id属性，从data中进行新增
     * 3、最后找到日志中修改的那些数据项，这些数据项将基于其id属性，恢复其一般属性和关联属性（方法可重用）
     * */
    if(itemLogs == null || itemLogs.isEmpty()) {
      return datas;
    }
    
    // 1、=======
    // 得到要删除的数据
    JSONArray deleteOpLogs = (JSONArray)JSONPath.eval(itemLogs, "$[?(@.changeType = 'NEW')]");
    if(deleteOpLogs != null && !deleteOpLogs.isEmpty()) {
      Validate.isTrue(datas != null && !datas.isEmpty() , "在恢复操作时，发现数据中的明细编辑项[%s]原始数据和操作日志不匹配，请检查!!" , propertyName);
    }
    DP:for(int index = 0 ; deleteOpLogs != null && index < deleteOpLogs.size() ; index++) {
      JSONObject deleteOpLog = deleteOpLogs.getJSONObject(index);
      String idLog = deleteOpLog.getString("id");
      for(int itemIndex = 0 ; itemIndex < datas.size() ; itemIndex++) {
        JSONObject itemData = datas.getJSONObject(itemIndex);
        String currentId = itemData.getString("id");
        Validate.notBlank(currentId , "在恢复操作时，明细编辑项[%s]原始数据[%s]未找到，请检查!!" , propertyName , idLog);
        // 如果条件成立，就是这条记录需要删除
        if(StringUtils.equals(idLog, currentId)) {
          datas.remove(itemIndex);
          continue DP;
        }
      }
    }
    
    // 2、=======
    // 得到要新增的数据
    JSONArray newOpLogs = (JSONArray)JSONPath.eval(itemLogs, "$[?(@.changeType = 'DELETE')]");
    for(int index = 0 ; newOpLogs != null && index < newOpLogs.size() ; index++) {
      JSONObject newOpLog = newOpLogs.getJSONObject(index);
      JSONObject newData = new JSONObject();
      // 首先新增一般属性
      JSONArray propertiesLogs = (JSONArray)JSONPath.eval(newOpLog, PROPERTIES);
      if(propertiesLogs != null && !propertiesLogs.isEmpty()) {
        this.recoveryProperties(newData, propertiesLogs);
      }
      // 然后新增关联属性
      JSONArray relationsLogs = (JSONArray)JSONPath.eval(newOpLog, RELATIONS);
      if(relationsLogs != null && !relationsLogs.isEmpty()) {
        this.recoveryRelations(newData, relationsLogs);
      }
      // 赋值到数组
      datas.add(newData);
    }
    
    // 3、=====
    // 得到要修改的数据
    JSONArray updateOpLogs = (JSONArray)JSONPath.eval(itemLogs, "$[?(@.changeType = 'UPDATE')]");
    NP:for(int index = 0 ; updateOpLogs != null && index < updateOpLogs.size() ; index++) {
      JSONObject updateOpLog = updateOpLogs.getJSONObject(index);
      String idLog = updateOpLog.getString("id");
      for(int itemIndex = 0 ; itemIndex < datas.size() ; itemIndex++) {
        JSONObject itemData = datas.getJSONObject(itemIndex);
        String currentId = itemData.getString("id");
        Validate.notBlank(currentId , "在恢复操作时，明细编辑项[%s]原始数据[%s]未找到，请检查!!" , propertyName , idLog);
        // 如果条件成立，就是这条记录需要修改
        if(StringUtils.equals(idLog, currentId)) {
          // 首先恢复一般数据
          JSONArray propertiesLogs = (JSONArray)JSONPath.eval(updateOpLog, PROPERTIES);
          if(propertiesLogs != null && !propertiesLogs.isEmpty()) {
            this.recoveryProperties(itemData, propertiesLogs);
          }
          // 然后恢复关联数据
          JSONArray relationsLogs = (JSONArray)JSONPath.eval(updateOpLog, RELATIONS);
          if(relationsLogs != null && !relationsLogs.isEmpty()) {
            this.recoveryRelations(itemData, relationsLogs);
          }
          continue NP;
        }
      }
    }
    
    return datas;
  }
  
  /**
   * 该方法对一组一般属性，基于变化日志进行恢复
   * @param dataJson 整体的数据信息的json格式表达 
   * @param propertiesLogJson 一组一般属性的变化日志 
   */
  private void recoveryProperties(JSONObject dataJson , JSONArray propertiesLogJson) {
    for(int index = 0 ; propertiesLogJson != null && index < propertiesLogJson.size() ; index++) {
      JSONObject propertyLogJson = propertiesLogJson.getJSONObject(index);
      Object beforeValue = propertyLogJson.get("beforeValue");
      String propertyName = propertyLogJson.getString(PROPERTY_NAME);
      
      // 如果没有取到任何前置，说明这个属性变化前为null
      if(beforeValue == null) {
        dataJson.put(propertyName, null);
      } else {
        dataJson.put(propertyName, beforeValue);
      }
    }
  }
  
  /**
   * 该方法对一组关联属性，基于变化日志进行恢复
   * @param dataJson 整体的数据信息的json格式表达
   * @param relationsLogJson 一组关联属性的变化日志 
   */
  private void recoveryRelations(JSONObject dataJson , JSONArray relationsLogJson) {
    for(int index = 0 ; relationsLogJson != null && index < relationsLogJson.size() ; index++) {
      JSONObject relationLogJsonItem = relationsLogJson.getJSONObject(index);
      Object beforeValue = relationLogJsonItem.get("beforeValue");
      String propertyName = relationLogJsonItem.getString(PROPERTY_NAME);
      
      // 如果条件成立，说明是ManyToMany性质的选择，这里是一个数组对象
      if(beforeValue instanceof JSONArray) {
        JSONArray beforeValues = (JSONArray)beforeValue;
        JSONArray manyToMany = new JSONArray();
        for (int itemIndex = 0 ; itemIndex < beforeValues.size() ; itemIndex++) {
          JSONObject idObject = new JSONObject();
          idObject.put("id", beforeValues.get(itemIndex));
          manyToMany.add(idObject);
        }
        dataJson.put(propertyName, manyToMany);
      } 
      // 否则就只是一个ManyToOne性质的选择
      else {
        JSONObject manyToOne = new JSONObject();
        manyToOne.put("id", beforeValue);
        dataJson.put(propertyName, manyToOne);
      }
    }
  }
  @Override
  @Transactional(rollbackOn = Exception.class)
  public void validateRecoveryByInstanceActivityId(String targetInstanceActivityId) throws Exception {
    this.recoveryByInstanceActivityId(targetInstanceActivityId);
    throw new Exception();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.InstanceActivityLogService#recoveryByInstanceActivityId(java.lang.String)
   */
  @Override
  @Transactional
  public JSONObject recoveryByInstanceActivityId(String targetInstanceActivityId) {
    /*
     * 进行单步回退的工作，实际上就是寻找以下两个关键点
     * a、寻找开始进行回退的点（这个很简单，只需要调用数据层的相关接口就可以找到）
     * b、寻找回退的结束点（就是说要回退到哪个点截止【但是这个点的回退操作不用做】）
     * .也就是说当前instanceActivityId前的所有活动变化都将被取消，在其中发生的交叉变化也将会被取消
     * 
     * .基于以上的重要特性，整个处理过程如下
     * 1、首先基于targetInstanceActivityId获得哪些活动（instanceActivity）将会被取消；
     * .原则是当前instanceActivityId创建后的所有instanceActivityId都将参与数据回退过程；且后者其下的所有操作动作都将被取消
     * 5月27日补充：注意：如果当前instanceActivityId下没有任何变化日志，则目标instanceActivityId按照时间顺序向前移动一个
     * .直到移动到create那个活动上，如果create那个活动上都没有任何的变化日志，则系统报错
     * 2、构建K-V容器，找到那些recoveryActivities中的活动涉及到的所有活动日志
     * .这里的key是活动id，value是一个当前活动下按照时间顺序排列的变化日志的集合(日志id)
     * 3、求得哪些日志记录，在本次数据恢复操作中，需要被恢复
     * .原则是，当recoveryActivityLogMapping中除去targetInstanceActivityId以外的所有活动log都已经被剔除
     * .则当前recoveryActivityLogMapping集合中targetInstanceActivityId为key的集合的最前面一个logid，就是本次恢复操作的日志终点（不包括这个点）
     * 4、开始进行恢复
     * 5、完成以上操作后，回退链X针对集合A的差集将进行删除（真删除），业务数据的差异将进行数据库正式更新
     * */
    
    // 1、=======
    Validate.notBlank(targetInstanceActivityId , "进行批量数据恢复时，请指定目标活动编号!!");
    InstanceActivityEntity currentInstanceActivity = this.instanceActivityService.findById(targetInstanceActivityId);
    Validate.notNull(currentInstanceActivity , "当前指定的目标活动未找到，请检查!!");
    // 当前表单实例
    InstanceEntity currentInstance = currentInstanceActivity.getInstance();
    String currentInstanceId = currentInstance.getId();
    // 当前的模板信息
    TemplateEntity template = currentInstance.getTemplate();
    String type = template.getType();
    Validate.isTrue(StringUtils.equals(type, FormTemplateTypeEnum.STATIC.getType()) || StringUtils.equals(type, FormTemplateTypeEnum.DYNAMIC.getType()) , "目前批量数据恢复功能仅支持静态/动态模型，请检查当前表单实例的模板类型!!");
    
    if(StringUtils.equals(type, FormTemplateTypeEnum.STATIC.getType())) {
      return this.recoveryByStaticInstanceActivityId(template, currentInstanceId, targetInstanceActivityId);
    } else if(StringUtils.equals(type, FormTemplateTypeEnum.DYNAMIC.getType())) {
      return this.recoveryByDynamicInstanceActivityId(template, currentInstanceId, targetInstanceActivityId);
    } else {
      throw new IllegalFormatFlagsException("目前数据恢复功能只支持静态/动态表单实例，请确认!!");
    }
  }
  
  /**
   * @param template
   * @param currentInstanceId
   * @param targetInstanceActivityId
   * @return
   */
  private JSONObject recoveryByStaticInstanceActivityId(TemplateEntity template , String currentInstanceId ,  String targetInstanceActivityId) {
    String persistentClassName = template.getPersistentClassName();
    Validate.notBlank(persistentClassName , "未发现指定的静态模型类名，请检查!!");
    Set<InstanceActivityEntity> instanceActivities = this.instanceActivityService.findByInstanceId(currentInstanceId);
    Validate.isTrue(instanceActivities != null && !instanceActivities.isEmpty() , "未在指定实例下找到任何活动信息，请检查!!");
    String templateId = template.getId();
    List<InstanceActivityLogEntity> recoveryLogs = this.analysisActivityLogs(currentInstanceId, instanceActivities, targetInstanceActivityId);
    
    // 4、===========
    Class<?> returnType = null;
    Object details = null;
    String interfaceName = null;
    String queryMethodName = null;
    String updateMethodName = null;
    Validate.notNull(recoveryLogs , "没有发现任何待恢复的日志信息，请联系运维人员上报bug!!");
    template = this.templateService.findDetailsById(templateId);
    Validate.notNull(template, "未找到指定的静态模板信息[%s]，请检查!!" , templateId);
    JSONObject resouceJSONObject = null;
    JSONObject targetSONObject = new JSONObject();
    for(int index = 0 ; index < recoveryLogs.size() ; index++) {
      InstanceActivityLogEntity currentLog = recoveryLogs.get(index);
      String currentLogId = currentLog.getId();
      // 如果条件成立，说明是开始节点，那么要调用invokeQueryDetails，找到最初的业务详情数据
      if(index == 0) {
        String queryDetailsService = currentLog.getQueryDetailsService();
        String updateDetailsService = currentLog.getUpdateDetailsService();
        ServicableMethodInfo servicableMethod = servicableMethodService.findDetailsByName(queryDetailsService);
        Validate.notNull(servicableMethod , "未找到对应的服务方法信息，请检查");
        interfaceName = servicableMethod.getInterfaceName();
        try {
          queryMethodName = queryDetailsService.split("\\.")[1];
          updateMethodName = updateDetailsService.split("\\.")[1];
        } catch(Exception e) {
          LOGGER.error(e.getMessage() , e);
          throw new IllegalArgumentException("错误的服务名，请检查服务源配置!!");
        }
        details = this.invokeQueryDetails(interfaceName, queryMethodName, currentInstanceId);
        Validate.notNull(details , "未找到当前表单实例的任何详细信息，请检查服务源接口!!");
        returnType = details.getClass();
        Validate.notNull(details , "根据服务方法[%s.%s]，未查询到任务业务数据明细（基于表单实例[%s]），请检查设定!!" , interfaceName, queryMethodName, currentInstanceId);
        Set<String> wWhiteLists = this.kuiperToolkitService.buildAllWhiteList(template);
        String[] propertiesFilter = wWhiteLists.toArray(new String[]{});
        Object resouceObject = this.kuiperToolkitService.copyObjectByWhiteList(details, returnType, LinkedHashSet.class, ArrayList.class, propertiesFilter);
        resouceJSONObject = (JSONObject)JSON.toJSON(resouceObject);
        targetSONObject = (JSONObject)JSON.toJSON(resouceObject);
      }
      
      // 读取当前的变更日志内容，并开始恢复
      JSONObject logJSONData = this.findContentById(currentLogId);
      Validate.notNull(logJSONData , "未找到变化日志内容，请检查文件系统!!");
      this.recoveryByStep(targetSONObject, logJSONData);
    }
    this.defineDiff(template, resouceJSONObject, targetSONObject);
    
    // 5、=========
    // 调用指定的更新方法
    if(StringUtils.equals(template.getType() , FormTemplateTypeEnum.STATIC.getType())) {
      this.invokeRecoveryStaticService(persistentClassName, template, targetSONObject, interfaceName, updateMethodName);
    } else if (StringUtils.equals(template.getType() , FormTemplateTypeEnum.DYNAMIC.getType())) {
      this.invokeRecoveryDynamicService(template, currentInstanceId, targetSONObject);
    } else {
      throw new IllegalFormatFlagsException("数据批量恢复功能目前只支持静态模板和动态模板类型的表单");
    }
    // 删除recoveryLogs集合中所有已被成功恢复的数据变化日志
    for (InstanceActivityLogEntity recoveryLog : recoveryLogs) {
      this.instanceActivityLogRepository.deleteById(recoveryLog.getId());
    }
    return targetSONObject;
  }
  
  /**
   * @param template
   * @param currentInstanceId
   * @param targetInstanceActivityId
   * @return
   */
  private JSONObject recoveryByDynamicInstanceActivityId(TemplateEntity template ,  String currentInstanceId , String targetInstanceActivityId) {
    Set<InstanceActivityEntity> instanceActivities = this.instanceActivityService.findByInstanceId(currentInstanceId);
    Validate.isTrue(instanceActivities != null && !instanceActivities.isEmpty() , "未在指定实例下找到任何活动信息，请检查!!");
    String templateId = template.getId();
    List<InstanceActivityLogEntity> recoveryLogs = this.analysisActivityLogs(currentInstanceId, instanceActivities, targetInstanceActivityId);
    
    // 4、===========
    Validate.notNull(recoveryLogs , "没有发现任何待恢复的日志信息，请联系运维人员上报bug!!");
    template = this.templateService.findDetailsById(templateId);
    Validate.notNull(template, "未找到指定的静态模板信息[%s]，请检查!!" , templateId);
    JSONObject resouceJSONObject = null;
    JSONObject targetSONObject = new JSONObject();
    for(int index = 0 ; index < recoveryLogs.size() ; index++) {
      InstanceActivityLogEntity currentLog = recoveryLogs.get(index);
      String currentLogId = currentLog.getId();
      // 如果条件成立，说明是开始节点，那么要调用invokeQueryDetails，找到最初的业务详情数据
      if(index == 0) {
        resouceJSONObject = this.dynamicInstanceService.findDetailsByFormInstanceId(currentInstanceId);
        Validate.notNull(resouceJSONObject , "未找到当前表单实例的任何详细信息，请检查服务源接口!!");
        // 做一次深考
        String jsonConent = resouceJSONObject.toJSONString();
        targetSONObject = JSON.parseObject(jsonConent);
      }
      
      // 读取当前的变更日志内容，并开始恢复
      JSONObject logJSONData = this.findContentById(currentLogId);
      Validate.notNull(logJSONData , "未找到变化日志内容，请检查文件系统!!");
      this.recoveryByStep(targetSONObject, logJSONData);
    }
    this.defineDiff(template, resouceJSONObject, targetSONObject);
    
    // 5、=========
    // 调用指定的更新方法
    this.invokeRecoveryDynamicService(template, currentInstanceId, targetSONObject);
    // 删除recoveryLogs集合中所有已被成功恢复的数据变化日志
    for (InstanceActivityLogEntity recoveryLog : recoveryLogs) {
      this.instanceActivityLogRepository.deleteById(recoveryLog.getId());
    }
    return targetSONObject;
  }

  /**
   * @param currentInstanceId
   * @param instanceActivities
   * @param targetInstanceActivityId
   * @return
   */
  private List<InstanceActivityLogEntity> analysisActivityLogs(String currentInstanceId , Set<InstanceActivityEntity> instanceActivities,  String targetInstanceActivityId) {
    // 5月27日增加的处理过程：如果已经寻找到了currnetTargetInstanceActivityId的位置，但这个位置又没有值
    // 则currnetTargetInstanceActivityId的定位向后移动
    Iterator<InstanceActivityEntity> instanceActivityIterator = instanceActivities.iterator();
    String currnetTargetInstanceActivityId = targetInstanceActivityId;
    FIND:while (instanceActivityIterator.hasNext()) {
      InstanceActivityEntity instanceActivityItem = instanceActivityIterator.next();
      String instanceActivityItemId = instanceActivityItem.getId();
      
      // 如果条件成立，说明还没有找到targetInstanceActivityId那个活动点
      if(!StringUtils.equals(instanceActivityItemId, targetInstanceActivityId)) {
        continue;
      } 
      else {
        while(true) {
          Set<InstanceActivityLogEntity> logs = this.instanceActivityLogRepository.findByInstanceActivity(currnetTargetInstanceActivityId);
          // 如果条件成立，则说明当前目标回退点存在至少一个操作日志，满足回退要求
          if(logs != null && !logs.isEmpty()) {
            break FIND;
          } 
          // 如果条件成立，说明还可以找其前置活动点
          else if((logs == null || logs.isEmpty()) && instanceActivityIterator.hasNext()) {
            currnetTargetInstanceActivityId = instanceActivityIterator.next().getId();
            continue;
          } 
          // 如果条件成立，说明已经是最前面一个活动点了
          else if((logs == null || logs.isEmpty()) && !instanceActivityIterator.hasNext()) {
            throw new IllegalArgumentException("未找到任何可用的目标活动日志，因为指定的数据恢复点已经没有任何活动日志，不能再进行回退了!!");
          } 
        } 
      } 
    }
    // 重新确认最终的活动退回点currnetTargetInstanceActivityId后，开始框定需要参与回退的所有活动
    List<String> recoveryActivityIds = new LinkedList<>();
    for (InstanceActivityEntity instanceActivityItem : instanceActivities) {
      String instanceActivityItemId = instanceActivityItem.getId();
      recoveryActivityIds.add(instanceActivityItemId);
      if(StringUtils.equals(instanceActivityItemId, currnetTargetInstanceActivityId)) {
        break;
      }
    }
    
    // 2、========
    Map<String , Collection<String>> recoveryActivityLogMapping = new HashMap<>();
    for (String recoveryActivityId : recoveryActivityIds) {
      Set<InstanceActivityLogEntity> logs = this.instanceActivityLogRepository.findByInstanceActivity(recoveryActivityId);
      if(logs != null && !logs.isEmpty()) {
        Collection<String> logIds = logs.stream().map(InstanceActivityLogEntity::getId).collect(Collectors.toList());
        recoveryActivityLogMapping.put(recoveryActivityId, logIds);
      }
    }
    Validate.isTrue(!recoveryActivityLogMapping.isEmpty() , "未发现恢复的目标活动下有任何操作日志，不允许恢复!!");
    Validate.isTrue(recoveryActivityLogMapping.size() > 1 , "批量恢复必须跨越至少两个活动，如果是在同一活动内进行恢复，请使用单步恢复");
    
    // 3、========
    Set<InstanceActivityLogEntity> allLogs = this.instanceActivityLogRepository.findByInstanceId(currentInstanceId);
    LinkedList<InstanceActivityLogEntity> recoveryLogs = new LinkedList<>();
    Validate.isTrue(allLogs != null && allLogs.size() > 1 , "未发现指定的表单实例有任何数据变化历史，或者仅有一条数据变化历史，不能进行数据恢复操作!!");
    String endActivityLogId = null;
    for (InstanceActivityLogEntity logItem : allLogs) {
      String logItemId = logItem.getId();
      String logActivityId = logItem.getInstanceActivity().getId();
      // 如果条件成立，则确认当前logItemId就是恢复过程中结束日志点
      if(StringUtils.equals(logActivityId, targetInstanceActivityId) && recoveryActivityLogMapping.size() == 1) {
        endActivityLogId = logItemId;
        break;
      }
      recoveryLogs.add(logItem);
      
      // 匹配可能存在于recoveryActivityLogMapping中的那些必须恢复的节点
      Collection<String> logActivityLogIds = recoveryActivityLogMapping.get(logActivityId);
      if(logActivityLogIds != null) {
        logActivityLogIds.remove(logItemId);
        if(logActivityLogIds.isEmpty()) {
          recoveryActivityLogMapping.remove(logActivityId);
        }
      }
      Validate.isTrue(!recoveryActivityLogMapping.isEmpty() , "错误的算法过程，请联系管理员!!");
    }
    Validate.notNull(endActivityLogId , "错误的算法过程，未找到终止日志点，请联系管理员!!");
    return recoveryLogs;
  }
}
