package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.kuiper.entity.PageEntity;
import com.bizunited.platform.kuiper.entity.PageEventEntity;
import com.bizunited.platform.kuiper.entity.PageFlowEntity;
import com.bizunited.platform.kuiper.starter.repository.PageFlowRepository;
import com.bizunited.platform.kuiper.starter.service.PageFlowService;
import com.bizunited.platform.kuiper.starter.service.PageService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.MigrateDataConstants.PAGE_FLOW_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * 页面流服务的接口实现
 * @Author: TanBoQiuYun
 * @Date: 2019/12/31 15:33
 */
@Service("PageFlowServiceImpl")
public class PageFlowServiceImpl implements PageFlowService {
  private static final Logger LOGGER = LoggerFactory.getLogger(PageFlowServiceImpl.class);

  private static final String JSON_FILE_SUFFIX = "json";
  private static final String SAVE_LOCATION = "/pageFlow/json/";

  @Autowired
  private PageFlowRepository pageFlowRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private VenusFileService kuiperFileService;
  @Autowired
  private PageService pageService;


  @Override
  @Transactional
  public PageFlowEntity create(PageFlowEntity pageFlowEntity, Principal principal) {
    //入参验证
    this.createValidation(pageFlowEntity, principal);
    //获取当前登录人信息
    UserVo user = SecurityUtils.getCurrentUser();

    Date nowDate = new Date();
    pageFlowEntity.setCreateTime(nowDate);
    pageFlowEntity.setModifyTime(nowDate);
    pageFlowEntity.setCreateUser(user);
    pageFlowEntity.setModifyUser(user);
    pageFlowEntity.setCreator(user.getAccount());
    pageFlowEntity.setModifier(user.getAccount());
    pageFlowEntity.setProjectName(ApplicationContextUtils.getProjectName());

    //保存内容到文件
    this.saveFile(pageFlowEntity);
    pageFlowRepository.save(pageFlowEntity);

    if (!CollectionUtils.isEmpty(pageFlowEntity.getPages())){
      pageService.save(pageFlowEntity, pageFlowEntity.getPages());
    }
    return pageFlowEntity;
  }

  @Override
  @Transactional
  public PageFlowEntity update(PageFlowEntity pageFlowEntity, Principal principal) {
    //入参验证
    this.updateValidation(pageFlowEntity, principal);
    //获取当前登录人信息
    UserVo user = SecurityUtils.getCurrentUser();

    PageFlowEntity updatePageFlow = this.findDetailsById(pageFlowEntity.getId());
    Validate.notNull(updatePageFlow, "当前更新对象不存在!");
    updatePageFlow.setModifyTime(new Date());
    updatePageFlow.setModifyUser(user);
    updatePageFlow.setModifier(user.getAccount());

    updatePageFlow.setName(pageFlowEntity.getName());
    //更新页面流说明字段
    if (StringUtils.isNotBlank(pageFlowEntity.getComment())){
      updatePageFlow.setComment(pageFlowEntity.getComment());
    }
    updatePageFlow.setPageCount(pageFlowEntity.getPageCount());
    updatePageFlow.setLineCount(pageFlowEntity.getLineCount());
    //更新文件内容
    if (!pageFlowEntity.getPageFlowContent().equals(updatePageFlow.getPageFlowContent())){
      updatePageFlow.setPageFlowContent(pageFlowEntity.getPageFlowContent());
      byte[] pageFlowContent = pageFlowEntity.getPageFlowContent().getBytes(StandardCharsets.UTF_8);
      String fileName = updatePageFlow.getFileName();
      this.kuiperFileService.saveFile(updatePageFlow.getRelativePath(), fileName, fileName, pageFlowContent);
    }
    pageFlowRepository.save(updatePageFlow);
    //更新具体页面信息(可能涉及到新增、删除以及修改)
    pageService.save(pageFlowEntity, pageFlowEntity.getPages());
    return updatePageFlow;
  }

  /**
   * 保存页面流内容到文件系统中
   * @param pageFlow
   */
  private void saveFile(PageFlowEntity pageFlow) {
    String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String renameImage = UUID.randomUUID().toString();
    String fileRename = renameImage + "." + JSON_FILE_SUFFIX;
    String relativePath = StringUtils.join(SAVE_LOCATION , folderName, "/", (new Random().nextInt(100) % 10));
    byte[] pageFlowContent = pageFlow.getPageFlowContent().getBytes(StandardCharsets.UTF_8);
    pageFlow.setFileName(fileRename);
    pageFlow.setRelativePath(relativePath);
    this.kuiperFileService.saveFile(relativePath, fileRename, fileRename, pageFlowContent);
  }

  @Override
  public PageFlowEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)){
      return null;
    }
    PageFlowEntity pageFlow = this.pageFlowRepository.findDetailsById(id);
    return this.loadDetails(pageFlow);
  }

  @Override
  public PageFlowEntity findById(String id) {
    if (StringUtils.isBlank(id)){
      return null;
    }
    return this.pageFlowRepository.findById(id).orElse(null);
  }

  @Override
  public PageFlowEntity findByCode(String code) {
    if (StringUtils.isBlank(code)){
      return null;
    }
    return this.pageFlowRepository.findByCode(code);
  }

  @Override
  public PageFlowEntity findDetailsByCode(String code) {
    if (StringUtils.isBlank(code)){
      return null;
    }
    PageFlowEntity entity = this.pageFlowRepository.findDetailsByCode(code);
    return this.loadDetails(entity);
  }

  /**
   * 分页查询
   * @param pageFlow
   * @param pageable
   * @return
   */
  @Override
  public Page<PageFlowEntity> findByConditions(PageFlowEntity pageFlow, Pageable pageable) {
    return pageFlowRepository.findByConditions(pageFlow, pageable);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    PageFlowEntity pageFlow = pageFlowRepository.findById(id).orElse(null);
    if (null != pageFlow){
      //先删除对应的页面信息
      pageService.deleteByPageFlowId(id);
      //再删除对应的父页面
      pageFlowRepository.delete(pageFlow);
      //删除文件
      kuiperFileService.deleteFile(pageFlow.getRelativePath(), pageFlow.getFileName(), pageFlow.getFileName());
    }
  }

  @Override
  public List<PageFlowEntity> findAll() {
    return pageFlowRepository.findAll(Sort.by(Sort.Direction.DESC, "createTime"));
  }

  @Override
  public long countByIds(String[] ids) {
    if(ArrayUtils.isEmpty(ids)) {
      return 0;
    }
    return pageFlowRepository.countByIds(ids);
  }

  @Override
  public List<PageFlowEntity> findDetailsByIds(String[] ids) {
    if(ArrayUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    return pageFlowRepository.findDetailsByids(ids);
  }

  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry templateEntry = zipFile.getEntry(PAGE_FLOW_FILENAME);
    if(templateEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
      return;
    }
    try (InputStream is = zipFile.getInputStream(templateEntry);
         ObjectInputStream ois = new ObjectInputStream(is)) {
      List<PageFlowEntity> pageFlows = (List<PageFlowEntity>) ois.readObject();
      if(CollectionUtils.isEmpty(pageFlows)) {
        importModel.appendLine("导入的数据为空，请检查数据");
        return;
      }
      this.importData(pageFlows, importModel);
    } catch (IOException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage(), e);
      importModel.append("读取模版数据失败：").appendLine(e.getMessage());
    }
  }

  /**
   * 批量导入页面流
   * @param pageFlows
   * @param importModel
   */
  private void importData(List<PageFlowEntity> pageFlows, MigrateImportModel importModel) {
    importModel.setTotalCount(pageFlows.size());
    for (int i = 0; i < pageFlows.size(); i++) {
      PageFlowEntity pageFlow = pageFlows.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(pageFlow, importModel);
    }
  }

  /**
   * 导入单个页面流数据
   * @param pageFlow
   * @param importModel
   */
  private void importData(PageFlowEntity pageFlow, MigrateImportModel importModel) {
    importModel.appendLine("开始导入页面流：name=%s,code=%s", pageFlow.getName(), pageFlow.getCode());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    PageFlowEntity dbPageFlow = pageFlowRepository.findByCode(pageFlow.getCode());
    if(dbPageFlow != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("页面流数据已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbPageFlow != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("页面流数据已存在，执行更新导入");
      this.handleUpdateData(pageFlow, dbPageFlow, importModel);
      return;
    }
    if(dbPageFlow != null && ImportExecuteModeEnum.ADD == executeMode) {
      importModel.appendLine("不支持的执行模式");
      importModel.addSkipCount();
      return;
    }
    if(dbPageFlow == null) {
      this.handleCreateData(pageFlow, importModel);
      return;
    }
  }

  /**
   * 处理新增数据
   * @param pageFlow
   * @param importModel
   */
  private void handleCreateData(PageFlowEntity pageFlow, MigrateImportModel importModel) {
    importModel.appendLine("导入新增数据");
    this.handleImportDataId(pageFlow);
    this.loadFileContentFromZip(pageFlow, importModel);
    this.create(pageFlow, SecurityUtils.getPrincipal());
    importModel.addCreateCount();
  }

  /**
   * 执行导入的数据更新操作
   * @param pageFlow
   * @param dbPageFlow
   * @param importModel
   */
  private void handleUpdateData(PageFlowEntity pageFlow, PageFlowEntity dbPageFlow, MigrateImportModel importModel) {
    this.handleImportDataId(pageFlow);
    pageFlow.setId(dbPageFlow.getId());
    this.loadFileContentFromZip(pageFlow, importModel);
    this.update(pageFlow, SecurityUtils.getPrincipal());
    importModel.addUpdateCount();
    importModel.appendLine("更新数据成功");
  }

  /**
   * 从zip文件中加载页面流数据
   * @param pageFlow
   * @param importModel
   * @return
   */
  private PageFlowEntity loadFileContentFromZip(PageFlowEntity pageFlow, MigrateImportModel importModel) {
    ZipFile zipFile = importModel.getZipFile();
    byte[] bytes = this.getBytesFromZip(zipFile, pageFlow.getRelativePath(), pageFlow.getFileName());
    pageFlow.setPageFlowContent(new String(bytes, StandardCharsets.UTF_8));
    Set<PageEntity> pages = pageFlow.getPages();
    if(CollectionUtils.isEmpty(pages)) {
      return pageFlow;
    }
    for (PageEntity page : pages) {
      byte[] pageBytes = this.getBytesFromZip(zipFile, page.getRelativePath(), page.getFileName());
      page.setPageContent(new String(pageBytes, StandardCharsets.UTF_8));
      Set<PageEventEntity> events = page.getEvents();
      if(CollectionUtils.isEmpty(events)) {
        continue;
      }
      for (PageEventEntity event : events) {
        byte[] eventBytes = this.getBytesFromZip(zipFile, event.getRelativePath(), event.getFileName());
        event.setEventContent(new String(eventBytes, StandardCharsets.UTF_8));
      }
    }
    return pageFlow;
  }

  /**
   * 从压缩文件读取指定文件
   * @param zipFile
   * @param relativePath
   * @param fileName
   * @return
   */
  private byte[] getBytesFromZip(ZipFile zipFile, String relativePath, String fileName) {
    byte[] bytes;
    try {
      bytes = ZipFileUtils.readZipFile(zipFile, relativePath, fileName);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("读取压缩包文件出错:%s/%s", relativePath, fileName), e);
    }
    Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "未从压缩包中读取到文件：%s/%s", relativePath, fileName);
    return bytes;
  }

  /**
   * 处理导入数据的ID，将所有对象的ID设置成null
   * @param pageFlow
   */
  private void handleImportDataId(PageFlowEntity pageFlow) {
    pageFlow.setId(null);
    if(!CollectionUtils.isEmpty(pageFlow.getPages())) {
      for (PageEntity page : pageFlow.getPages()) {
        page.setId(null);
        if(!CollectionUtils.isEmpty(page.getEvents())) {
          for (PageEventEntity event : page.getEvents()) {
            event.setId(null);
          }
        }
      }
    }
  }

  /**
   * 页面流加载页面流文件内容
   * @param pageFlow
   * @return
   */
  private PageFlowEntity loadDetails(PageFlowEntity pageFlow) {
    if (Objects.isNull(pageFlow)){
      return null;
    }
    Set<PageEntity> pages = pageService.findDetailsByPageFlowId(pageFlow.getId());
    pageFlow.setCreateUser(userService.findByAccount(pageFlow.getCreator()));
    pageFlow.setModifyUser(userService.findByAccount(pageFlow.getModifier()));
    pageFlow.setPageFlowContent(this.readFileContent(pageFlow.getRelativePath(), pageFlow.getFileName()));
    pageFlow.setPages(pages);
    return pageFlow;
  }

  /**
   * 读取文件内容
   * @param relativePath
   * @param fileRename
   * @return
   */
  private String readFileContent(String relativePath, String fileRename) {
    byte[] bytes = this.kuiperFileService.readFileContent(relativePath, fileRename);
    String str = null;
    if (bytes != null){
      str = new String(bytes, StandardCharsets.UTF_8);
    }
    return str;
  }

  /**
   * 更新-参数验证
   * @param pageFlow
   * @param principal
   */
  private void updateValidation(PageFlowEntity pageFlow, Principal principal) {
    Validate.notNull(principal, "获取当前登录人信息失败！");
    Validate.notBlank(principal.getName(), "获取当前登录人的用户名失败!");
    Validate.notBlank(pageFlow.getId(), "更新数据必须要有ID");
    Validate.notBlank(pageFlow.getName(), "页面流名称不能为空!");
    Validate.notBlank(pageFlow.getCode(), "页面流编码不能为空!");
    Validate.notNull(pageFlow.getPageCount(), "页面流页面数不能为空!");
    Validate.notNull(pageFlow.getLineCount(), "页面流规则连线数不能为空!");
    Validate.notBlank(pageFlow.getPageFlowContent(), "保存的页面流文件内容不能为空!");
    //验证名称和编码重复性
    long countByName = this.pageFlowRepository.countByNameWithoutIdAndProjectName(pageFlow.getName(), pageFlow.getId(), ApplicationContextUtils.getProjectName());
    Validate.isTrue(countByName == 0L, "页面流名称已存在，请替换");
    long countByCode = this.pageFlowRepository.countByCodeWithoutId(pageFlow.getCode(), pageFlow.getId());
    Validate.isTrue(countByCode == 0L, "页面流编码已存在，请替换");
  }

  /**
   * 新增-参数验证
   * @param pageFlow
   * @param principal
   */
  private void createValidation(PageFlowEntity pageFlow, Principal principal) {
    Validate.notNull(principal, "获取当前登录人信息失败！");
    Validate.notBlank(principal.getName(), "获取当前登录人的用户名失败!");
    Validate.notNull(pageFlow, "保存对象不存在!");
    Validate.isTrue(pageFlow.getId() == null, "创建数据不能有ID");
    Validate.notBlank(pageFlow.getName(), "页面流名称不能为空!");
    Validate.notBlank(pageFlow.getCode(), "页面流编码不能为空!");
    Validate.notNull(pageFlow.getPageCount(), "页面流页面数不能为空!");
    Validate.notNull(pageFlow.getLineCount(), "页面流规则连线数不能为空!");
    Validate.notBlank(pageFlow.getPageFlowContent(), "保存的页面流文件内容不能为空!");
    //验证名称和编码重复性
    long countByName = this.pageFlowRepository.countByNameAndProjectName(pageFlow.getName(), ApplicationContextUtils.getProjectName());
    Validate.isTrue(countByName == 0L, "页面流名称已存在，请替换");
    long countByCode = this.pageFlowRepository.countByCode(pageFlow.getCode());
    Validate.isTrue(countByCode == 0L, "页面流编码已存在，请替换");
  }
}
