package com.bizunited.platform.kuiper.starter.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateLayoutEntity;
import com.bizunited.platform.kuiper.starter.repository.TemplateLayoutRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateLayoutService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;
import java.util.Set;

@Service("TemplateLayoutImpl")
public class TemplateLayoutServiceImpl implements TemplateLayoutService {
  private static final Logger LOGGER = LoggerFactory.getLogger(TemplateLayoutServiceImpl.class);
  @Autowired
  private TemplateLayoutRepository templateLayoutRepository;
  @Autowired
  private VenusFileService kuiperFileService;
  /**
   * 布局信息文件内容存放的相对路径
   */
  private static final String LAYOUTDIR = "/layout";
  private static final String EXTENDTYPE = ".txt";

  /**
   * 保存表单布局信息，布局内容保存到磁盘上，文件信息保存到数据库
   * @param templateId 模板编号
   * @param layout     表单布局
   * @return
   */
  @Transactional
  @Override
  public TemplateLayoutEntity save(String templateId, Integer layoutType, Object layout) throws IOException {
    /*
     * 1、验证各种入参的合理性
     * 2、保存文件
     * 3、保存数据库信息
     * */
    Validate.notBlank(templateId, "模板id不能为空,请检查!!");
    Validate.notNull(layout, "表单布局信息不能为空，请检查");
    Integer currentLayoutType = 1;
    if(layoutType != null) {
      currentLayoutType = layoutType;
    }
    Validate.isTrue(currentLayoutType == 1 || currentLayoutType == 2 || currentLayoutType == 3 , "目前表单布局只支持pc端、移动端和打印端，请检查");
    
    // 确定文件名、文件保存的相对路径等信息
    String relativePath = LAYOUTDIR;
    String fileName = StringUtils.join(templateId, "-type", currentLayoutType, EXTENDTYPE);
    //保存布局内容信息到文件
    byte[] objectContents = writeObjectToBytes(layout);
    this.kuiperFileService.saveFile(relativePath, fileName, fileName, objectContents);
    TemplateLayoutEntity templateLayout = this.findByTemplateId(templateId , currentLayoutType);
    
    // 如果条件成立，则是对文件进行创建
    if (null == templateLayout) { 
      templateLayout = new TemplateLayoutEntity();
      templateLayout.setCreateTime(new Date());
      templateLayout.setRelativePath(relativePath);
      templateLayout.setFileName(fileName);
      templateLayout.setLayoutType(currentLayoutType);
      TemplateEntity currentTemplate = new TemplateEntity();
      currentTemplate.setId(templateId);
      templateLayout.setTemplate(currentTemplate);
      templateLayout.setProjectName(ApplicationContextUtils.getProjectName());
    } 
    // 否则就是对已有文件进行修改
    else { 
      templateLayout.setRelativePath(relativePath);
      templateLayout.setFileName(fileName);
      templateLayout.setCreateTime(new Date());
      templateLayout.setLayoutType(currentLayoutType);
    }
    templateLayoutRepository.save(templateLayout);
    return templateLayout;
  }

  /**
   * 将object读成bytes内容
   * @param obj
   * @return
   */
  private byte[] writeObjectToBytes(Object obj){
    byte[] bytes = null;
    try (ByteArrayOutputStream out = new ByteArrayOutputStream();
        ObjectOutputStream sOut = new ObjectOutputStream(out);){
        sOut.writeObject(obj);
        sOut.flush();
        bytes= out.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
      return new byte[0];
    }
    return bytes;
}

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.TemplateLayoutService#deleteByTemplateId(java.lang.String)
   */
  @Override
  @Transactional
  public void deleteByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return;
    }
    Set<TemplateLayoutEntity> templateLayouts = this.findByTemplateId(templateId);
    if(templateLayouts == null || templateLayouts.isEmpty()) {
      return;
    }
    
    for (TemplateLayoutEntity templateLayout : templateLayouts) {
      String fileName = templateLayout.getFileName();
      String relativePath = templateLayout.getRelativePath();
      // 删除文件
      this.kuiperFileService.deleteFile(relativePath, fileName, fileName);
    }
    // 再删除数据库
    this.templateLayoutRepository.deleteByTemplateId(templateId);
  }
  
  @Override
  @Transactional
  public void deleteByTemplateId(String templateId, Integer layoutType) {
    if(StringUtils.isBlank(templateId)) {
      return;
    }
    TemplateLayoutEntity templateLayout = this.findByTemplateId(templateId, layoutType);
    if(templateLayout == null) {
      return;
    }
    
    String fileName = templateLayout.getFileName();
    String relativePath = templateLayout.getRelativePath();
    // 删除文件
    this.kuiperFileService.deleteFile(relativePath, fileName, fileName);
    // 再删除数据库
    this.templateLayoutRepository.deleteById(templateLayout.getId());
  }

  /**
   * 根据模板编号查找表单布局
   * @param templateId 模板编号
   * @return
   */
  @Override
  public JSONObject findDetailsByTemplateId(String templateId , Integer layoutType) {
    if(StringUtils.isBlank(templateId)) {
      return null;
    }
    
    // 如果条件成立，则查询默认模板
    TemplateLayoutEntity templateLayout = this.findByTemplateId(templateId, layoutType);
    if(templateLayout == null) {
      return null;
    }
    
    // 从文件系统读取文件内容
    String fileName = templateLayout.getFileName();
    String relativePath = templateLayout.getRelativePath();
    byte[] fileContents = this.kuiperFileService.readFileContent(relativePath, fileName);
    if(fileContents == null || fileContents.length == 0) {
      return null;
    }
    
    // 转换成json
    return readBytesToJson(fileContents);
  }

  /**
   * 将bytes内容读取成json信息
   * @param path 文件保存路径
   * @return
   */
  private JSONObject readBytesToJson(byte[] bytes) {
    Object layoutObject = null;
    try (ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ObjectInputStream sIn = new ObjectInputStream(in);){
      layoutObject = sIn.readObject();
    } catch (Exception e) {
      LOGGER.error(e.getMessage() , e);
      return null;
    }
    if(layoutObject == null) {
      return null;
    }
    
    return JSON.parseObject(JSON.toJSONString(layoutObject));
  }

  @Override
  public Set<TemplateLayoutEntity> findByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    return this.templateLayoutRepository.findByTemplateId(templateId);
  }

  @Override
  public TemplateLayoutEntity findByTemplateId(String templateId, Integer layoutType) {
    if(StringUtils.isBlank(templateId)) {
      return null;
    }
    int currentLayoutType = 1;
    if(layoutType != null) {
      currentLayoutType = layoutType;
    }
    
    // 如果条件成立，则查询默认模板
    return templateLayoutRepository.findByTemplateIdAndType(templateId, currentLayoutType);
  }
}
