package com.bizunited.platform.kuiper.starter.service.internal;

import com.bizunited.platform.kuiper.entity.TemplateEntity;
import com.bizunited.platform.kuiper.entity.TemplateMaintainerEntity;
import com.bizunited.platform.kuiper.service.TemplateService;
import com.bizunited.platform.kuiper.starter.repository.TemplateMaintainerRepository;
import com.bizunited.platform.kuiper.starter.service.TemplateMaintainerService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.Date;
import java.util.List;
import java.util.Set;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_ADMIN_ACCOUNT;

@Service("TemplateMaintainerServiceImpl")
public class TemplateMaintainerServiceImpl implements TemplateMaintainerService {

  @Autowired
  private UserService userService;
  @Autowired
  private TemplateService templateService;
  @Autowired
  private TemplateMaintainerRepository templateMaintainerRepository;

  /**
   * 具有授权权限的人，授权用户能维护其他用户创建的模板
   * 注：该方式是对具有不同版本的模板进行授权
   * @param code 指定模板code
   * @param maintainers 指定的可操作指定模板的维护人员账号(数组)
   * @param maintainerAccounts 配置文件中的具有授权权限的人员账号信息
   * @param principal 当前人
   */
  @Override
  @Transactional
  public void binding(String code, String[] maintainers, String[] maintainerAccounts, Principal principal) {
    //1 验证传入参数
    Validate.notBlank(code , "授权时，指定的模板code必须传入!!");
    String account = principal.getName();
    Validate.notBlank(account, "未能获取到当前用户信息，请检查!!");
    if(ArrayUtils.isEmpty(maintainerAccounts)) {
      maintainerAccounts = new String[] {DEFAULT_ADMIN_ACCOUNT};
    }
    List<String> maintainerAccountList = Lists.newArrayList(maintainerAccounts);
    Validate.isTrue(maintainerAccountList.contains(account), "当前用户没有授权权限，请联系管理员!!");

    //2 检查模板信息
    List<TemplateEntity> dbTemplates = templateService.findByCode(code);
    Validate.notEmpty(dbTemplates, "根据指定的模板code，未能获取到模板信息!!");
    
    //3 更新维护人员
    //--a>先删除中间表
    for(TemplateEntity template : dbTemplates) {
      templateMaintainerRepository.unbindByTemplateId(template.getId());
    }
    if(ArrayUtils.isEmpty(maintainers)) {
      return;
    }
    //--b>添加中间表数据
    for(String maintainer : maintainers) {
      for(TemplateEntity template : dbTemplates) {
        TemplateMaintainerEntity templateMaintainer = new TemplateMaintainerEntity();
        templateMaintainer.setCreateTime(new Date());
        templateMaintainer.setTemplate(template);
        templateMaintainer.setUserAccount(maintainer);
        templateMaintainerRepository.save(templateMaintainer);
      }
    }
  }

  @Override
  public Set<TemplateMaintainerEntity> findDetailsByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    Set<TemplateMaintainerEntity> maintainers = templateMaintainerRepository.findByTemplateId(templateId);
    if(CollectionUtils.isEmpty(maintainers)) {
      return Sets.newHashSet();
    }
    for (TemplateMaintainerEntity maintainer : maintainers) {
      UserVo user = userService.findByAccount(maintainer.getUserAccount());
      maintainer.setUser(user);
    }
    return maintainers;
  }

  @Override
  public Set<TemplateMaintainerEntity> findByTemplateId(String templateId) {
    if(StringUtils.isBlank(templateId)) {
      return Sets.newHashSet();
    }
    return templateMaintainerRepository.findByTemplateId(templateId);
  }

  @Override
  @Transactional
  public void binding(String templateId, String[] accounts) {
    Validate.notEmpty(templateId, "表单模版ID不能为空");
    Validate.notEmpty(accounts, "绑定账号不能为空");
    TemplateEntity template = templateService.findById(templateId);
    Validate.notNull(template, "未找到模版对象：%s", templateId);
    for (String account : accounts) {
      TemplateMaintainerEntity templateMaintainer = new TemplateMaintainerEntity();
      templateMaintainer.setCreateTime(new Date());
      templateMaintainer.setTemplate(template);
      templateMaintainer.setUserAccount(account);
      templateMaintainerRepository.save(templateMaintainer);
    }
  }
}
