package com.bizunited.platform.mars.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.mars.entity.RuleDefinitionEntity;
import com.bizunited.platform.mars.repository.RuleDefinitionEntityRepository;
import com.bizunited.platform.mars.service.RuleDefinitionService;
import com.bizunited.platform.mars.service.RuleDefinitionListener;

import java.lang.Override;
import java.lang.String;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import javax.transaction.Transactional;

import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

/**
 * RuleDefinitionEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("RuleDefinitionEntityServiceImpl")
public class RuleDefinitionEntityServiceImpl extends  AbstractMarsSaveableService implements RuleDefinitionService { 
  @Autowired
  private RuleDefinitionEntityRepository ruleDefinitionEntityRepository;
  @Autowired(required = false)
  private List<RuleDefinitionListener> ruleDefinitionListeners;
  @Autowired
  private VenusFileService venusFileService;
  
  @Transactional
  @Override
  public RuleDefinitionEntity create(RuleDefinitionEntity ruleDefinitionEntity) {
    // 创建者、修改者信息
    String account = this.findCreator();
    ruleDefinitionEntity.setCreateAccount(account);
    ruleDefinitionEntity.setCreateTime(new Date());
    ruleDefinitionEntity.setModifyAccount(account);
    ruleDefinitionEntity.setModifyTime(new Date());
    ruleDefinitionEntity.setProjectName(ApplicationContextUtils.getProjectName());
    
    RuleDefinitionEntity current = this.createForm(ruleDefinitionEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public RuleDefinitionEntity createForm(RuleDefinitionEntity ruleDefinitionEntity) { 
    this.createValidation(ruleDefinitionEntity);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    this.ruleDefinitionEntityRepository.save(ruleDefinitionEntity);
    if(ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionModify(ruleDefinitionEntity));
    }
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return ruleDefinitionEntity;
  }
  /**
   * 在创建一个新的RuleDefinitionEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RuleDefinitionEntity ruleDefinitionEntity) { 
    Validate.notNull(ruleDefinitionEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(ruleDefinitionEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    ruleDefinitionEntity.setId(null);
    Validate.notBlank(ruleDefinitionEntity.getCreateAccount(), "添加信息时，创建人账号不能为空！");
    Validate.notNull(ruleDefinitionEntity.getCreateTime(), "添加信息时，创建时间不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getModifyAccount(), "添加信息时，更新人账号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getCode(), "添加信息时，规则定义业务编号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getCverion(), "添加信息时，规则定义版本不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getDesc(), "添加信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateFilename(), "添加信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateRelativepath(), "添加信息时，前端图形化展示结构描述文件的相对路径不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(ruleDefinitionEntity.getCreateAccount() == null || ruleDefinitionEntity.getCreateAccount().length() < 255 , "创建人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getModifyAccount() == null || ruleDefinitionEntity.getModifyAccount().length() < 255 , "更新人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getCode() == null || ruleDefinitionEntity.getCode().length() < 128 , "规则定义业务编号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getCverion() == null || ruleDefinitionEntity.getCverion().length() < 128 , "规则定义版本,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getDesc() == null || ruleDefinitionEntity.getDesc().length() < 512 , "规则定义用途说明,在进行添加时填入值超过了限定长度(512)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateFilename() == null || ruleDefinitionEntity.getTemplateFilename().length() < 256 , "规则定义用途说明,在进行添加时填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateRelativepath() == null || ruleDefinitionEntity.getTemplateRelativepath().length() < 256 , "前端图形化展示结构描述文件的相对路径,在进行添加时填入值超过了限定长度(256)，请检查!");
  }
  @Transactional
  @Override
  public RuleDefinitionEntity update(RuleDefinitionEntity ruleDefinitionEntity) { 
    // 修改者信息
    String account = this.findModifyor();
    ruleDefinitionEntity.setModifyAccount(account);
    ruleDefinitionEntity.setModifyTime(new Date());
    
    RuleDefinitionEntity current = this.updateForm(ruleDefinitionEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public RuleDefinitionEntity updateForm(RuleDefinitionEntity ruleDefinitionEntity) { 
    /* 
     * 注意：修改时规则定义的基本信息只能修改描述信息
     * code、cversion等信息不能进行修改
     * */
    
    this.updateValidation(ruleDefinitionEntity);
    // ===================基本信息
    String currentId = ruleDefinitionEntity.getId();
    Optional<RuleDefinitionEntity> op_currentRuleDefinitionEntity = this.ruleDefinitionEntityRepository.findById(currentId);
    RuleDefinitionEntity currentRuleDefinitionEntity = op_currentRuleDefinitionEntity.orElse(null);
    Validate.notNull(currentRuleDefinitionEntity ,"未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentRuleDefinitionEntity.setModifyAccount(ruleDefinitionEntity.getModifyAccount());
    currentRuleDefinitionEntity.setModifyTime(ruleDefinitionEntity.getModifyTime());
    currentRuleDefinitionEntity.setDesc(ruleDefinitionEntity.getDesc());
    currentRuleDefinitionEntity.setTemplateFilename(ruleDefinitionEntity.getTemplateFilename());
    currentRuleDefinitionEntity.setTemplateRelativepath(ruleDefinitionEntity.getTemplateRelativepath());
    
    this.ruleDefinitionEntityRepository.saveAndFlush(currentRuleDefinitionEntity);
    if(ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionModify(currentRuleDefinitionEntity));
    }
    return currentRuleDefinitionEntity;
  }
  /**
   * 在更新一个已有的RuleDefinitionEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RuleDefinitionEntity ruleDefinitionEntity) { 
    Validate.isTrue(!StringUtils.isBlank(ruleDefinitionEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(ruleDefinitionEntity.getModifyAccount(), "修改信息时，更新人账号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getDesc(), "修改信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateFilename(), "修改信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateRelativepath(), "修改信息时，前端图形化展示结构描述文件的相对路径不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(ruleDefinitionEntity.getModifyAccount() == null || ruleDefinitionEntity.getModifyAccount().length() < 255 , "更新人账号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getDesc() == null || ruleDefinitionEntity.getDesc().length() < 512 , "规则定义用途说明,在进行修改时填入值超过了限定长度(512)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateFilename() == null || ruleDefinitionEntity.getTemplateFilename().length() < 256 , "规则定义用途说明,在进行修改时填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateRelativepath() == null || ruleDefinitionEntity.getTemplateRelativepath().length() < 256 , "前端图形化展示结构描述文件的相对路径,在进行修改时填入值超过了限定长度(256)，请检查!");
  } 
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    RuleDefinitionEntity current = this.findById(id);
    if(current != null) { 
      this.ruleDefinitionEntityRepository.delete(current);
    }
    if(ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionDelete(current));
    }
  }
  @Override
  public RuleDefinitionEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.ruleDefinitionEntityRepository.findDetailsById(id);
  }
  @Override
  public RuleDefinitionEntity findDetailsByCodeAndVersion(String code, String cversion) {
    if(StringUtils.isBlank(code) || StringUtils.isBlank(cversion)) { 
      return null;
    }
    return this.ruleDefinitionEntityRepository.findDetailsByCodeAndVersionAndProjectName(code, cversion, ApplicationContextUtils.getProjectName());
  }
  /**
   * 按照版本和业务编号查询详情
   * @param code 指定的业务编号
   * @param cversion 指定的版本信息
   * @return 
   */
  public RuleDefinitionEntity findByCodeAndVersion(String code , String cversion) {
    if(StringUtils.isBlank(code) || StringUtils.isBlank(cversion)) { 
      return null;
    }
    return this.ruleDefinitionEntityRepository.findByCodeAndVersionAndProjectName(code, cversion, ApplicationContextUtils.getProjectName());
  }
  @Override
  public RuleDefinitionEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<RuleDefinitionEntity> op = ruleDefinitionEntityRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  public List<RuleDefinitionEntity> findAll() {
    return this.ruleDefinitionEntityRepository.findAll(Sort.by("createTime"));
  }


  /**
   * 按照给定的模板编码和版本，查询模板文件内容。
   * @param code
   * @param version
   * @return
   */
  @Override
  public String findContentByCodeAndVersion(String code, String version) {
    if (StringUtils.isBlank(code) || StringUtils.isBlank(version)) {
      return null;
    }
    RuleDefinitionEntity ruleDefinition = ruleDefinitionEntityRepository.findByCodeAndVersionAndProjectName(code, version, ApplicationContextUtils.getProjectName());
    if(ruleDefinition == null){
      return null;
    }
    byte[] content = venusFileService.readFileContent(ruleDefinition.getTemplateRelativepath(), ruleDefinition.getTemplateFilename());
    if(content == null || content.length == 0){
      return null;
    }
    return new String(content, StandardCharsets.UTF_8);
  }
} 
