package com.bizunited.platform.mars.service.internal;

import com.bizunited.platform.mars.entity.RuleTemplateGroupEntity;
import com.bizunited.platform.mars.repository.RuleTemplateGroupEntityRepository;
import com.bizunited.platform.mars.service.RuleTemplateGroupEntityService;
import com.bizunited.platform.mars.service.RuleTemplateNodeEntityService;

import java.lang.Override;
import java.lang.String;
import java.util.Date;
import java.util.Optional;
import java.util.Set;

import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * RuleTemplateGroupEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("RuleTemplateGroupEntityServiceImpl")
public class RuleTemplateGroupEntityServiceImpl extends AbstractMarsSaveableService implements RuleTemplateGroupEntityService { 
  @Autowired
  private RuleTemplateGroupEntityRepository ruleTemplateGroupEntityRepository;
  @Autowired
  private RuleTemplateNodeEntityService ruleTemplateNodeEntityService;
  @Transactional
  @Override
  public RuleTemplateGroupEntity create(RuleTemplateGroupEntity ruleTemplateGroupEntity) {
    // 创建者、修改者信息
    String account = this.findCreator();
    ruleTemplateGroupEntity.setCreateAccount(account);
    ruleTemplateGroupEntity.setCreateTime(new Date());
    ruleTemplateGroupEntity.setModifyAccount(account);
    ruleTemplateGroupEntity.setModifyTime(new Date());
    // 状态都是1：正常
    ruleTemplateGroupEntity.setGroupStatus(1);
    
    RuleTemplateGroupEntity current = this.createForm(ruleTemplateGroupEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public RuleTemplateGroupEntity createForm(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */

    this.createValidation(ruleTemplateGroupEntity);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.ruleTemplateGroupEntityRepository.save(ruleTemplateGroupEntity);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return ruleTemplateGroupEntity;
  }
  /**
   * 在创建一个新的RuleTemplateGroupEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    Validate.notNull(ruleTemplateGroupEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(ruleTemplateGroupEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    ruleTemplateGroupEntity.setId(null);
    Validate.notBlank(ruleTemplateGroupEntity.getCreateAccount(), "添加信息时，创建人账号不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getCreateTime(), "添加信息时，创建时间不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getModifyAccount(), "添加信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getCode(), "添加信息时，分组编号编号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getName(), "添加信息时，模板分组名称（中文）不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getGroupStatus(), "添加信息时，分组可用状态不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getType(), "添加信息时，分组类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(ruleTemplateGroupEntity.getCreateAccount() == null || ruleTemplateGroupEntity.getCreateAccount().length() < 255 , "创建人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getModifyAccount() == null || ruleTemplateGroupEntity.getModifyAccount().length() < 255 , "更新人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getCode() == null || ruleTemplateGroupEntity.getCode().length() < 128 , "分组编号编号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getName() == null || ruleTemplateGroupEntity.getName().length() < 128 , "模板分组名称（中文）,在进行添加时填入值超过了限定长度(128)，请检查!");
    RuleTemplateGroupEntity currentRuleTemplateGroupEntity = this.findByCode(ruleTemplateGroupEntity.getCode());
    Validate.isTrue(currentRuleTemplateGroupEntity == null, "分组编号编号已存在,请检查");
  }
  @Transactional
  @Override
  public RuleTemplateGroupEntity update(RuleTemplateGroupEntity ruleTemplateGroupEntity) {
    // 修改者信息
    String account = this.findModifyor();
    ruleTemplateGroupEntity.setModifyAccount(account);
    ruleTemplateGroupEntity.setModifyTime(new Date());
    
    RuleTemplateGroupEntity current = this.updateForm(ruleTemplateGroupEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public RuleTemplateGroupEntity updateForm(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    /* 
     * 只能对状态、名称进行修改。分组的业务编号、支持的组件类型不能修改
     * */
    
    this.updateValidation(ruleTemplateGroupEntity);
    // ===================基本信息
    String currentId = ruleTemplateGroupEntity.getId();
    Optional<RuleTemplateGroupEntity> op_currentRuleTemplateGroupEntity = this.ruleTemplateGroupEntityRepository.findById(currentId);
    RuleTemplateGroupEntity currentRuleTemplateGroupEntity = op_currentRuleTemplateGroupEntity.orElse(null);
    currentRuleTemplateGroupEntity = Validate.notNull(currentRuleTemplateGroupEntity ,"未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentRuleTemplateGroupEntity.setModifyAccount(ruleTemplateGroupEntity.getModifyAccount());
    currentRuleTemplateGroupEntity.setModifyTime(ruleTemplateGroupEntity.getModifyTime());
    currentRuleTemplateGroupEntity.setName(ruleTemplateGroupEntity.getName());
    currentRuleTemplateGroupEntity.setGroupStatus(ruleTemplateGroupEntity.getGroupStatus());
    
    this.ruleTemplateGroupEntityRepository.saveAndFlush(currentRuleTemplateGroupEntity);
    return currentRuleTemplateGroupEntity;
  }
  /**
   * 在更新一个已有的RuleTemplateGroupEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    Validate.isTrue(!StringUtils.isBlank(ruleTemplateGroupEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(ruleTemplateGroupEntity.getModifyAccount(), "修改信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getName(), "修改信息时，模板分组名称（中文）不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getGroupStatus(), "修改信息时，分组可用状态不能为空！");
    
    // 重复性判断，基本属性，需要满足unique = true
    RuleTemplateGroupEntity currentForCode = this.findByCode(ruleTemplateGroupEntity.getCode());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , ruleTemplateGroupEntity.getId()) , "分组编号编号已存在,请检查"); 
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(ruleTemplateGroupEntity.getModifyAccount() == null || ruleTemplateGroupEntity.getModifyAccount().length() < 255 , "更新人账号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getName() == null || ruleTemplateGroupEntity.getName().length() < 128 , "模板分组名称（中文）,在进行修改时填入值超过了限定长度(128)，请检查!");
  } 
  @Override
  public RuleTemplateGroupEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.ruleTemplateGroupEntityRepository.findDetailsById(id);
  }
  @Override
  public RuleTemplateGroupEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<RuleTemplateGroupEntity> op = ruleTemplateGroupEntityRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    RuleTemplateGroupEntity current = this.findById(id);
    // 如果当前分组下已经绑定了节点模板，则不允许删除
    Validate.isTrue(ruleTemplateNodeEntityService.countByTemplateGroup(id) == 0 , "当前要被删除的分组信息还存在关联的节点模板，请检查!!");
    if(current != null) { 
      this.ruleTemplateGroupEntityRepository.delete(current);
    }
  }
  @Override
  public RuleTemplateGroupEntity findByCode(String code) { 
    if(StringUtils.isBlank(code)) { 
      return null;
    }
    return this.ruleTemplateGroupEntityRepository.findByCode(code);
  }
  @Override
  public Set<RuleTemplateGroupEntity> findByGroupStatus(Integer groupStatus) {
    if(groupStatus == null) {
      return this.ruleTemplateGroupEntityRepository.findAllOrderByTypeAndName();
    } else {
      return this.ruleTemplateGroupEntityRepository.findByGroupStatus(groupStatus);
    }
  }
  @Override
  public Set<RuleTemplateGroupEntity> findByType(Integer type) {
    if(type == null || type <= 0) {
      return null;
    }
    return this.ruleTemplateGroupEntityRepository.findByType(type);
  } 
} 
