package com.bizunited.platform.mars.service.init;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.mars.entity.RuleTemplateNodeEntity;
import com.bizunited.platform.mars.entity.RuleTemplateParamsEntity;
import com.bizunited.platform.mars.repository.RuleTemplateNodeEntityRepository;
import com.bizunited.platform.mars.repository.RuleTemplateParamsEntityRepository;
import com.bizunited.platform.mars.service.RuleTemplateParamsService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.Set;

/**
 * 该初始化服务负责在第一次启动mars规则实例时,进行默认模板节点的初始化工作。</br>
 * 包括了默认的开始节点、默认的结束节点、默认的条件节点、默认的动态脚本逻辑节点
 *
 * @author yinwenjie
 */
@Component("RuleTemplateNodeInitProcessService")
public class RuleTemplateNodeInitProcessService implements InitProcessService {
  @Autowired
  private RuleTemplateNodeEntityRepository ruleTemplateNodeEntityRepository;
  @Autowired
  private RuleTemplateParamsEntityRepository ruleTemplateParamsEntityRepository;
  @Autowired
  private RuleTemplateParamsService ruleTemplateParamsService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 默认的开始节点模板编号
   */
  private static final String START_TEMPLATE_CODE = "defaultStart";
  /**
   * 默认的结束节点模板编号
   */
  private static final String END_TEMPLATE_CODE = "defaultEnd";
  /**
   * 默认的条件节点模板编号
   */
  private static final String CONDITION_TEMPLATE_CODE = "defaultCondition";
  /**
   * 默认的动态脚本模板编号
   */
  private static final String DYNAMIC_SCRIPT_TEMPLATE_CODE = "dynamicScript";
  /**
   * 默认的动态脚本模板编号
   */
  private static final String TIMER_CODE = "defaultTimer";
  /**
   * 默认的并发汇聚节点编号
   */
  private static final String CONCURRENT_AGGREGATION = "defaultConcurrentAggregation";
  /**
   * 默认的A/B并行分支节点编号
   */
  private static final String PARALLEL_BRANCH = "defaultParallelBranch";

  @Override
  public boolean doProcess() {
    RuleTemplateNodeEntity defaultStart = this.ruleTemplateNodeEntityRepository.findByCode(START_TEMPLATE_CODE);
    RuleTemplateNodeEntity defaultEnd = this.ruleTemplateNodeEntityRepository.findByCode(END_TEMPLATE_CODE);
    RuleTemplateNodeEntity defaultCondition = this.ruleTemplateNodeEntityRepository.findByCode(CONDITION_TEMPLATE_CODE);
    RuleTemplateNodeEntity dynamicScript = this.ruleTemplateNodeEntityRepository.findByCode(DYNAMIC_SCRIPT_TEMPLATE_CODE);
    Set<RuleTemplateParamsEntity> paramsEntities = null;
    if (dynamicScript != null) {
      paramsEntities = this.ruleTemplateParamsService.findDetailsByTemplateNode(dynamicScript.getId());
    }
    RuleTemplateNodeEntity timerCode = this.ruleTemplateNodeEntityRepository.findByCode(TIMER_CODE);
    RuleTemplateNodeEntity concurrentAggregation = this.ruleTemplateNodeEntityRepository.findByCode(CONCURRENT_AGGREGATION);
    RuleTemplateNodeEntity parallelBranch = this.ruleTemplateNodeEntityRepository.findByCode(PARALLEL_BRANCH);

    if (defaultStart == null || defaultEnd == null || defaultCondition == null
            || dynamicScript == null || (dynamicScript != null && CollectionUtils.isEmpty(paramsEntities))
            || timerCode == null || concurrentAggregation == null || parallelBranch == null) {
      return true;
    }
    return false;
  }

  @Override
  @Transactional
  public void init() {
    RuleTemplateNodeEntity defaultStart = this.ruleTemplateNodeEntityRepository.findByCode(START_TEMPLATE_CODE);
    if (defaultStart == null) {
      this.initDefaultStart();
    }
    RuleTemplateNodeEntity defaultEnd = this.ruleTemplateNodeEntityRepository.findByCode(END_TEMPLATE_CODE);
    if (defaultEnd == null) {
      this.initDefaultEnd();
    }
    RuleTemplateNodeEntity defaultCondition = this.ruleTemplateNodeEntityRepository.findByCode(CONDITION_TEMPLATE_CODE);
    if (defaultCondition == null) {
      this.initDefaultCondition();
    }
    RuleTemplateNodeEntity dynamicScript = this.ruleTemplateNodeEntityRepository.findByCode(DYNAMIC_SCRIPT_TEMPLATE_CODE);
    if (dynamicScript == null) {
      this.initDefaultDynamicScript();
    } else {
      Set<RuleTemplateParamsEntity> paramsEntities = this.ruleTemplateParamsService.findDetailsByTemplateNode(dynamicScript.getId());
      if (CollectionUtils.isEmpty(paramsEntities)) {
        this.initDefaultDynamicScriptParams(dynamicScript);
      }
    }
    RuleTemplateNodeEntity timerCode = this.ruleTemplateNodeEntityRepository.findByCode(TIMER_CODE);
    if (timerCode == null) {
      this.initDefaultTimer();
    }
    RuleTemplateNodeEntity concurrentAggregation = this.ruleTemplateNodeEntityRepository.findByCode(CONCURRENT_AGGREGATION);
    if (concurrentAggregation == null) {
      this.initDefaultConcurrentAggregation();
    }
    RuleTemplateNodeEntity parallelBranch = this.ruleTemplateNodeEntityRepository.findByCode(PARALLEL_BRANCH);
    if (parallelBranch == null) {
      this.initDefaultParallelBranch();
    }
  }

  /**
   * 初始化开始节点的模板
   */
  private void initDefaultStart() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(START_TEMPLATE_CODE);
    ruleTemplateNode.setName("默认开始");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(4);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }

  /**
   * 初始化开始节点的模板
   */
  private void initDefaultEnd() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(END_TEMPLATE_CODE);
    ruleTemplateNode.setName("默认结束");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(5);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }

  /**
   * 初始化默认的条件判定节点的模板
   */
  private void initDefaultCondition() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(CONDITION_TEMPLATE_CODE);
    ruleTemplateNode.setName("默认条件");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(1);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }

  /**
   * 初始化默认的动态脚本节点的模板
   */
  private void initDefaultDynamicScript() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(DYNAMIC_SCRIPT_TEMPLATE_CODE);
    ruleTemplateNode.setName("默认动态脚本逻辑");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(2);
    ruleTemplateNode.setSourceType(5);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
    initDefaultDynamicScriptParams(ruleTemplateNode);
  }

  /**
   * 初始化动态脚本的默认入参与出参
   * @param ruleTemplateNode
   */
  private void initDefaultDynamicScriptParams(RuleTemplateNodeEntity ruleTemplateNode) {
    RuleTemplateParamsEntity inputRuleTemplateParams = new RuleTemplateParamsEntity();
    inputRuleTemplateParams.setParamName("_script");
    inputRuleTemplateParams.setType(1);
    inputRuleTemplateParams.setNullable(false);
    inputRuleTemplateParams.setParamType("string");
    inputRuleTemplateParams.setParamDesc("动态脚本");
    inputRuleTemplateParams.setTemplateNode(ruleTemplateNode);
    this.ruleTemplateParamsEntityRepository.save(inputRuleTemplateParams);

    RuleTemplateParamsEntity outputRuleTemplateParams = new RuleTemplateParamsEntity();
    outputRuleTemplateParams.setParamName("_result");
    outputRuleTemplateParams.setType(2);
    outputRuleTemplateParams.setNullable(true);
    outputRuleTemplateParams.setParamType("object");
    outputRuleTemplateParams.setParamDesc("动态脚本出参");
    outputRuleTemplateParams.setTemplateNode(ruleTemplateNode);
    this.ruleTemplateParamsEntityRepository.save(outputRuleTemplateParams);
  }

  /**
   * 初始化默认的时间延迟等待节点的模板
   */
  private void initDefaultTimer() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(TIMER_CODE);
    ruleTemplateNode.setName("默认延迟等待节点");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(8);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }

  /**
   * 初始化默认的并发汇聚模板
   */
  private void initDefaultConcurrentAggregation() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(CONCURRENT_AGGREGATION);
    ruleTemplateNode.setName("默认并发汇聚节点");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(7);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }

  /**
   * 初始化默认的A/B并行分支节点的模板
   */
  private void initDefaultParallelBranch() {
    Date currentTime = new Date();
    RuleTemplateNodeEntity ruleTemplateNode = new RuleTemplateNodeEntity();
    ruleTemplateNode.setCode(PARALLEL_BRANCH);
    ruleTemplateNode.setName("默认A/B并行分支节点");
    ruleTemplateNode.setCreateAccount("admin");
    ruleTemplateNode.setCreateTime(currentTime);
    ruleTemplateNode.setModifyAccount("admin");
    ruleTemplateNode.setModifyTime(currentTime);
    ruleTemplateNode.setNodeStatus(1);
    ruleTemplateNode.setType(6);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    this.ruleTemplateNodeEntityRepository.save(ruleTemplateNode);
  }
}
