package com.bizunited.platform.mars.service.internal;

import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.mars.entity.RuleSourceAggregateDataViewEntity;
import com.bizunited.platform.mars.entity.RuleSourceDataViewEntity;
import com.bizunited.platform.mars.entity.RuleSourceScriptEntity;
import com.bizunited.platform.mars.entity.RuleSourceServicableEntity;
import com.bizunited.platform.mars.entity.RuleTemplateGroupEntity;
import com.bizunited.platform.mars.entity.RuleTemplateNodeEntity;
import com.bizunited.platform.mars.entity.RuleTemplateParamsEntity;
import com.bizunited.platform.mars.repository.RuleTemplateNodeEntityRepository;
import com.bizunited.platform.mars.service.RuleSourceAggregateDataViewEntityService;
import com.bizunited.platform.mars.service.RuleSourceDataViewService;
import com.bizunited.platform.mars.service.RuleSourceScriptService;
import com.bizunited.platform.mars.service.RuleSourceServicableService;
import com.bizunited.platform.mars.service.RuleTemplateGroupService;
import com.bizunited.platform.mars.service.RuleTemplateNodeService;
import com.bizunited.platform.mars.service.RuleTemplateParamsService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * RuleTemplateNodeEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("RuleTemplateNodeEntityServiceImpl")
public class RuleTemplateNodeServiceImpl extends AbstractMarsSaveableService implements RuleTemplateNodeService {
  @Autowired
  private RuleTemplateNodeEntityRepository ruleTemplateNodeEntityRepository;
  @Autowired
  private RuleTemplateGroupService ruleTemplateGroupEntityService;
  @Autowired
  private RuleTemplateParamsService ruleTemplateParamsService;
  @Autowired
  private RuleSourceServicableService ruleSourceServicableService;
  @Autowired
  private RuleSourceDataViewService ruleSourceDataViewEntityService;
  @Autowired
  private RuleSourceAggregateDataViewEntityService ruleSourceAggregateDataViewService;
  @Autowired
  private RuleSourceScriptService ruleSourceScriptService;
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private PlatformContext platformContext;

  @Transactional
  @Override
  public RuleTemplateNodeEntity create(RuleTemplateNodeEntity ruleTemplateNodeEntity) {
    // 创建者、修改者信息
    String account = this.findCreator();
    ruleTemplateNodeEntity.setCreateAccount(account);
    ruleTemplateNodeEntity.setCreateTime(new Date());
    ruleTemplateNodeEntity.setModifyAccount(account);
    ruleTemplateNodeEntity.setModifyTime(new Date());
    ruleTemplateNodeEntity.setNodeStatus(1);
    RuleTemplateNodeEntity current = this.createForm(ruleTemplateNodeEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public RuleTemplateNodeEntity createForm(RuleTemplateNodeEntity ruleTemplateNode) {
    this.createValidation(ruleTemplateNode);
    this.createValidationEngine(ruleTemplateNode);
    // 检验和创建涉及的处理源信息
    this.createValidationSource(ruleTemplateNode);
    ruleTemplateNode.setProjectName(platformContext.getAppName());
    // 创建和保存基本信息
    this.ruleTemplateNodeEntityRepository.saveAndFlush(ruleTemplateNode);
    // 创建入参和出参信息
    Set<RuleTemplateParamsEntity> intpus = ruleTemplateNode.getInputs();
    Set<RuleTemplateParamsEntity> outpus = ruleTemplateNode.getOutputs();
    this.createValidationParams(ruleTemplateNode, intpus, outpus);
    // 返回最终处理的结果，里面带有详细的关联信息
    return ruleTemplateNode;
  }

  /**
   * 在创建一个新的RuleTemplateNodeEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RuleTemplateNodeEntity ruleTemplateNode) {
    Validate.notNull(ruleTemplateNode, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(ruleTemplateNode.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    ruleTemplateNode.setId(null);
    Validate.notBlank(ruleTemplateNode.getCreateAccount(), "添加信息时，创建人账号不能为空！");
    Validate.notNull(ruleTemplateNode.getCreateTime(), "添加信息时，创建时间不能为空！");
    Validate.notBlank(ruleTemplateNode.getModifyAccount(), "添加信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateNode.getCode(), "添加信息时，节点模板业务编号不能为空！");
    Validate.notBlank(ruleTemplateNode.getName(), "添加信息时，节点模板业务名称（中文）不能为空！");
    Validate.notNull(ruleTemplateNode.getNodeStatus(), "添加信息时，节点可用状态不能为空！");
    Integer type = ruleTemplateNode.getType();
    Validate.notNull(type, "添加信息时，节点模板类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(ruleTemplateNode.getCreateAccount() == null || ruleTemplateNode.getCreateAccount().length() < 255, "创建人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateNode.getModifyAccount() == null || ruleTemplateNode.getModifyAccount().length() < 255, "更新人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateNode.getCode() == null || ruleTemplateNode.getCode().length() < 128, "节点模板业务编号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleTemplateNode.getName() == null || ruleTemplateNode.getName().length() < 128, "节点模板业务名称（中文）,在进行添加时填入值超过了限定长度(128)，请检查!");
    // 业务编码是否重复
    RuleTemplateNodeEntity currentRuleTemplateNodeEntity = this.findByCode(ruleTemplateNode.getCode());
    Validate.isTrue(currentRuleTemplateNodeEntity == null, "节点模板业务编号已存在,请检查");
    // 入参和出参是否正确
    Set<RuleTemplateParamsEntity> intpus = ruleTemplateNode.getInputs();
    if (!CollectionUtils.isEmpty(intpus)) {
      Set<String> paramNames = intpus.stream().map(RuleTemplateParamsEntity::getParamName).filter(item -> StringUtils.isNotBlank(item)).collect(Collectors.toSet());
      Validate.isTrue(paramNames.size() == intpus.size(), "在进行节点模板添加时，发现至少一组入参名称重复，或为null。请检查!!");
    }
    Set<RuleTemplateParamsEntity> outtpus = ruleTemplateNode.getInputs();
    if (!CollectionUtils.isEmpty(outtpus)) {
      Set<String> paramNames = outtpus.stream().map(RuleTemplateParamsEntity::getParamName).filter(item -> StringUtils.isNotBlank(item)).collect(Collectors.toSet());
      Validate.isTrue(paramNames.size() == outtpus.size(), "在进行节点模板添加时，发现至少一组出参名称重复，或为null。请检查!!");
    }

    // 验证type为9（业务自定义组件的情况）
    if (type == 9 || type == 10 || type == 11) {
      Integer extType = ruleTemplateNode.getExtType();
      Validate.notNull(extType, "当创建业务自定义节点模板时，必须添加业务扩展类型（extType），如果缺少业务扩展类型的选择，操作者可以自行在数据字典中进行维护!!");
      RuleTemplateNodeEntity exsitNode = this.findByExtType(type, extType);
      Validate.isTrue(exsitNode == null, "指定的业务自定义节点类型已被使用(extType)，请重新选择，如果缺少业务扩展类型的选择，操作者可以自行在数据字典中进行维护!!");
      Validate.isTrue(ruleTemplateNode.getFormEngine() || ruleTemplateNode.getUserPage(), "进行业务自定义节点模板添加时，要么使用（动态）表单引擎作为页面呈现方式，要么使用自定义页面作为页面呈现方式");
      // 如果使用页面引擎，则以下信息需要填写
      if (ruleTemplateNode.getFormEngine()) {
        String formEngineCode = ruleTemplateNode.getFormEngineCode();
        Validate.notBlank(formEngineCode, "（动态）表单引擎的业务编号必须指定!!");
      } else {
        String taskName = ruleTemplateNode.getTaskName();
        Validate.notBlank(taskName, "使用用户自定义页面呈现时，url的参数名必须填写!!");
      }
      Validate.isTrue(ruleTemplateNode.getSourceType() == null, "自定义扩展节点(SourceType)处理器类型应为空，请检查");
    }
  }

  /**
   * 该私有方法负责在基本信息进行添加或者修改时，对入参和出参信息进行全量添加的操作
   *
   * @param ruleTemplateNode
   */
  private void createValidationParams(RuleTemplateNodeEntity currentTemplateNode, Set<RuleTemplateParamsEntity> intpus, Set<RuleTemplateParamsEntity> outpus) {
    // 处理入参和出参信息
    if (!CollectionUtils.isEmpty(intpus)) {
      intpus.stream().forEach(item -> {
        item.setId(null);
        item.setType(1);
        item.setTemplateNode(currentTemplateNode);
        this.ruleTemplateParamsService.create(item);
      });
    }
    if (!CollectionUtils.isEmpty(outpus)) {
      outpus.stream().forEach(item -> {
        item.setId(null);
        item.setType(2);
        item.setTemplateNode(currentTemplateNode);
        this.ruleTemplateParamsService.create(item);
      });
    }
  }

  /**
   * 该私有方法用于在添加节点模板前，验证其可能传入的页面介质信息
   */
  private void createValidationEngine(RuleTemplateNodeEntity node) {
    Boolean listEngine = node.getListEngine();
    String listEngineCode = node.getListEngineCode();
    String listEngineVersion = node.getListEngineVersion();
    Boolean formEngine = node.getFormEngine();
    String formEngineCode = node.getFormEngineCode();

    Validate.isTrue(!(listEngine && formEngine), "在设定节点模板的输入介质时，要么使用列表引擎要么使用");
    if (listEngine) {
      Validate.notBlank(listEngineCode, "错误的列表引擎业务编号，请检查!!");
      Validate.notBlank(listEngineVersion, "错误的列表引擎业务版本号，请检查!!");
    }
    if (formEngine) {
      Validate.notBlank(formEngineCode, "错误的页面引擎业务编号，请检查!!");
    }
  }

  private void createValidationSource(RuleTemplateNodeEntity node) {
    String nodeCode = node.getCode();
    int type = node.getType();
    Integer sourceType = node.getSourceType();
    // 如果不是逻辑组件，则不需要扩展描述处理源信息
    if (type != 2) {
      return;
    }
    if (sourceType == null) {
      throw new IllegalArgumentException("当节点组件类型为“逻辑组件”（type==2）时，必须为该组件设定处理源");
    }
    switch (sourceType) {
      case 1:
        // 服务源
        RuleSourceServicableEntity ruleSourceServicable = node.getSourceServicable();
        Validate.notNull(ruleSourceServicable, "未发现为该节点模板指定的服务源信息");
        ruleSourceServicable.setId(null);
        ruleSourceServicable.setServicableMethod(ruleSourceServicable.getServicableMethod());
        this.ruleSourceServicableService.create(ruleSourceServicable);
        node.setSourceServicable(ruleSourceServicable);
        node.setSourceAggregateDataView(null);
        node.setSourceDataView(null);
        node.setSourceScript(null);
        break;
      case 2:
        // 数据视图（聚集）
        RuleSourceAggregateDataViewEntity ruleSourceAggregateDataView = node.getSourceAggregateDataView();
        Validate.notNull(ruleSourceAggregateDataView, "在创建编号为[%s]的节点模板时，未发现设定的数据源（数据视图[聚合]），请检查!!", nodeCode);
        String aggregateField = ruleSourceAggregateDataView.getAggregateField();
        Validate.notBlank(aggregateField, "在创建编号为[%s]的节点模板时，未发现指定的数据视图设定的聚合字段，请检查!!", nodeCode);
        Integer aggregateType = ruleSourceAggregateDataView.getAggregateType();
        Validate.notNull(aggregateType, "在创建编号为[%s]的节点模板时，未发现指定的数据视图设定的聚合类型，请检查!!", nodeCode);
        Integer scale = ruleSourceAggregateDataView.getScale() == null ? 0 : ruleSourceAggregateDataView.getScale();
        String viewCode = ruleSourceAggregateDataView.getViewCode();
        Validate.notBlank(viewCode, "在创建编号为[%s]的节点模板时，未发现指定的数据视图设定的数据视图编号，请检查!!", nodeCode);
        // 开始添加
        ruleSourceAggregateDataView.setId(null);
        ruleSourceAggregateDataView.setAggregateField(aggregateField);
        ruleSourceAggregateDataView.setAggregateType(aggregateType);
        ruleSourceAggregateDataView.setScale(scale);
        ruleSourceAggregateDataView.setViewCode(viewCode);
        this.ruleSourceAggregateDataViewService.create(ruleSourceAggregateDataView);
        node.setSourceServicable(null);
        node.setSourceAggregateDataView(ruleSourceAggregateDataView);
        node.setSourceDataView(null);
        node.setSourceScript(null);
        break;
      case 3:
        // 固定的后端groovy脚本
        RuleSourceScriptEntity ruleSourceScript = node.getSourceScript();
        Validate.notNull(ruleSourceScript, "在创建编号为[%s]的节点模板时，未发现设定的数据源（后端脚本），请检查!!", nodeCode);
        ScriptEntity currentScript = ruleSourceScript.getScript();
        Validate.notNull(currentScript, "在创建编号为[%s]的节点模板时，未发现设定的脚本内容，应首先创建/修改脚本内容，请检查!!", nodeCode);
        String scriptId = currentScript.getId();
        ScriptEntity exsitScript = this.scriptService.findById(scriptId);
        Validate.notNull(exsitScript, "在创建编号为[%s]的节点模板时，未发现设定的脚本内容，应首先创建/修改脚本内容，请检查!!", nodeCode);
        this.ruleSourceScriptService.create(ruleSourceScript);
        node.setSourceServicable(null);
        node.setSourceAggregateDataView(null);
        node.setSourceDataView(null);
        node.setSourceScript(ruleSourceScript);
        break;
      case 4:
        // 这个是数据权限，暂时不进行编码，后续再进行提供支持
        break;
      case 5:
        // 这个是全动态groovy脚本处理器，不需要记录任何相关的扩展信息
        break;
      case 6:
        // 数据视图（集合）
        RuleSourceDataViewEntity ruleSourceDataView = node.getSourceDataView();
        Validate.notNull(ruleSourceDataView, "未发现指定的数据视图，请检查!!");
        String currentViewCode = ruleSourceDataView.getViewCode();
        Validate.notBlank(currentViewCode, "未发现指定的数据视图编号，请检查!!");
        this.ruleSourceDataViewEntityService.create(ruleSourceDataView);
        node.setSourceServicable(null);
        node.setSourceAggregateDataView(null);
        node.setSourceDataView(ruleSourceDataView);
        node.setSourceScript(null);
        break;
      default:
        throw new IllegalArgumentException("发现未知的处理源设定，请检查sourceType的传入值!!");
    }
  }

  @Transactional
  @Override
  public RuleTemplateNodeEntity update(RuleTemplateNodeEntity ruleTemplateNodeEntity) {
    // 修改者信息
    String account = this.findModifyor();
    ruleTemplateNodeEntity.setModifyAccount(account);
    ruleTemplateNodeEntity.setModifyTime(new Date());
    return this.updateForm(ruleTemplateNodeEntity);
  }

  @Transactional
  @Override
  public RuleTemplateNodeEntity updateForm(RuleTemplateNodeEntity ruleTemplateNodeEntity) {
    /*
     * 过程为：
     * 1、首先检查节点模板的基本信息是否正确，注意有几个信息不能更新（例如type、sourceType信息）
     * 2、然后更新基本信息
     * 3、接着更新处理器信息（处理器类型不会变，主要就是填写的若干属性变了）
     * 4、最后是全量更新（就是删除后添加）出参和入参信息
     * */

    // 1、======
    this.updateValidation(ruleTemplateNodeEntity);
    // ===================基本信息
    String currentId = ruleTemplateNodeEntity.getId();
    RuleTemplateNodeEntity currentRuleTemplateNodeEntity = this.ruleTemplateNodeEntityRepository.findDetailsById(currentId);
    currentRuleTemplateNodeEntity = Validate.notNull(currentRuleTemplateNodeEntity, "未发现指定的原始模型对象信息");
    // 2、开始重新赋值——一般属性
    currentRuleTemplateNodeEntity.setModifyAccount(ruleTemplateNodeEntity.getModifyAccount());
    currentRuleTemplateNodeEntity.setModifyTime(ruleTemplateNodeEntity.getModifyTime());
    currentRuleTemplateNodeEntity.setName(ruleTemplateNodeEntity.getName());
    currentRuleTemplateNodeEntity.setNodeStatus(ruleTemplateNodeEntity.getNodeStatus());
    currentRuleTemplateNodeEntity.setShowTitle(ruleTemplateNodeEntity.getShowTitle());
    currentRuleTemplateNodeEntity.setShowWidth(ruleTemplateNodeEntity.getShowWidth());
    currentRuleTemplateNodeEntity.setShowHight(ruleTemplateNodeEntity.getShowHight());
    currentRuleTemplateNodeEntity.setFormEngine(ruleTemplateNodeEntity.getFormEngine());
    currentRuleTemplateNodeEntity.setFormEngineCode(ruleTemplateNodeEntity.getFormEngineCode());
    currentRuleTemplateNodeEntity.setFormEngineVersion(ruleTemplateNodeEntity.getFormEngineVersion());
    currentRuleTemplateNodeEntity.setFormEngineDefault1Version(ruleTemplateNodeEntity.getFormEngineDefault1Version());
    currentRuleTemplateNodeEntity.setFormEngineVisibility(ruleTemplateNodeEntity.getFormEngineVisibility());
    currentRuleTemplateNodeEntity.setListEngine(ruleTemplateNodeEntity.getListEngine());
    currentRuleTemplateNodeEntity.setListEngineCode(ruleTemplateNodeEntity.getListEngineCode());
    currentRuleTemplateNodeEntity.setListEngineVersion(ruleTemplateNodeEntity.getListEngineVersion());
    currentRuleTemplateNodeEntity.setIconPath(ruleTemplateNodeEntity.getIconPath());
    currentRuleTemplateNodeEntity.setIconColor(ruleTemplateNodeEntity.getIconColor());
    currentRuleTemplateNodeEntity.setIconBgColor(ruleTemplateNodeEntity.getIconBgColor());
    currentRuleTemplateNodeEntity.setFontColor(ruleTemplateNodeEntity.getFontColor());
    currentRuleTemplateNodeEntity.setUserPage(ruleTemplateNodeEntity.getUserPage());
    currentRuleTemplateNodeEntity.setTaskName(ruleTemplateNodeEntity.getTaskName());

    this.ruleTemplateNodeEntityRepository.saveAndFlush(currentRuleTemplateNodeEntity);

    // 3、========
    // 检验及重新创建组件类型信息
    if (currentRuleTemplateNodeEntity.getSourceAggregateDataView() != null) {
      RuleSourceAggregateDataViewEntity newRuleSourceAggregateDataViewEntity = ruleTemplateNodeEntity.getSourceAggregateDataView();
      Validate.notNull(newRuleSourceAggregateDataViewEntity, "修改时（处理器类型不能修改），必须设定数据视图（聚集）的相关信息，请检查!!");
      // 注意，调用者可能没有传递扩展信息的id，这里要进行设定
      newRuleSourceAggregateDataViewEntity.setId(currentRuleTemplateNodeEntity.getSourceAggregateDataView().getId());
      this.ruleSourceAggregateDataViewService.update(newRuleSourceAggregateDataViewEntity);
    }
    if (currentRuleTemplateNodeEntity.getSourceDataView() != null) {
      RuleSourceDataViewEntity newRuleSourceDataViewEntity = ruleTemplateNodeEntity.getSourceDataView();
      Validate.notNull(newRuleSourceDataViewEntity, "修改时（处理器类型不能修改），必须设定数据视图（集合）的相关信息，请检查!!");
      newRuleSourceDataViewEntity.setId(ruleTemplateNodeEntity.getSourceDataView().getId());
      this.ruleSourceDataViewEntityService.update(newRuleSourceDataViewEntity);
    }
    if (currentRuleTemplateNodeEntity.getSourceScript() != null) {
      RuleSourceScriptEntity newRuleSourceScriptEntity = ruleTemplateNodeEntity.getSourceScript();
      Validate.notNull(newRuleSourceScriptEntity, "修改时（处理器类型不能修改），必须设定后端脚本（groovy）的相关信息，请检查!!");
      // 注意，调用者可能没有传递扩展信息的id，这里要进行设定
      newRuleSourceScriptEntity.setId(currentRuleTemplateNodeEntity.getSourceScript().getId());
      this.ruleSourceScriptService.update(newRuleSourceScriptEntity);
    }
    if (currentRuleTemplateNodeEntity.getSourceServicable() != null) {
      RuleSourceServicableEntity newRuleSourceServicableEntity = ruleTemplateNodeEntity.getSourceServicable();
      Validate.notNull(newRuleSourceServicableEntity, "修改时（处理器类型不能修改），必须设定服务源的相关信息，请检查!!");
      // 注意，调用者可能没有传递扩展信息的id，这里要进行设定
      newRuleSourceServicableEntity.setId(currentRuleTemplateNodeEntity.getSourceServicable().getId());
      this.ruleSourceServicableService.update(newRuleSourceServicableEntity);
    }

    // 4、======
    // 注意，模板的出参、入参要进行全量的更新
    String templateNodeId = ruleTemplateNodeEntity.getId();
    this.ruleTemplateParamsService.deleteByTemplateNode(templateNodeId);
    Set<RuleTemplateParamsEntity> intpus = ruleTemplateNodeEntity.getInputs();
    Set<RuleTemplateParamsEntity> outpus = ruleTemplateNodeEntity.getOutputs();
    this.createValidationParams(ruleTemplateNodeEntity, intpus, outpus);
    return ruleTemplateNodeEntity;
  }

  /**
   * 在更新一个已有的RuleTemplateNodeEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RuleTemplateNodeEntity ruleTemplateNodeEntity) {
    Validate.isTrue(!StringUtils.isBlank(ruleTemplateNodeEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(ruleTemplateNodeEntity.getModifyAccount(), "修改信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateNodeEntity.getName(), "修改信息时，节点模板业务名称（中文）不能为空！");
    Validate.notNull(ruleTemplateNodeEntity.getNodeStatus(), "修改信息时，节点可用状态不能为空！");

    // 重复性判断，基本属性，需要满足unique = true
    RuleTemplateNodeEntity currentForCode = this.findByCode(ruleTemplateNodeEntity.getCode());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId(), ruleTemplateNodeEntity.getId()), "节点模板业务编号已存在,请检查");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(ruleTemplateNodeEntity.getModifyAccount() == null || ruleTemplateNodeEntity.getModifyAccount().length() < 255, "更新人账号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateNodeEntity.getName() == null || ruleTemplateNodeEntity.getName().length() < 128, "节点模板业务名称（中文）,在进行修改时填入值超过了限定长度(128)，请检查!");

    // sourceType和扩展信息的验证
    Integer type = ruleTemplateNodeEntity.getType();
    Integer sourceType = ruleTemplateNodeEntity.getSourceType();
    if (type == 2) {
      Validate.notNull(sourceType, "修改信息时，当前节点模板的类型为“逻辑处理节点”，那么处理器类型（sourceType）必须选择");
      if (sourceType == 1) {
        Validate.isTrue(ruleTemplateNodeEntity.getSourceServicable() != null, "当处理器类型为“服务源”时，则服务源名称等扩展信息必须填写!!");
      }
      if (sourceType == 2) {
        Validate.isTrue(ruleTemplateNodeEntity.getSourceAggregateDataView() != null, "当处理器类型为“数据视图（聚集）”时，则数据视图编号等扩展信息必须填写!!");
      }
      if (sourceType == 3) {
        Validate.isTrue(ruleTemplateNodeEntity.getSourceScript() != null, "当处理器类型为“后台脚本”时，则脚本内容等扩展信息必须填写!!");
      }
    }
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    RuleTemplateNodeEntity current = this.findById(id);
    if (current != null) {
      this.ruleTemplateNodeEntityRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public void bindTemplateGroup(String templateGroupId, String[] templateNodeIds) {
    RuleTemplateGroupEntity templateGroup = this.ruleTemplateGroupEntityService.findById(templateGroupId);
    Validate.notNull(templateGroup, "未找到指定的分组信息，请检查!!");
    Set<RuleTemplateNodeEntity> nodes = this.validateOnMapping(templateNodeIds);
    for (RuleTemplateNodeEntity node : nodes) {
      Validate.isTrue(node.getNodeStatus() == 1, "在绑定节点模板[%s]时，发现该节点模板已被禁用，不能参与绑定过程", node.getCode());
    }

    // 接下来做绑定更新
    for (RuleTemplateNodeEntity node : nodes) {
      node.setTemplateGroup(templateGroup);
    }
    this.ruleTemplateNodeEntityRepository.saveAll(nodes);
  }

  @Override
  @Transactional
  public void rebindTemplateGroup(String templateGroupId, String[] templateNodeIds) {
    // 取消当前模板分组和其它所有节点模板间的绑定关系，然后重新进行绑定
    this.unbindAllTemplateGroups(templateGroupId);
    // 然后再进行绑定
    this.bindTemplateGroup(templateGroupId, templateNodeIds);
  }

  @Override
  @Transactional
  public void unbindTemplateGroups(String templateGroupId, String[] templateNodeIds) {
    RuleTemplateGroupEntity templateGroup = this.ruleTemplateGroupEntityService.findById(templateGroupId);
    Validate.notNull(templateGroup, "未找到指定的分组信息，请检查!!");
    Set<RuleTemplateNodeEntity> nodes = this.validateOnMapping(templateNodeIds);
    // 接下来做取消绑定的更新
    for (RuleTemplateNodeEntity node : nodes) {
      node.setTemplateGroup(null);
    }
    this.ruleTemplateNodeEntityRepository.saveAll(nodes);
  }

  @Override
  @Transactional
  public void unbindAllTemplateGroups(String templateGroupId) {
    RuleTemplateGroupEntity templateGroup = this.ruleTemplateGroupEntityService.findById(templateGroupId);
    Validate.notNull(templateGroup, "在进行解绑操作时，未找到指定的分组信息，请检查!!");
    this.ruleTemplateNodeEntityRepository.unbindAllTemplateGroups(templateGroupId);
  }

  /**
   * 该私有方法在节点模板和分组模板进行绑定/重绑定前进行验证
   *
   * @param templateNodeIds
   * @return
   */
  private Set<RuleTemplateNodeEntity> validateOnMapping(String[] templateNodeIds) {
    Set<RuleTemplateNodeEntity> nodes = this.ruleTemplateNodeEntityRepository.findByIds(templateNodeIds);
    Validate.isTrue(!CollectionUtils.isEmpty(nodes) && templateNodeIds.length == nodes.size(),
            "在进行模板分组和模板节点绑定/重绑定/解绑，发现传入的模板id编号信息为null，或者未全部找到匹配的模板节点信息，请检查");
    return nodes;
  }

  @Override
  public RuleTemplateNodeEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.ruleTemplateNodeEntityRepository.findDetailsById(id);
  }

  @Override
  public RuleTemplateNodeEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<RuleTemplateNodeEntity> op = ruleTemplateNodeEntityRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public RuleTemplateNodeEntity findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.ruleTemplateNodeEntityRepository.findByCode(code);
  }

  @Override
  public RuleTemplateNodeEntity findDetailsByCode(String code) {
    if (StringUtils.isEmpty(code)) {
      return null;
    }
    return this.ruleTemplateNodeEntityRepository.findDetailsByCode(code);
  }

  @Override
  public Set<RuleTemplateNodeEntity> findByTemplateGroupAndStatus(String templateGroup, Integer nodeStatus) {
    if (StringUtils.isBlank(templateGroup)) {
      return null;
    }

    if (nodeStatus == null) {
      return this.ruleTemplateNodeEntityRepository.findByTemplateGroup(templateGroup);
    }
    return this.ruleTemplateNodeEntityRepository.findByTemplateGroupAndStatus(templateGroup, nodeStatus);
  }

  @Override
  public List<RuleTemplateNodeEntity> findByTemplateGroupAndStatusAndType(String templateGroup,
                                                                          Integer nodeStatus, Integer type) {
    return this.ruleTemplateNodeEntityRepository.findByTemplateGroupAndStatusAndType(templateGroup, nodeStatus, type);
  }

  @Override
  public Set<RuleTemplateNodeEntity> findByStatus(Integer nodeStatus) {
    if (nodeStatus == null) {
      return null;
    }
    return this.ruleTemplateNodeEntityRepository.findByStatus(nodeStatus);
  }

  @Override
  public Page<RuleTemplateNodeEntity> findByConditions(Pageable pageable) {
    Pageable cPageable = pageable;
    if (cPageable == null) {
      cPageable = PageRequest.of(0, 50);
    }
    return this.ruleTemplateNodeEntityRepository.findByConditions(cPageable);
  }

  @Override
  public RuleTemplateNodeEntity findByExtType(Integer type, Integer extType) {
    if (extType == null) {
      return null;
    }
    return this.ruleTemplateNodeEntityRepository.findByExtType(type, extType);
  }

  @Override
  public long countByTemplateGroup(String templateGroup) {
    if (StringUtils.isBlank(templateGroup)) {
      return 0;
    }
    return this.ruleTemplateNodeEntityRepository.countByTemplateGroup(templateGroup);
  }

  @Override
  @Transactional
  public RuleTemplateNodeEntity enable(String id) {
    Validate.notBlank(id, "启用节点模板主键不能为空，请检查！");
    RuleTemplateNodeEntity current = this.ruleTemplateNodeEntityRepository.findById(id).orElse(null);
    Validate.notNull(current, "节点模板数据错误，请检查");
    current.setNodeStatus(1);
    return this.ruleTemplateNodeEntityRepository.saveAndFlush(current);
  }

  @Override
  @Transactional
  public RuleTemplateNodeEntity disable(String id) {
    Validate.notBlank(id, "禁用节点模板主键不能为空，请检查！");
    RuleTemplateNodeEntity current = this.ruleTemplateNodeEntityRepository.findById(id).orElse(null);
    Validate.notNull(current, "节点模板数据错误，请检查");
    current.setNodeStatus(0);
    return this.ruleTemplateNodeEntityRepository.saveAndFlush(current);
  }

  @Override
  public Page<RuleTemplateNodeEntity> findByTypesAndStatus(Pageable pageable, Set<Integer> types, Integer nodeStatus, String name, String groupCode) {
    Validate.isTrue(CollectionUtils.isNotEmpty(types), "节点模板类型不能为空，请检查！");
    Pageable cPageable = pageable;
    if (cPageable == null) {
      cPageable = PageRequest.of(0, 50);
    }
    return ruleTemplateNodeEntityRepository.findByTypesAndStatus(cPageable, types, nodeStatus, name, groupCode);
  }
} 
