package com.bizunited.platform.mars.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.PlatformContext;
import com.bizunited.platform.mars.entity.RuleConcurrentAggregationEntity;
import com.bizunited.platform.mars.entity.RuleDefinitionEntity;
import com.bizunited.platform.mars.entity.RuleLockNodeEntity;
import com.bizunited.platform.mars.entity.RuleNodeEntity;
import com.bizunited.platform.mars.entity.RuleNodeNextsEntity;
import com.bizunited.platform.mars.entity.RuleNodeParamsEntity;
import com.bizunited.platform.mars.entity.RuleTemplateNodeEntity;
import com.bizunited.platform.mars.entity.RuleTemplateParamsEntity;
import com.bizunited.platform.mars.entity.RuleTimerNodeEntity;
import com.bizunited.platform.mars.repository.RuleDefinitionEntityRepository;
import com.bizunited.platform.mars.service.RuleConcurrentAggregationService;
import com.bizunited.platform.mars.service.RuleDefinitionListener;
import com.bizunited.platform.mars.service.RuleDefinitionService;
import com.bizunited.platform.mars.service.RuleLockNodeService;
import com.bizunited.platform.mars.service.RuleNodeNextsService;
import com.bizunited.platform.mars.service.RuleNodeParamsService;
import com.bizunited.platform.mars.service.RuleNodeService;
import com.bizunited.platform.mars.service.RuleTemplateNodeService;
import com.bizunited.platform.mars.service.RuleTimerNodeService;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * RuleDefinitionEntity业务模型的服务层接口实现
 *
 * @author yinwenjie
 */
@Service("RuleDefinitionEntityServiceImpl")
public class RuleDefinitionServiceImpl extends AbstractMarsSaveableService implements RuleDefinitionService {
  @Autowired
  private RuleDefinitionEntityRepository ruleDefinitionEntityRepository;
  @Autowired(required = false)
  private List<RuleDefinitionListener> ruleDefinitionListeners;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private RuleTemplateNodeService ruleTemplateNodeService;
  @Autowired
  private RuleNodeService ruleNodeService;
  @Autowired
  private RuleNodeParamsService ruleNodeParamsService;
  @Autowired
  private RuleNodeNextsService ruleNodeNextsService;
  @Autowired
  private RuleLockNodeService ruleLockNodeService;
  @Autowired
  private RuleConcurrentAggregationService ruleConcurrentAggregationService;
  @Autowired
  private RuleTimerNodeService ruleTimerNodeService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RuleDefinitionServiceImpl.class);

  @Transactional
  public RuleDefinitionEntity create(RuleDefinitionEntity definition, JSONObject layoutContent) {
    Validate.notNull(definition, "创建规则定义时，定义信息必须传入");
    Validate.notNull(layoutContent, "规则定义结构内容必须传入!!");
    byte[] fileContent;
    try {
      fileContent = layoutContent.toString().getBytes("UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e);
    }
    Validate.isTrue(fileContent != null && fileContent.length > 0, "创建规则定义时，页面布局文件必须传入");
    String code = definition.getCode();
    String version = definition.getCverion();
    String desc = definition.getDesc();
    String id = definition.getId();
    Validate.isTrue(id == null, "创建规则定义时，其id信息不能有值!!");
    definition.setId(null);
    Validate.notBlank(desc, "创建规则定义时，其desc描述信息必须有值!!");
    RuleDefinitionEntity exsitRuleDefinition = this.findByCodeAndVersion(code, version);
    Validate.isTrue(exsitRuleDefinition == null, "创建规则定义时，指定的code【%s】和version【%s】已经存在于规则定义中，请重新设定", code, version);
    Set<RuleNodeEntity> nodes = definition.getNodes();
    Validate.isTrue(!CollectionUtils.isEmpty(nodes), "创建规则定义时，未发现任何规则节点定义，请检查!!");
    // 确认只有一个开始节点，有一个或者多个结束节点
    long startNodeCount = nodes.stream().filter(item -> item.getType() == 4).count();
    long endNodeCount = nodes.stream().filter(item -> item.getType() == 5).count();
    Validate.isTrue(startNodeCount == 1, "规则定义中必须有且只有一个开始节点，请检查!!");
    Validate.isTrue(endNodeCount >= 1, "规则定义中必须至少有一个结束节点，请检查!!");

    /*
     * 在完成边界校验后，处理步骤为：
     * 1、首先进行基本信息的添加，包括页面布局文件内容信息
     * 2、然后进行规则节点信息的添加
     *  2.1、创建规则基本信息前，如果这个规则定义类型是3（锁）、6（A/B并行分支）、8（计时）、9（业务自定义）
     *  则还需要处理关联信息
     *  2.2、包括规则节点的出入参明细信息
     * 3、最后处理节点定义间的连线信息
     * */

    // 1、===============
    // 确认文件信息
    String relativePath = this.saveScriptContent();
    String fileName = StringUtils.join(code, "_", version, ".txt");
    this.venusFileService.saveFile(relativePath, fileName, fileName, fileContent);
    definition.setTemplateFilename(fileName);
    definition.setTemplateRelativepath(relativePath);
    // 基本信息保存
    RuleDefinitionEntity newDefinition = this.create(definition);
    definition.setId(newDefinition.getId());

    // 2、===============
    Map<String, RuleNodeEntity> nodeCodeMapping = new HashMap<>();
    nodes.stream().forEach(node -> {
      // 2.1、处理节点基本信息（不包括入参和出参信息）
      this.validateAndSaveRuleNode(definition, node, nodeCodeMapping);
      // 处理节点参数信息(参数和使用的处理源情况挂钩)
      this.validateAndSaveRuleNodeParams(definition, node, nodeCodeMapping);
    });

    // 3、==========
    // 最后是进行连线信息的创建
    this.validateAndSaveNexts(nodes, nodeCodeMapping);
    return definition;
  }

  /**
   * 该私有方法在创建规则定义时，对每一个传入的运行时节点定义信息进行检查和处理（不包括入参和出参信息）
   *
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndSaveRuleNode(RuleDefinitionEntity ruleDefinition, RuleNodeEntity node, Map<String, RuleNodeEntity> nodeCodeMapping) {
    // 以下是边界校验信息
    String nodeId = node.getId();
    Validate.isTrue(nodeId == null, "创建规则节点定义时，其id信息不能有值!!");
    node.setId(null);
    String nodeCode = node.getCode();
    RuleTemplateNodeEntity templateNode = node.getTemplateNode();
    Validate.notNull(templateNode, "创建规则节点定义时，其使用的节点模板必须指定");
    String templateNodeId = templateNode.getId();
    Validate.notBlank(templateNodeId, "创建规则节点定义时，其使用的节点模板编号需要指定");
    RuleTemplateNodeEntity exsitTemplateNode = this.ruleTemplateNodeService.findDetailsById(templateNodeId);
    Validate.notNull(exsitTemplateNode, "创建规则节点定义时，未发现指定的节点模板信息[%s]", templateNodeId);
    Validate.isTrue(templateNode.getNodeStatus() == 1, "创建规则节点定义时，发现至少一个节点模板状态不可用，请检查！！");
    // 模板信息和节点定义信息必须匹配
    RuleNodeEntity exsitRuleNode = this.ruleNodeService.findByCode(nodeCode);
    Validate.isTrue(exsitRuleNode == null, "创建规则节点定义时，发现重复的节点定义code，请检查!!");
    Integer type = node.getType();
    Validate.notNull(type, "创建规则节点定义时，节点类型必须填写");
    Validate.isTrue(type == exsitTemplateNode.getType(), "创建规则定义时，发现至少一个节点定义的类型和对应的模板节点中的不一致");
    // 如果是逻辑组件，则必须填写处理器扩展信息
    if (type == 2) {
      Integer sourceType = node.getSourceType();
      Validate.notNull(sourceType, "创建规则节点定义时，节点工作逻辑类型必须填写");
      Validate.isTrue(sourceType == exsitTemplateNode.getSourceType(), "创建规则定义时，发现至少一个节点定义的数据源类型和对应的模板节点中的不一致");
    }
    // 处理锁节点定义的相关扩展
    else if (type == 3) {
      RuleLockNodeEntity ruleLockNode = node.getRuleLockExt();
      Validate.notNull(ruleLockNode, "锁节点定义[%s]，没有发现传入的扩展信息ruleLockExt，请检查", nodeCode);
      ruleLockNode = this.ruleLockNodeService.create(ruleLockNode);
    }
    // 处理A/B并行分支节点定义的相关扩展
    else if (type == 6) {
      RuleConcurrentAggregationEntity ruleConcurrentAggregation = node.getRuleAggregationExt();
      Validate.notNull(ruleConcurrentAggregation, "A/B并行分支处理节点定义[%s]，没有发现传入的扩展信息ruleAggregationExt ，请检查", nodeCode);
      ruleConcurrentAggregation = this.ruleConcurrentAggregationService.create(ruleConcurrentAggregation);
    }
    // 处理计时节点定义的相关扩展
    else if (type == 8) {
      RuleTimerNodeEntity ruleTimerNode = node.getRuleTimerExt();
      Validate.notNull(ruleTimerNode, "计时节点定义[%s]，没有发现传入的扩展信息ruleTimerExt，请检查", nodeCode);
      Validate.isTrue(ruleTimerNode.getTimerCount() >= 0, "暂停的时间数不能为负数，请检查");
      Validate.isTrue(ruleTimerNode.getTimeUnit() > 0 && ruleTimerNode.getTimeUnit() < 5, "暂停的时间单位错误，请检查");
      ruleTimerNode = this.ruleTimerNodeService.create(ruleTimerNode);
      // 处理业务扩展节点的相关扩展
    } else if (type == 9) {
      Integer extType = node.getExtType();
      Validate.notNull(extType, "创建业务扩展节点时，节点扩展类型必须填写");
    } else if (type == 10) {
      Integer extType = node.getExtType();
      Validate.notNull(extType, "创建A/B并行扩展节点时，节点扩展类型必须填写");
      RuleConcurrentAggregationEntity ruleConcurrentAggregation = node.getRuleAggregationExt();
      Validate.notNull(ruleConcurrentAggregation, "自定义A/B并行分支处理节点定义[%s]，没有发现传入的扩展信息ruleAggregationExt ，请检查", nodeCode);
      ruleConcurrentAggregation = this.ruleConcurrentAggregationService.create(ruleConcurrentAggregation);
    } else if (type == 11) {
      Integer extType = node.getExtType();
      Validate.notNull(extType, "创建条件判定扩展节点时，节点扩展类型必须填写");
    }


    // 处理规则定义的基本信息（不包括入参和出参信息）
    node.setDefinition(ruleDefinition);
    this.ruleNodeService.create(node);
    nodeCodeMapping.put(nodeCode, node);
  }

  /**
   * 该私有方法在创建规则定义时，对每一个传入的运行时节点定义的参数信息（包括入参和出参）进行检查和处理
   *
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndSaveRuleNodeParams(RuleDefinitionEntity ruleDefinition, RuleNodeEntity node, Map<String, RuleNodeEntity> nodeCodeMapping) {
    String templateNodeId = node.getTemplateNode().getId();
    RuleTemplateNodeEntity templateNode = this.ruleTemplateNodeService.findDetailsById(templateNodeId);
    // 检验可能的入参和可能的出参信息
    Set<RuleTemplateParamsEntity> inputTemplates = templateNode.getInputs();
    Set<RuleTemplateParamsEntity> outputTemplates = templateNode.getOutputs();
    final Set<RuleNodeParamsEntity> inputs = node.getInputs() == null ? new HashSet<>() : node.getInputs();
    final Set<RuleNodeParamsEntity> outputs = node.getOutputs() == null ? new HashSet<>() : node.getOutputs();
    Set<String> inputTemplateParamNames = inputs.stream().map(RuleNodeParamsEntity::getTemplateParamName).collect(Collectors.toSet());
    Set<String> outputTemplateParamNames = outputs.stream().map(RuleNodeParamsEntity::getTemplateParamName).collect(Collectors.toSet());
    // 检验模板中规定的入参，是否都进行了设定
    if (!CollectionUtils.isEmpty(inputTemplates)) {
      long count = inputTemplates.stream().filter(item -> inputTemplateParamNames.contains(item.getParamName())).count();
      Validate.isTrue(count == inputTemplates.size() && inputs.size() == inputTemplates.size(), "至少一个节点模板设定的入参，在节点定义中并没有绑定上下文信息，请检查!!");
    }
    // 检验模板中规定的出参，是否都进行了设定
    if (!CollectionUtils.isEmpty(outputTemplates)) {
      long count = outputTemplates.stream().filter(item -> outputTemplateParamNames.contains(item.getParamName())).count();
      Validate.isTrue(count == outputTemplates.size() && outputs.size() == outputTemplates.size(), "至少一个节点模板设定的出参，在节点定义中并没有绑定上下文信息，请检查!!");
    }
    // 接着检查和处理可能的入参
    inputs.stream().forEach(inputItem -> {
      String paramDesc = inputItem.getParamDesc();
      String paramType = inputItem.getParamType();
      String templateParamName = inputItem.getTemplateParamName();
      inputItem.setType(1);
      inputItem.setNode(node);
      Validate.notBlank(paramDesc, "入参数描述信息必须填写!!");
      Validate.notBlank(templateParamName, "入参【实参】-模板参数名信息必须填写!!");
      Validate.notNull(paramType, "入参数类型信息必须填写!!");
      this.ruleNodeParamsService.create(inputItem);
    });
    // 然后检查和处理可能的出参
    outputs.stream().forEach(outputItem -> {
      String paramDesc = outputItem.getParamDesc();
      String paramType = outputItem.getParamType();
      String templateParamName = outputItem.getTemplateParamName();
      outputItem.setType(2);
      outputItem.setNode(node);
      Validate.notBlank(paramDesc, "出参数描述信息必须填写!!");
      Validate.notBlank(templateParamName, "出参【实参】-模板参数名信息必须填写!!");
      Validate.notNull(paramType, "出参数类型信息必须填写!!");
      this.ruleNodeParamsService.create(outputItem);
    });
  }

  /**
   * 该私有方法在创建规则定义时，对节点间连线信息进行检查和处理
   *
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndSaveNexts(Set<RuleNodeEntity> nodes, Map<String, RuleNodeEntity> nodeCodeMapping) {
    nodes.stream().forEach(node -> {
      Integer type = node.getType();
      String nodeCode = node.getCode();
      Set<RuleNodeNextsEntity> nexts = node.getNexts() == null ? new HashSet<>() : node.getNexts();
      Set<RuleNodeNextsEntity> normalNexts = nexts.stream().filter(item -> item.getLineType() == 1).collect(Collectors.toSet());
      RuleNodeEntity fromNode = nodeCodeMapping.get(node.getCode());
      String fromNodeCode = fromNode.getCode();
      /*
       * a、如果当前节点类型是条件节点，则才支持一个或者多个分支
       * b、如果当前节点类型是终结，则才支持没有分支
       * b、其余类型的节点，都只支持一个分支路径
       * */
      switch (type) {
        case 1:
        case 11:
          Validate.isTrue(normalNexts.size() >= 1, "业务编号为[%s]的节点定义,其至少有一个（正常的）后续分支连线，请检查!!", nodeCode);
          break;
        case 5:
          Validate.isTrue(normalNexts == null || normalNexts.size() == 0, "业务编号为[%s]的节点定义,其不能有任何（正常的）后续分支连线，请检查!!", nodeCode);
          break;
        case 6:
        case 10:
          Validate.isTrue(normalNexts.size() >= 2, "业务编号为[%s]的节点定义,其至少有两个（正常的）后续分支连线，请检查!!", nodeCode);
          break;
        default:
          Validate.isTrue(normalNexts.size() == 1, "业务编号为[%s]的节点定义,其必须有一个（正常的）后续分支连线，请检查!!", nodeCode);
          break;
      }
      nexts.stream().forEach(nextItem -> {
        String conditions = nextItem.getConditions();
        String exceptions = nextItem.getExceptions();
        Integer lineType = nextItem.getLineType();
        Integer sort = nextItem.getSort() == null ? 100 : nextItem.getSort();
        String toNodeCode = nextItem.getToNodeCode();
        String bindParam = nextItem.getBindParam();
        RuleNodeEntity toNode = nodeCodeMapping.get(toNodeCode);
        Validate.notNull(toNode, "指定的目标定义节点[%s]未被发现，请检查!!", toNodeCode);
        // 在一个规则定义中，一个定义节点到另一个定义节点，只能有一条连线
        RuleNodeNextsEntity exsitNext = this.ruleNodeNextsService.findByFromAndTo(fromNodeCode, toNodeCode);
        Validate.isTrue(exsitNext == null, "指定的连线从[%s]到[%s]已经存在，不能重复连接，请检查!!", fromNodeCode, toNodeCode);
        // 开始进行添加
        RuleNodeNextsEntity ruleNext = new RuleNodeNextsEntity();
        ruleNext.setConditions(conditions);
        ruleNext.setExceptions(exceptions);
        ruleNext.setFromNode(fromNode);
        ruleNext.setToNode(toNode);
        ruleNext.setLineType(lineType);
        ruleNext.setSort(sort);
        ruleNext.setBindParam(bindParam);
        this.ruleNodeNextsService.create(ruleNext);
      });
    });
  }

  private RuleDefinitionEntity create(RuleDefinitionEntity ruleDefinitionEntity) {
    // 创建者、修改者信息
    String account = this.findCreator();
    ruleDefinitionEntity.setCreateAccount(account);
    ruleDefinitionEntity.setCreateTime(new Date());
    ruleDefinitionEntity.setModifyAccount(account);
    ruleDefinitionEntity.setModifyTime(new Date());
    ruleDefinitionEntity.setProjectName(platformContext.getAppName());

    RuleDefinitionEntity current = this.createForm(ruleDefinitionEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  private RuleDefinitionEntity createForm(RuleDefinitionEntity ruleDefinitionEntity) {
    this.createValidation(ruleDefinitionEntity);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    this.ruleDefinitionEntityRepository.save(ruleDefinitionEntity);
    if (ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionModify(ruleDefinitionEntity));
    }

    // 返回最终处理的结果，里面带有详细的关联信息
    return ruleDefinitionEntity;
  }

  /**
   * 重复的代码，进行私有方法封装，文件相对路径
   *
   * @param scriptContent
   * @return
   */
  private String saveScriptContent() {
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String relativePath = StringUtils.join("/ruleengine/", folderName, "/", (new Random().nextInt(100) % 10));
    return relativePath;
  }

  /**
   * 在创建一个新的RuleDefinitionEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RuleDefinitionEntity ruleDefinitionEntity) {
    Validate.notNull(ruleDefinitionEntity, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(ruleDefinitionEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    ruleDefinitionEntity.setId(null);
    Validate.notBlank(ruleDefinitionEntity.getCreateAccount(), "添加信息时，创建人账号不能为空！");
    Validate.notNull(ruleDefinitionEntity.getCreateTime(), "添加信息时，创建时间不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getModifyAccount(), "添加信息时，更新人账号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getCode(), "添加信息时，规则定义业务编号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getCverion(), "添加信息时，规则定义版本不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getDesc(), "添加信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateFilename(), "添加信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateRelativepath(), "添加信息时，前端图形化展示结构描述文件的相对路径不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(ruleDefinitionEntity.getCreateAccount() == null || ruleDefinitionEntity.getCreateAccount().length() < 255, "创建人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getModifyAccount() == null || ruleDefinitionEntity.getModifyAccount().length() < 255, "更新人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getCode() == null || ruleDefinitionEntity.getCode().length() < 128, "规则定义业务编号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getCverion() == null || ruleDefinitionEntity.getCverion().length() < 128, "规则定义版本,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getDesc() == null || ruleDefinitionEntity.getDesc().length() < 512, "规则定义用途说明,在进行添加时填入值超过了限定长度(512)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateFilename() == null || ruleDefinitionEntity.getTemplateFilename().length() < 256, "规则定义用途说明,在进行添加时填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateRelativepath() == null || ruleDefinitionEntity.getTemplateRelativepath().length() < 256, "前端图形化展示结构描述文件的相对路径,在进行添加时填入值超过了限定长度(256)，请检查!");
  }

  @Override
  @Transactional
  public RuleDefinitionEntity update(RuleDefinitionEntity definition, JSONObject layoutContent) {
    Validate.notNull(definition, "修改规则定义时，定义信息必须传入");
    Validate.notNull(layoutContent, "规则定义结构内容必须传入!!");
    byte[] fileContent;
    try {
      fileContent = layoutContent.toString().getBytes("UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e);
    }
    Validate.isTrue(fileContent != null && fileContent.length > 0, "修改规则定义时，页面布局文件必须传入");
    String desc = definition.getDesc();
    String definitionId = definition.getId();
    Validate.notBlank(definitionId, "修改规则定义时，其id信息必须传入!!");
    Validate.notBlank(desc, "修改规则定义时，其desc描述信息不能没有值!!");
    RuleDefinitionEntity exsitRuleDefinition = this.findById(definitionId);
    Validate.notNull(exsitRuleDefinition, "修改规则定义时，指定的规则定义未找到[%s]，请重新设定", definitionId);

    Set<RuleNodeEntity> nodes = definition.getNodes();
    // 确认只有一个开始节点，有一个或者多个结束节点
    long startNodeCount = nodes.stream().filter(item -> item.getType() == 4).count();
    long endNodeCount = nodes.stream().filter(item -> item.getType() == 5).count();
    Validate.isTrue(startNodeCount == 1, "规则定义中必须有且只有一个开始节点，请检查!!");
    Validate.isTrue(endNodeCount >= 1, "规则定义中必须至少有一个结束节点，请检查!!");

    /*
     * 规则定义的全量修改，可以简单理解为：
     * 1、根据指定的id进行参数定义、连线定义、节点定义的删除
     * 2、在进行规则定义基本信息的修改后
     * 3、再重新进行节点定义、参数定义信息的添加操作
     * （第3步进行时，需要清除掉由调用者传入的各种关联信息的id信息）
     * 4、最后重新进行连线定义信息的添加操作
     * */

    // 1、======
    Set<RuleNodeEntity> ruleNodes = this.ruleNodeService.findDetailsByDefinition(definitionId);
    if (!CollectionUtils.isEmpty(ruleNodes)) {
      Set<String> ruleNodeIds = ruleNodes.stream().map(RuleNodeEntity::getId).collect(Collectors.toSet());
      // 删除可能的扩展信息（锁组件扩展、A/B并行分支扩展、计时组件扩展）
      this.ruleLockNodeService.deleteByIds(ruleNodeIds.toArray(new String[]{}));
      this.ruleConcurrentAggregationService.deleteByIds(ruleNodeIds.toArray(new String[]{}));
      this.ruleTimerNodeService.deleteByIds(ruleNodeIds.toArray(new String[]{}));
      // 进行所有明细信息的删除
      this.ruleNodeParamsService.deleteByRuleNodes(ruleNodeIds.toArray(new String[]{}));
      // 进行所有连线的删除
      this.ruleNodeNextsService.deleteByFromNodes(ruleNodeIds.toArray(new String[]{}));
      // 删除定义下所有的节点信息
      this.ruleNodeService.deleteByIds(ruleNodeIds.toArray(new String[]{}));
    }

    // 2、======
    // 确认文件信息
    String relativePath = this.saveScriptContent();
    String fileName = StringUtils.join(exsitRuleDefinition.getCode(), "_", exsitRuleDefinition.getCverion(), ".txt");
    this.venusFileService.saveFile(relativePath, fileName, fileName, fileContent);
    definition.setTemplateFilename(fileName);
    definition.setTemplateRelativepath(relativePath);
    this.update(definition);

    // 3、======
    Map<String, RuleNodeEntity> nodeCodeMapping = new HashMap<>();
    nodes.stream().forEach(node -> {
      node.setId(null);
      if (node.getInputs() != null) {
        node.getInputs().stream().forEach(item -> item.setId(null));
      }
      if (node.getOutputs() != null) {
        node.getOutputs().stream().forEach(item -> item.setId(null));
      }
      if (node.getNexts() != null) {
        node.getNexts().stream().forEach(item -> item.setId(null));
      }
      if (node.getRuleAggregationExt() != null) {
        node.getRuleAggregationExt().setId(null);
      }
      if (node.getRuleLockExt() != null) {
        node.getRuleLockExt().setId(null);
      }
      if (node.getRuleTimerExt() != null) {
        node.getRuleTimerExt().setId(null);
      }

      // 处理节点基本信息
      this.validateAndSaveRuleNode(definition, node, nodeCodeMapping);
      // 处理节点参数信息
      this.validateAndSaveRuleNodeParams(definition, node, nodeCodeMapping);
    });

    // 4、======
    this.validateAndSaveNexts(nodes, nodeCodeMapping);
    if (ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionModify(definition));
    }
    return definition;
  }

  private RuleDefinitionEntity update(RuleDefinitionEntity ruleDefinitionEntity) {
    // 修改者信息
    String account = this.findModifyor();
    ruleDefinitionEntity.setModifyAccount(account);
    ruleDefinitionEntity.setModifyTime(new Date());

    RuleDefinitionEntity current = this.updateForm(ruleDefinitionEntity);
    return current;
  }

  private RuleDefinitionEntity updateForm(RuleDefinitionEntity ruleDefinitionEntity) {
    /*
     * 注意：修改时规则定义的基本信息只能修改描述信息
     * code、cversion等信息不能进行修改
     * */

    this.updateValidation(ruleDefinitionEntity);
    // ===================基本信息
    String currentId = ruleDefinitionEntity.getId();
    Optional<RuleDefinitionEntity> op_currentRuleDefinitionEntity = this.ruleDefinitionEntityRepository.findById(currentId);
    RuleDefinitionEntity currentRuleDefinitionEntity = op_currentRuleDefinitionEntity.orElse(null);
    Validate.notNull(currentRuleDefinitionEntity, "未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentRuleDefinitionEntity.setModifyAccount(ruleDefinitionEntity.getModifyAccount());
    currentRuleDefinitionEntity.setModifyTime(ruleDefinitionEntity.getModifyTime());
    currentRuleDefinitionEntity.setDesc(ruleDefinitionEntity.getDesc());
    currentRuleDefinitionEntity.setTemplateFilename(ruleDefinitionEntity.getTemplateFilename());
    currentRuleDefinitionEntity.setTemplateRelativepath(ruleDefinitionEntity.getTemplateRelativepath());
    currentRuleDefinitionEntity.setReturnParam(ruleDefinitionEntity.getReturnParam());

    this.ruleDefinitionEntityRepository.saveAndFlush(currentRuleDefinitionEntity);
    return currentRuleDefinitionEntity;
  }

  /**
   * 在更新一个已有的RuleDefinitionEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RuleDefinitionEntity ruleDefinitionEntity) {
    Validate.isTrue(!StringUtils.isBlank(ruleDefinitionEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(ruleDefinitionEntity.getModifyAccount(), "修改信息时，更新人账号不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getDesc(), "修改信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateFilename(), "修改信息时，规则定义用途说明不能为空！");
    Validate.notBlank(ruleDefinitionEntity.getTemplateRelativepath(), "修改信息时，前端图形化展示结构描述文件的相对路径不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(ruleDefinitionEntity.getModifyAccount() == null || ruleDefinitionEntity.getModifyAccount().length() < 255, "更新人账号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getDesc() == null || ruleDefinitionEntity.getDesc().length() < 512, "规则定义用途说明,在进行修改时填入值超过了限定长度(512)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateFilename() == null || ruleDefinitionEntity.getTemplateFilename().length() < 256, "规则定义用途说明,在进行修改时填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(ruleDefinitionEntity.getTemplateRelativepath() == null || ruleDefinitionEntity.getTemplateRelativepath().length() < 256, "前端图形化展示结构描述文件的相对路径,在进行修改时填入值超过了限定长度(256)，请检查!");
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    RuleDefinitionEntity current = this.findById(id);
    if (current != null) {
      this.ruleDefinitionEntityRepository.delete(current);
    }
    if (ruleDefinitionListeners != null) {
      ruleDefinitionListeners.forEach(item -> item.onDefinitionDelete(current));
    }
  }

  @Override
  public RuleDefinitionEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.ruleDefinitionEntityRepository.findDetailsById(id);
  }

  @Override
  public RuleDefinitionEntity findDetailsByCodeAndVersion(String code, String cversion) {
    if (StringUtils.isBlank(code) || StringUtils.isBlank(cversion)) {
      return null;
    }
    return this.ruleDefinitionEntityRepository.findDetailsByCodeAndVersion(code, cversion);
  }

  /**
   * 按照版本和业务编号查询详情
   *
   * @param code     指定的业务编号
   * @param cversion 指定的版本信息
   * @return
   */
  public RuleDefinitionEntity findByCodeAndVersion(String code, String cversion) {
    if (StringUtils.isBlank(code) || StringUtils.isBlank(cversion)) {
      return null;
    }
    return this.ruleDefinitionEntityRepository.findByCodeAndVersion(code, cversion);
  }

  @Override
  public RuleDefinitionEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<RuleDefinitionEntity> op = ruleDefinitionEntityRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public Page<RuleDefinitionEntity> findByConditions(Pageable pageable, String code, String desc) {
    Pageable cPageable = pageable;
    if (cPageable == null) {
      cPageable = PageRequest.of(0, 50);
    }
    return this.ruleDefinitionEntityRepository.findByConditions(cPageable, code, desc);
  }

  @Override
  public List<RuleDefinitionEntity> findAll() {
    return this.ruleDefinitionEntityRepository.findAll(Sort.by("createTime"));
  }

  /**
   * 按照给定的模板编码和版本，查询模板文件内容。
   *
   * @param code
   * @param version
   * @return
   */
  @Override
  public String findContentByCodeAndVersion(String code, String version) {
    if (StringUtils.isBlank(code) || StringUtils.isBlank(version)) {
      return null;
    }
    RuleDefinitionEntity ruleDefinition = ruleDefinitionEntityRepository.findByCodeAndVersion(code, version);
    if (ruleDefinition == null) {
      return null;
    }
    byte[] content = venusFileService.readFileContent(ruleDefinition.getTemplateRelativepath(), ruleDefinition.getTemplateFilename());
    if (content == null || content.length == 0) {
      return null;
    }
    return new String(content, StandardCharsets.UTF_8);
  }

  @Override
  public RuleDefinitionEntity updateVersion(RuleDefinitionEntity definition, String newVersion, JSONObject layoutContent) {
    Validate.notNull(definition, "规则定义为空，请检查！");
    Validate.notBlank(definition.getCode(), "规则定义编号为空，请检查！");
    Validate.notBlank(definition.getCverion(), "规则定义版本为空，请检查！");
    Validate.notBlank(newVersion, "规则定义升级版本号为空，请检查！");
    Validate.isTrue(!definition.getCverion().equals(newVersion), "升级版本号与原版本号相同，请检查！");
    RuleDefinitionEntity ruleDefinitionEntity = ruleDefinitionEntityRepository.findByCodeAndVersion(definition.getCode(), definition.getCverion());
    Validate.notNull(ruleDefinitionEntity, "原升级规则定义为空，请检查！");
    //设置新版本信息
    definition.setId(null);
    definition.setCverion(newVersion);
    RuleDefinitionEntity result = this.create(definition, layoutContent);
    return result;
  }
} 
