package com.bizunited.platform.mars.service.internal;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.mars.entity.RuleTemplateGroupEntity;
import com.bizunited.platform.mars.repository.RuleTemplateGroupEntityRepository;
import com.bizunited.platform.mars.service.RuleTemplateGroupService;
import com.bizunited.platform.mars.service.RuleTemplateNodeService;
import com.google.common.collect.Lists;

import java.lang.Override;
import java.lang.String;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * RuleTemplateGroupEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("RuleTemplateGroupEntityServiceImpl")
public class RuleTemplateGroupServiceImpl extends AbstractMarsSaveableService implements RuleTemplateGroupService { 
  @Autowired
  private RuleTemplateGroupEntityRepository ruleTemplateGroupEntityRepository;
  @Autowired
  private RuleTemplateNodeService ruleTemplateNodeEntityService;
  @Transactional
  @Override
  public List<RuleTemplateGroupEntity> create(Set<RuleTemplateGroupEntity> ruleTemplateGroupEntities) {
    Validate.notNull(ruleTemplateGroupEntities , "进行分组添加时，至少传入一个新的分组信息");
    List<RuleTemplateGroupEntity> results = Lists.newArrayList();
    for (RuleTemplateGroupEntity ruleTemplateGroupEntity : ruleTemplateGroupEntities) {
      // 创建者、修改者信息
      String account = this.findCreator();
      ruleTemplateGroupEntity.setCreateAccount(account);
      ruleTemplateGroupEntity.setCreateTime(new Date());
      ruleTemplateGroupEntity.setModifyAccount(account);
      ruleTemplateGroupEntity.setModifyTime(new Date());
      // 状态都是1：正常
      ruleTemplateGroupEntity.setGroupStatus(1);
      
      RuleTemplateGroupEntity current = this.createForm(ruleTemplateGroupEntity);
      results.add(current);
    }
    return results;
  } 
  @Transactional
  @Override
  public RuleTemplateGroupEntity createForm(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    this.createValidation(ruleTemplateGroupEntity);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    ruleTemplateGroupEntity.setProjectName(ApplicationContextUtils.getProjectName());
    this.ruleTemplateGroupEntityRepository.save(ruleTemplateGroupEntity);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return ruleTemplateGroupEntity;
  }
  /**
   * 在创建一个新的RuleTemplateGroupEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    Validate.notNull(ruleTemplateGroupEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(ruleTemplateGroupEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    ruleTemplateGroupEntity.setId(null);
    Validate.notBlank(ruleTemplateGroupEntity.getCreateAccount(), "添加信息时，创建人账号不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getCreateTime(), "添加信息时，创建时间不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getModifyAccount(), "添加信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getCode(), "添加信息时，分组编号编号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getName(), "添加信息时，模板分组名称（中文）不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getGroupStatus(), "添加信息时，分组可用状态不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getType(), "添加信息时，分组类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(ruleTemplateGroupEntity.getCreateAccount() == null || ruleTemplateGroupEntity.getCreateAccount().length() < 255 , "创建人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getModifyAccount() == null || ruleTemplateGroupEntity.getModifyAccount().length() < 255 , "更新人账号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getCode() == null || ruleTemplateGroupEntity.getCode().length() < 128 , "分组编号编号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getName() == null || ruleTemplateGroupEntity.getName().length() < 128 , "模板分组名称（中文）,在进行添加时填入值超过了限定长度(128)，请检查!");
    RuleTemplateGroupEntity currentRuleTemplateGroupEntity = this.findByCode(ruleTemplateGroupEntity.getCode());
    Validate.isTrue(currentRuleTemplateGroupEntity == null, "分组编号编号已存在,请检查");
  }
  @Transactional
  @Override
  public List<RuleTemplateGroupEntity> update(Set<RuleTemplateGroupEntity> ruleTemplateGroupEntities) {
    Validate.notNull(ruleTemplateGroupEntities , "进行分组修改时，至少传入一个已有的分组信息");
    List<RuleTemplateGroupEntity> results = Lists.newArrayList();
    for (RuleTemplateGroupEntity ruleTemplateGroupEntity : ruleTemplateGroupEntities) {
      // 修改者信息
      String account = this.findModifyor();
      ruleTemplateGroupEntity.setModifyAccount(account);
      ruleTemplateGroupEntity.setModifyTime(new Date());
      RuleTemplateGroupEntity current = this.updateForm(ruleTemplateGroupEntity);
      results.add(current);
    }
    
    return results;
  } 
  @Transactional
  @Override
  public RuleTemplateGroupEntity updateForm(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    /* 
     * 只能对状态、名称进行修改。分组的业务编号、支持的组件类型不能修改
     * */
    
    this.updateValidation(ruleTemplateGroupEntity);
    // ===================基本信息
    String currentId = ruleTemplateGroupEntity.getId();
    Optional<RuleTemplateGroupEntity> op_currentRuleTemplateGroupEntity = this.ruleTemplateGroupEntityRepository.findById(currentId);
    RuleTemplateGroupEntity currentRuleTemplateGroupEntity = op_currentRuleTemplateGroupEntity.orElse(null);
    currentRuleTemplateGroupEntity = Validate.notNull(currentRuleTemplateGroupEntity ,"未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentRuleTemplateGroupEntity.setModifyAccount(ruleTemplateGroupEntity.getModifyAccount());
    currentRuleTemplateGroupEntity.setModifyTime(ruleTemplateGroupEntity.getModifyTime());
    currentRuleTemplateGroupEntity.setName(ruleTemplateGroupEntity.getName());
    currentRuleTemplateGroupEntity.setGroupStatus(ruleTemplateGroupEntity.getGroupStatus());
    
    this.ruleTemplateGroupEntityRepository.saveAndFlush(currentRuleTemplateGroupEntity);
    return currentRuleTemplateGroupEntity;
  }
  /**
   * 在更新一个已有的RuleTemplateGroupEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RuleTemplateGroupEntity ruleTemplateGroupEntity) { 
    Validate.isTrue(!StringUtils.isBlank(ruleTemplateGroupEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(ruleTemplateGroupEntity.getModifyAccount(), "修改信息时，更新人账号不能为空！");
    Validate.notBlank(ruleTemplateGroupEntity.getName(), "修改信息时，模板分组名称（中文）不能为空！");
    Validate.notNull(ruleTemplateGroupEntity.getGroupStatus(), "修改信息时，分组可用状态不能为空！");
    
    // 重复性判断，基本属性，需要满足unique = true
    RuleTemplateGroupEntity currentForCode = this.findByCode(ruleTemplateGroupEntity.getCode());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , ruleTemplateGroupEntity.getId()) , "分组编号编号已存在,请检查"); 
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(ruleTemplateGroupEntity.getModifyAccount() == null || ruleTemplateGroupEntity.getModifyAccount().length() < 255 , "更新人账号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(ruleTemplateGroupEntity.getName() == null || ruleTemplateGroupEntity.getName().length() < 128 , "模板分组名称（中文）,在进行修改时填入值超过了限定长度(128)，请检查!");
  } 
  @Override
  public RuleTemplateGroupEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.ruleTemplateGroupEntityRepository.findDetailsById(id);
  }
  @Override
  public RuleTemplateGroupEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<RuleTemplateGroupEntity> op = ruleTemplateGroupEntityRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    RuleTemplateGroupEntity current = this.findById(id);
    // 如果当前分组下已经绑定了节点模板，则不允许删除
    Validate.isTrue(ruleTemplateNodeEntityService.countByTemplateGroup(id) == 0 , "当前要被删除的分组信息还存在关联的节点模板，请检查!!");
    if(current != null) { 
      this.ruleTemplateGroupEntityRepository.delete(current);
    }
  }
  @Override
  public RuleTemplateGroupEntity findByCode(String code) { 
    if(StringUtils.isBlank(code)) { 
      return null;
    }
    return this.ruleTemplateGroupEntityRepository.findByCode(code);
  }
  @Override
  public Set<RuleTemplateGroupEntity> findByGroupStatus(Integer groupStatus) {
    if(groupStatus == null) {
      return this.ruleTemplateGroupEntityRepository.findAllOrderByTypeAndNameAndProjectName(ApplicationContextUtils.getProjectName());
    } else {
      return this.ruleTemplateGroupEntityRepository.findByGroupStatusAndProjectName(groupStatus, ApplicationContextUtils.getProjectName());
    }
  }
  @Override
  public Set<RuleTemplateGroupEntity> findByTypeAndGroupStatus(Integer type ,Integer groupStatus) {
    if(type == null || type <= 0) {
      return null;
    }
    // 注意groupStatus可以不传入
    if(groupStatus == null) {
      return this.ruleTemplateGroupEntityRepository.findByTypeAndProjectName(type, ApplicationContextUtils.getProjectName());
    }
    return this.ruleTemplateGroupEntityRepository.findByTypeAndGroupStatusAndProjectName(type, groupStatus , ApplicationContextUtils.getProjectName());
  }
} 
