package com.bizunited.platform.mars.local.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.mars.entity.RuleDefinitionEntity;
import com.bizunited.platform.mars.service.RuleDefinitionService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

/**
 * RuleDefinitionEntity业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@RequestMapping("/v1/mars/ruleDefinitions")
public class RuleDefinitionController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RuleDefinitionController.class);
  @Autowired
  private RuleDefinitionService ruleDefinitionEntityService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @ApiOperation(value = "针对一个新的规则定义，进行创建操作。注意：创建操作时，规则定义的基本信息和附属信息，都不能包含ID信息，具体的信息结构可参见文档示例")
  @PostMapping(value = "")
  public ResponseModel create(@RequestParam("runtimeDefinition") @ApiParam(name = "runtimeDefinition", value = "符合前后端规则定义数据结构描述的json信息") String runtimeDefinitionJson,
                              @RequestParam("layoutContent") @ApiParam(name = "layoutContent", value = "前端布局文件排版信息") String layoutContent,
                              HttpServletRequest request) {
    // 构建可能的params信息
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        if ("runtimeDefinition".equals(name) || "layoutContent".equals(name)) {
          continue;
        }
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }
    try {
      JSONObject jsonObject = JSON.parseObject(runtimeDefinitionJson);
      RuleDefinitionEntity definition = JSON.toJavaObject(jsonObject, RuleDefinitionEntity.class);
      definition = this.ruleDefinitionEntityService.create(definition, JSONObject.parseObject(layoutContent),params);
      return this.buildHttpResultW(definition, new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "针对一个已有的规则定义，进行该规则定义的全量修改。所谓全量修改，就是之前的规则定义信息会全部作废")
  @PatchMapping(value = "")
  public ResponseModel update(@RequestParam("runtimeDefinition") @ApiParam(name = "runtimeDefinition", value = "符合前后端规则定义数据结构描述的json信息") String runtimeDefinitionJson,
                              @RequestParam("layoutContent") @ApiParam(name = "layoutContent", value = "前端布局文件排版信息") String layoutContent,
                              HttpServletRequest request) {
    // 构建可能的params信息
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        if ("runtimeDefinition".equals(name) || "layoutContent".equals(name)) {
          continue;
        }
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }
    try {
      RuleDefinitionEntity definition = JSON.toJavaObject(JSON.parseObject(runtimeDefinitionJson), RuleDefinitionEntity.class);
      definition = this.ruleDefinitionEntityService.update(definition, JSONObject.parseObject(layoutContent), params);
      return this.buildHttpResultW(definition, new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "通过给定的规则定义编号和版本信息，查询完整的规则描述。规则描述信息将基于持久层存储的规则定义")
  @RequestMapping(value = "/findDetailsByCodeAndVersion", method = {RequestMethod.GET})
  public ResponseModel findDetailsByCodeAndVersion(@RequestParam("code") @ApiParam(name = "code", value = "前端布局文件排版信息") String code,
                                                   @RequestParam("version") @ApiParam(name = "version", value = "前端布局文件排版信息") String version) {
    try {
      RuleDefinitionEntity result = this.ruleDefinitionEntityService.findDetailsByCodeAndVersion(code, version);
      // 注意，由于涉及对象循环引用的问题，所以需要使用中考来解决
      RuleDefinitionEntity target = this.nebulaToolkitService.copyObjectByWhiteList(result, RuleDefinitionEntity.class, LinkedHashSet.class, ArrayList.class, new String[]{"nodes", "nodes.templateNode", "nodes.ruleLockExt",
              "nodes.ruleAggregationExt", "nodes.ruleTimerExt",
              "nodes.inputs", "nodes.outputs", "nodes.nexts",
              "nodes.nexts.fromNode", "nodes.nexts.toNode"});
      return this.buildHttpResult(target);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照给定的模板编码和版本，查询模板文件内容。
   *
   * @param code
   * @param version
   */
  @ApiOperation(value = "按照给定的模板编码和版本，查询模板文件内容。")
  @GetMapping("/findContentByCodeAndVersion")
  public ResponseModel findContentByCodeAndVersion(@ApiParam(name = "code", value = "模板编码", required = true) @RequestParam("code") String code,
                                                   @ApiParam(name = "version", value = "版本", required = true) @RequestParam("version") String version) {
    try {
      String content = ruleDefinitionEntityService.findContentByCodeAndVersion(code, version);
      return buildHttpResultW(content);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 分页查询当前已建立的所有规则定义信息。
   *
   * @param pageable 分页要求
   * @param code     规则定义编号
   * @param desc     规则定义描述
   */
  @ApiOperation(value = "分页查询当前已建立的所有规则定义信息")
  @RequestMapping(value = "/findByConditions", method = {RequestMethod.GET})
  public ResponseModel findByConditions(@PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable,
                                        @RequestParam(name = "code", required = false) @ApiParam(value = "规则定义编号（可以不传入）", required = false) String code,
                                        @RequestParam(name = "desc", required = false) @ApiParam(value = "规则定义描述（可以不传入）", required = false) String desc) {
    try {
      Page<RuleDefinitionEntity> result = this.ruleDefinitionEntityService.findByConditions(pageable, code, desc);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询当前已创建的所有规则定义基本信息，并按照创建时间进行排序
   *
   * @return
   */
  @ApiOperation(value = "查询当前已创建的所有规则定义基本信息，并按照创建时间进行排序")
  @GetMapping(value = "/findAll")
  public ResponseModel findAll() {
    try {
      List<RuleDefinitionEntity> result = this.ruleDefinitionEntityService.findAll();
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 规则定义版本升级
   *
   * @return
   */
  @ApiOperation(value = "版本升级")
  @PostMapping(value = "/updateVersion")
  public ResponseModel updateVersion(@RequestParam("runtimeDefinition") @ApiParam(name = "runtimeDefinition", value = "符合前后端规则定义数据结构描述的json信息") String runtimeDefinitionJson,
                                     @RequestParam("layoutContent") @ApiParam(name = "layoutContent", value = "前端布局文件排版信息") String layoutContent,
                                     @RequestParam("newVersion") String newVersion) {
    try {
      JSONObject jsonObject = JSON.parseObject(runtimeDefinitionJson);
      RuleDefinitionEntity definition = JSON.toJavaObject(jsonObject, RuleDefinitionEntity.class);
      definition = this.ruleDefinitionEntityService.updateVersion(definition, newVersion, JSONObject.parseObject(layoutContent));
      return this.buildHttpResultW(definition, new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}