package com.bizunited.platform.mars.policy.process.rule.waiter;

import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeParallelBranchNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.rule.starter.StarterRuleable;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

/**
 * 可用于A/B测试的并行分支
 *
 * @author yinwenjie
 */
@Component("_ParallelBranchRuleable")
public class ParallelBranchRuleable<T extends RuntimeParallelBranchNode> implements StarterRuleable<RuntimeParallelBranchNode> {

  private static final Logger LOGGER = LoggerFactory.getLogger(ParallelBranchRuleable.class);

  /**
   * 判定集合大小同步或异步执行数字
   */
  @Value("${mars.branch.judge-size:1000}")
  private Integer judgeSize;

  @Autowired
  private ParallelBranchTask parallelBranchTask;

  @Override
  public int getType() {
    return RuntimeNodeType.MERGE.getValue();
  }

  @Override
  public void doProcess(RuntimeParallelBranchNode currentNode, RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked, ProcessorChain processChain) {
    processChain.doProcessNode(context, runtimeProcessorLinked);
  }

  @Override
  public Set<RuntimeProcessorLinked> createProcessorLinkeds(RuntimeParallelBranchNode currentNode, RuleRuntimeContext context) {
    /*
     * 在接收到继续的通知，正式开始继续执行前，ParallelBranchRuleable节点会依次检查所有的条件分支，
     * 找到所有匹配条件的分支，并为这些分支的后续执行创建多个规则链路的初始状态，并向下执行
     *
     * 1、链路检查
     *  1.1、是否存在后续节点数据
     *  1.2、判断当前节点是否为A/B并行节点
     *  1.3、当前节点参数是否有效(集合或者数据类型)
     * 2、根据序号进行依次判定
     * 3、通过集合数据判定同步或异步并行执行数据判断与拆分
     *  3.1、异步执行拆分数据
     *  3.2、同步执行拆分数据
     * 4、根据各分支获得下一节点运行时信息，构建规则链路分支
     * */

    // 1、链路检查
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    Validate.isTrue(CollectionUtils.isNotEmpty(nexts), "当前节点后续运行时连线数据为空，请检查！");
    RuntimeNodeType type = currentNode.getType();
    Validate.isTrue(type == RuntimeNodeType.CONCURRENCY, "当前节点不是A/B并发类型的运行时节点，请检查!!");
    Validate.notBlank(currentNode.getParamName(), "A/B并行节点参数名为空，请检查！");

    // 2、根据序号进行依次判定
    long noneConditionsSize = nexts.stream().filter(item -> StringUtils.isBlank(item.getConditions())).count();
    Validate.isTrue(noneConditionsSize <= 1, "A/B并行节点链线超过1条没有配置条件，请检查！");
    //当只有一条链线没有配置条件时，默认设置为其余条件的剩余数据
    String defaultCondition = null;
    if (noneConditionsSize == 1L) {
      StringBuilder condition = new StringBuilder();
      condition.append("!( ");
      condition.append(nexts.stream().filter(item -> StringUtils.isNoneBlank(item.getConditions())).map(next -> StringUtils.appendIfMissing(StringUtils.prependIfMissing(next.getConditions(),"("),")")).collect(Collectors.joining(" || ")));
      condition.append(" )");
      defaultCondition = condition.toString();
    }

    Set<RuntimeNodeNexts> sortNexts = nexts.stream().filter(item -> item.getLineType() == 1).sorted((source, target) -> source.getSort() - target.getSort()).collect(Collectors.toSet());
    Set<RuntimeProcessorLinked> runtimeProcessorLinkeds = Sets.newHashSet();
    List<?> params = getParams(context, currentNode.getParamName());
    Validate.notNull(params, "A/B并行节点集合数据为空，请检查");
    int size = params.size();
    // 3、判定同步或异步执行
    if (size > judgeSize) {
      LOGGER.debug("异步执行【{}】,数据集合大小【{}】", currentNode.getCode(), size);
      List<Future<RuntimeProcessorLinked>> futures = Lists.newArrayList();
      // 3.1、异步并行执行数据判断与拆分
      for (RuntimeNodeNexts next : sortNexts) {
        if (StringUtils.isBlank(next.getConditions())) {
          next.setConditions(defaultCondition);
        }
        Future<RuntimeProcessorLinked> future = parallelBranchTask.doAsyncParallelBranchTask(currentNode, context, next, params);
        futures.add(future);
      }
      try {
        // 4、根据各分支获得下一节点运行时信息，构建规则链路分支
        for (Future<RuntimeProcessorLinked> future : futures) {
          runtimeProcessorLinkeds.add(future.get());
        }
      } catch (InterruptedException | ExecutionException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException("A/B并行分支异步执行异常,请检查！");
      }
    } else {
      LOGGER.debug("同步执行【{}】,数据集合大小【{}】", currentNode.getCode(), size);
      // 3.2、同步执行数据判断与拆分
      for (RuntimeNodeNexts next : sortNexts) {
        if (StringUtils.isBlank(next.getConditions())) {
          next.setConditions(defaultCondition);
        }
        runtimeProcessorLinkeds.add(parallelBranchTask.doSyncParallelBranchTask(currentNode, context, next, params));
      }
    }
    return runtimeProcessorLinkeds;
  }

  /**
   * 从上下文中获取指定参数并验证有效性（集合或者数组）转型为List
   *
   * @param context   运行时上下文
   * @param paramName 参数名
   * @return
   */
  private List<?> getParams(RuleRuntimeContext context, String paramName) {
    Object params = context.getParams().get(paramName);
    Validate.notNull(params, "A/B并行节点上下文参数为空,请检查!");
    // 验证参数是否为集合或者数组
    Validate.isTrue(params.getClass().isArray() || Collection.class.isAssignableFrom(params.getClass()), "制定参数不是集合或者数组，请检查！");
    List<?> paramList = null;
    if (params.getClass().isArray()) {
      Object[] arr = (Object[]) params;
      paramList = Arrays.asList(arr);
    } else if (Collection.class.isAssignableFrom(params.getClass())) {
      Collection<?> collection = (Collection<?>) params;
      paramList = Lists.newLinkedList(collection);
    }
    return paramList;
  }

  @Override
  public Class<RuntimeParallelBranchNode> mapping() {
    return RuntimeParallelBranchNode.class;
  }
}
