package com.bizunited.platform.mars.policy.process.rule.logic;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeParams;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.logic.RuntimeDynamicScriptNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeDefinitionService;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeNodeService;


/**
 * 基于全动态脚本的逻辑节点处理组件
 * @author yinwenjie
 */
@Component("dynamicGroovyScriptProcessRuleable")
public class DynamicGroovyScriptProcessRuleable<T extends RuntimeDynamicScriptNode> extends AbstractLogicRuleable<RuntimeDynamicScriptNode> implements LogicRuleable<RuntimeDynamicScriptNode> {
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  @Autowired
  private ScriptService scriptService;
  /**
   * 代表执行内容的入参名
   */
  private static final String SCRIPT_CONTENT_PARAM = "_script";
  
  @Override
  public void doProcess(RuntimeDynamicScriptNode currentNode, RuleRuntimeContext context,RuntimeProcessorLinked runtimeProcessorLinked, ProcessorChain processChain) {
    super.process(currentNode, runtimeProcessorLinked);
    processChain.doProcessNode(context, runtimeProcessorLinked);
  }

  @Override
  protected void execute(RuntimeDynamicScriptNode currentNode, Map<String, Object> inputParamValues, RuleRuntimeContext context) {
    Validate.notNull(currentNode , "在进行节点定义执行时，必须要有至少一个节点定义RuntimeNode信息，请检查入参!!");
    String code = currentNode.getCode();
    Validate.notBlank(code , "在进行节点定义执行时，节点定义的code必须有值，请检查入参!!");
    // 取得并判定要执行的脚本（使用的固定入参是script）
    Object scriptContext = inputParamValues.get(SCRIPT_CONTENT_PARAM);
    Validate.notNull(scriptContext , "当前正在运行的节点[%s]，是一个动态脚本节点，但是并没有发现入参名为_script的脚本代码内容，请检查!!" , code);
    String scriptContextValue = scriptContext.toString();
    Validate.notBlank(scriptContextValue , "当前正在运行的节点[%s]，是一个动态脚本节点，但是并没有发现入参名为_script的脚本代码内容，请检查!!" , code);
    
    // 2、======
    // 将当前上下文中的所有K-V信息，作为当前脚本执行的入参
    Map<String , Object> contentMap = context.getParams();
    Map<String , Object> scriptInputsMap = new HashMap<>(contentMap);
    scriptInputsMap.put("runtimeNodeService", runtimeNodeService);
    scriptInputsMap.put("runtimeDefinitionService", runtimeDefinitionService);
    Object _returnValue = this.scriptService.invoke(scriptContextValue, scriptInputsMap);
    
    // 3、======
    // 根据设定的出参信息和上下问进行交互
    // 注意：当前出参如果设定了_return，则将调用返回值设定到上下文中
    Set<RuntimeNodeParams> outputParams = currentNode.getOutputs();
    if(outputParams != null) {
      for(RuntimeNodeParams item : outputParams) {
        boolean nullable = item.getNullable();
        String contextParamName = item.getContextParamName();
        String templateParamName = item.getTemplateParamName();
        if(StringUtils.equals(templateParamName, "_return")) {
          contentMap.put(contextParamName, _returnValue);
          continue;
        }
        Object variableValue = scriptInputsMap.get(templateParamName);
        if(variableValue == null && nullable) {
          String errorMsg = String.format("分析运行节点[%s]返回值时，发现指定的出参[%s]设定为not nullable，但是执行结果却没有返回值，请检查", code , templateParamName);
          throw new IllegalArgumentException(errorMsg);
        }
        contentMap.put(contextParamName, variableValue);
      }
    }
  }

  @Override
  public Class<RuntimeDynamicScriptNode> mapping() {
    return RuntimeDynamicScriptNode.class;
  }
}