package com.bizunited.platform.mars.policy.process.executor;

import java.util.List;
import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.locks.LockSupport;

import com.bizunited.platform.mars.policy.process.cache.ext.RuntimeConditionExtNode;
import com.bizunited.platform.mars.policy.process.cache.ext.RuntimeParallelBranchExtNode;
import com.bizunited.platform.mars.policy.process.rule.ext.AbstractConditionExtRuleable;
import com.bizunited.platform.mars.policy.process.rule.ext.AbstractParallelBranchExtRuleable;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.mars.policy.process.cache.RuntimeNode;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.ext.RuntimeExtNode;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeDelayNode;
import com.bizunited.platform.mars.policy.process.rule.Ruleable;
import com.bizunited.platform.mars.policy.process.rule.ext.AbstractExtRuleable;
import com.bizunited.platform.mars.policy.process.rule.starter.StarterRuleable;
import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeProcessorLinkedService;

/**
 * 这个线程运行器工作于mars_timer_thread_pool线程池中，
 * 主要作用是当某个继承了RuntimeDelayNode类的节点的已经到达阻塞等待时间，
 * 并且离开DelayQueue队列后，就由该线程运行器进行运行。</p>
 * 
 * 运行中的执行器，主要将进行 
 * @author yinwenjie 
 */
@Primary
@Component("TimerNotifyRunnable")
@Scope("prototype")
public class TimerNotifyRunnable implements Runnable {
  @Autowired
  @Qualifier("_delayTimerQueue")
  private DelayQueue<RuntimeDelayNode> delayQueue;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private RuntimeProcessorLinkedService runtimeProcessorLinkedService;
  @Autowired
  private List<Ruleable<RuntimeNode>> ruleables;
  /**
   * 业务扩展组件的ruleable处理器
   */
  @Autowired
  private List<Ruleable<? extends RuntimeNode>> extRuleables;
  
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(TimerNotifyRunnable.class);
  
  
  @Override
  public void run() {
    /* 
     * 这个线程，只要不是发现了线程终止信号（已知的场景就只可能是操作系统 kill process id）
     * 都会坚持循环执行
     * */
    Long count = 0l;
    while(count++ < Long.MAX_VALUE) {
      Thread currentThread = Thread.currentThread();
      // TODO 收到终止信号，就进行落盘处理，然后结束
      if(currentThread.isInterrupted()) {
        break;
      }
      
      RuntimeDelayNode currentNode = null;
      try {
        currentNode = this.delayQueue.take();
      } catch(InterruptedException e) {
        // TODO 在后续版本中，mars模块将再退出前完成未处理任务的落盘备份处理!!
        LOGGER.error("发现终止信号，请避免使用kill -9的方式强制终结进程，在后续版本中，mars模块将再退出前完成未处理任务的落盘备份处理!!");
        Thread.currentThread().interrupt();
        break;
      }
      String instanceId = currentNode.getInstanceId();
      RuleRuntimeContext ruleRuntimeContext = RuleRuntimeContext.getRuleRuntimeContext(instanceId);
      
      /*
       * 执行过程为：
       * 1、首先根据RuntimeDelayMapping映射信息，找到当前节点对应的运行器
       * 注意，由于这里都是继承了RuntimeDelayNode类的节点，所以其运行器一定实现了StarterRuleable接口
       * 否则会报错。
       * 
       * 2、接着调用StarterRuleable接口，以便根据不同的开始节点完成规则链路的创建
       * 由于这个开始节点可能存在多个链路同时执行的情况，所以本消息消费者，需要将执行过程送入到另一个线程池
       * （mars_extcutor_thread_pool）中进行执行
       * */
      try {
        this.doChain(currentNode, ruleRuntimeContext);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        ruleRuntimeContext.setCurrentThrowable(e);
        ruleRuntimeContext.setExceptionLevel(0);
        ruleRuntimeContext.setStatus(RuleRunTimeStatus.EXCEPTION);
        Thread requestThread = ruleRuntimeContext.getRequestThread();
        if(requestThread != null) {
          LockSupport.unpark(requestThread);
        }
      }
    }
  }
  
  @SuppressWarnings({"rawtypes", "unchecked"})
  public void doChain(RuntimeDelayNode currentNode , RuleRuntimeContext ruleRuntimeContext) {
    // 1、=====
    Ruleable currentRuleable = this.findRuleable(currentNode);
    Validate.notNull(currentRuleable , "没发现指定运行时节点[%s]，对应的执行器Runable，请检查!!" , currentNode.getClass().getName());
    Validate.isTrue(currentRuleable instanceof StarterRuleable , "当前节点并不是一种链路开始节点，请检查设定!!");
    
    // 2、=====
    StarterRuleable<RuntimeDelayNode> starterRuleable = (StarterRuleable<RuntimeDelayNode>)currentRuleable;
    Set<RuntimeProcessorLinked> runtimeProcessorLinkeds = starterRuleable.createProcessorLinkeds(currentNode, ruleRuntimeContext);
    if(CollectionUtils.isEmpty(runtimeProcessorLinkeds)) {
      LOGGER.debug("基于节点运行时[%]为开始节点的链路，未被建立；可能是因为未达到继续运行的触发条件");
      return;
    }
    
    // 放入真实的链路执行线程池，开始执行
    ThreadPoolExecutor marsExtcutorExecutor = applicationContext.getBean("mars_extcutor_thread_pool" , ThreadPoolExecutor.class);
    for (RuntimeProcessorLinked runtimeProcessorLinked : runtimeProcessorLinkeds) {
      this.runtimeProcessorLinkedService.start(ruleRuntimeContext, runtimeProcessorLinked);
      ProcessorLinkedChainRunnable processorLinkedChain = (ProcessorLinkedChainRunnable)applicationContext.getBean("ProcessorLinkedChainRunnable" , ruleRuntimeContext , runtimeProcessorLinked);
      marsExtcutorExecutor.submit(processorLinkedChain);
    }
  }

  /**
   * 该私有方法为指定的运行时节点找到对应的ruleable处理器
   *
   * @param currentNode
   * @return
   */
  @SuppressWarnings("unchecked")
  private Ruleable<RuntimeNode> findRuleable(RuntimeNode currentNode) {
    /*
     * 分为两种找到ruleable的方式，一种是通过Ruleable.mapping()方法的实现去找
     * 如果当type == 9 时，就是通过AbstractExtRuleable.mappingExtType()方法的返回值去找
     * */
    RuntimeNodeType runtimeNodeType = currentNode.getType();
    // 如果条件成立，则满足type == 9 业务定义节点的处理器转换
    if (runtimeNodeType == RuntimeNodeType.CUSTOM && currentNode instanceof RuntimeExtNode) {
      RuntimeExtNode runtimeExtNode = (RuntimeExtNode) currentNode;
      for (Ruleable<? extends RuntimeNode> item : extRuleables) {
        if (!AbstractExtRuleable.class.isAssignableFrom(item.getClass())) {
          continue;
        }
        AbstractExtRuleable extRuleable = (AbstractExtRuleable) item;
        int mappingExtType = extRuleable.mappingExtType();
        if (mappingExtType == runtimeExtNode.getExtType().intValue()) {
          return (Ruleable<RuntimeNode>) item;
        }
      }
    }
    // 如果条件成立，则满足type == 10 业务扩展A/B并节点的处理器转换
    else if (runtimeNodeType == RuntimeNodeType.CUSTOM_CONCURRENCY && currentNode instanceof RuntimeParallelBranchExtNode) {
      RuntimeParallelBranchExtNode runtimeParallelBranchExtNode = (RuntimeParallelBranchExtNode) currentNode;
      for (Ruleable<? extends RuntimeNode> item : extRuleables) {
        if (!AbstractParallelBranchExtRuleable.class.isAssignableFrom(item.getClass())) {
          continue;
        }
        AbstractParallelBranchExtRuleable parallelBranchExtRuleable = (AbstractParallelBranchExtRuleable) item;
        int mappingExtType = parallelBranchExtRuleable.mappingExtType();
        if (mappingExtType == runtimeParallelBranchExtNode.getExtType().intValue()) {
          return (Ruleable<RuntimeNode>) item;
        }
      }
    }
    // 如果条件成立，则满足type == 11 业务扩展条件判定节点的处理器转换
    else if (runtimeNodeType == RuntimeNodeType.CUSTOM_CONDITION && currentNode instanceof RuntimeConditionExtNode) {
      RuntimeConditionExtNode runtimeConditionExtNode = (RuntimeConditionExtNode) currentNode;
      for (Ruleable<? extends RuntimeNode> item : extRuleables) {
        if (!AbstractConditionExtRuleable.class.isAssignableFrom(item.getClass())) {
          continue;
        }
        AbstractConditionExtRuleable conditionExtRuleable = (AbstractConditionExtRuleable) item;
        int mappingExtType = conditionExtRuleable.mappingExtType();
        if (mappingExtType == runtimeConditionExtNode.getExtType().intValue()) {
          return (Ruleable<RuntimeNode>) item;
        }
      }
    }
    // 其它type时，进行此分支的执行
    else {
      for (Ruleable<RuntimeNode> item : ruleables) {
        if (item.mapping() == currentNode.getClass()) {
          return item;
        }
      }
    }
    return null;
  }
}