package com.bizunited.platform.mars.policy.process.cache.waiter;

import java.util.Date;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;

/**
 * 带有延迟等待效果的节点，例如所有可以作为规则链路起始标记的节点
 * @author yinwenjie
 *
 */
public abstract class RuntimeDelayNode extends RuntimeStartAble implements Delayed {
  /**
   * 从当前规则节点将要开始执行的时间开始计算，需要暂停的时间数（默认为毫秒）
   */
  private int timerCount = 0;
  /**
   * 如果存在延期，则该信息记录到期时间
   */
  private Long expireTime = null;
  /**
   * 当该节点运行时存入timer队列时，需要设置引用的运行时上下文，
   * 以便保证规则实例上下文不会因为弱引用的原因给GC掉
   */
  private RuleRuntimeContext ruleRuntimeContext;
  
  /**
   * 该方法指定当前延迟性质节点的具体处理节点是否真的存在延迟效果。</br>
   * 一旦来说继承了RuntimeDelayNode节点，但是又没有真实延迟效果的节点，其目的只是告诉执行器，当前节点之后可能存在执行分支，
   * 或者当前节点前可能存在多条分支合并，需要执行器终止当前规则链路的执行，并随后启动一条新的执行链路
   * @return
   */
  public boolean realDelayed() {
    return false;
  }
  
  /**
   * 当一个规则链路定义中的开始节点类被定义的时候，应该设定后者对应的规则实例上下文信息
   * 这样设置的目的，是为了保证规则实例上下文不会因为弱引用的原因给GC掉
   * 
   * 在规则链路执行完成后（无论是正常执行完成还是异常执行完成），都应该清理这个链路中
   * 所有的规则实例上下文的依赖信息
   * @param ruleRuntimeContext
   */
  public void setRuleRuntimeContext(RuleRuntimeContext ruleRuntimeContext) {
    this.ruleRuntimeContext = ruleRuntimeContext;
  }
  
  RuleRuntimeContext getRuleRuntimeContext() {
    return ruleRuntimeContext;
  }
  
  @Override
  public int compareTo(Delayed o) {
    if(!(o instanceof RuntimeDelayNode) || !this.realDelayed()) {
      return 0;
    }
    return this.timerCount - ((RuntimeDelayNode)o).timerCount;
  }

  @Override
  public long getDelay(TimeUnit unit) {
    // 如果不存在真实的延迟情况，则返回0
    if(!this.realDelayed()) {
      return 0l;
    }
    // 默认的最小时间单位为毫秒
    TimeUnit currentUnit = unit;
    if(currentUnit == null || currentUnit == TimeUnit.NANOSECONDS || currentUnit == TimeUnit.MILLISECONDS) {
      currentUnit = TimeUnit.SECONDS;
    }
    
    // 如果没有设定过期时间点，则进行设定
    if(expireTime == null) {
      long e = TimeUnit.MILLISECONDS.convert(this.timerCount, currentUnit);
      expireTime = new Date().getTime() + e;
      return e;
    }
    return expireTime -  new Date().getTime();
  }

  public void setTimerCount(int timerCount) {
    this.timerCount = timerCount; 
  }
}
