package com.bizunited.platform.mars.policy.process.rule.mutex;

import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.mutex.RuntimeMutexNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;

/**
 * 基于Redis组件工作的分布式锁处理器，其使用当前规则定义中的规则code+version作为版本号
 * 为了保证多个不同的业务集群能够正常工作，其又加入了当前spring app name作为整个key的前缀
 * @author yinwenjie
 */
@Component("redisMutexLockRuleable")
public class RedisMutexLockRuleable<T extends RuntimeMutexNode> implements MutexLockRuleable<RuntimeMutexNode> {
  @Autowired
  private RedissonClient redissonClient;
  @Value("${spring.application.name}")
  private String appName;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RedisMutexLockRuleable.class);
  
  @Override
  public void doProcess(RuntimeMutexNode currentNode ,RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked, ProcessorChain processChain) {
    // 该处理器将应用名 + 规则定义code + 规则定义version作为key
    RuntimeDefinition currentDefinition = context.getRuntimeDefinition();
    String code = currentDefinition.getCode().toString();
    String version = currentDefinition.getCverion().toString();
    String key = StringUtils.join(appName, "_", code , "_" , version);
    
    // 加锁后递归进行
    RLock lock = redissonClient.getLock(key);
    try {
      lock.lock();
      processChain.doProcessNode(context, runtimeProcessorLinked);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      throw e;
    } finally {
      lock.unlock();
    }
  }

  @Override
  public String description() {
    return "基于redis的分布式锁实现，key = 应用名_规则定义code_规则定义version";
  }

  @Override
  public Class<RuntimeMutexNode> mapping() {
    return RuntimeMutexNode.class;
  }
}