package com.bizunited.platform.mars.policy.process.runtime.contexts;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.service.EndRuleableCallback;

/**
 * 规则实例运行时上下文信息，不允许继承、修改，只允许在一个工作线程的范围进行使用
 * @author yinwenjie
 */
public final class RuleRuntimeContext {
  /**
   * 当前正在处理（包括延迟等待的）的规则实例上下文映射信息，存储在这里
   * 该映射关联是弱引用的，以便在本规则实例没有任何处理步骤的情况下，进行规则上下文的回收
   */
  private static Map<String , WeakReference<RuleRuntimeContext>> ruleRuntimeContexts = new ConcurrentHashMap<>();
  /**
   * 引用回收队列
   */
  public static ReferenceQueue<RuleRuntimeContext> referenceQueue = new ReferenceQueue<>();
  /**
   * 当前可能处于阻塞状态的，请求当次规则实例执行的调用请求线程
   */
  private Thread requestThread;
  /**
   * 当前规则的运行状态
   */
  private volatile RuleRunTimeStatus status;
  /**
   * 这个map结构存储当前上下文中的各种参数信息
   */
  private Map<String, Object> params = new ConcurrentHashMap<>();
  /**
   * 记录本次运行实例中，已经运行完成或者正在运行的规则链路信息(使用有序跳跃表进行存储)
   */
  private Set<RuntimeProcessorLinked> runtimeProcessorLinkeds = new ConcurrentSkipListSet<>();
  /**
   * 规则正常运行结束后的返回结果在这里存储
   */
  private volatile Object _return;
  /**
   * 如果当前规则运行过程已经出现异常，则异常信息在这里
   */
  private Throwable currentThrowable;
  /**
   * 记录当前抛出异常的运行时节点层次（即递归深度）
   */
  private Integer exceptionLevel;
  /**
   * 当前规则实例运行上下文中对应的规定定义描述（运行时）
   */
  private RuntimeDefinition runtimeDefinition;
  /**
   * 记录在上下文中的，由程序员定义的结束节点回调
   * 如果一个规则定义存在多个结束节点，则每一次结束节点执行，都会调用该回调
   */
  private EndRuleableCallback endRuleableCallback;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RuleRuntimeContext.class);
  
  public RuleRuntimeContext(Thread requestThread) {
    this.requestThread = requestThread;
  }
  public Thread getRequestThread() {
    return requestThread;
  }
  public void setRequestThread(Thread requestThread) {
    this.requestThread = requestThread;
  }
  public RuleRunTimeStatus getStatus() {
    return status;
  }
  public void setStatus(RuleRunTimeStatus status) {
    this.status = status;
  }
  public Map<String, Object> getParams() {
    return params;
  }
  public void setParams(Map<String, Object> params) {
    this.params = params;
  }
  public Object get_return() {
    return _return;
  }
  public void set_return(Object _return) {
    this._return = _return;
  }
  public Throwable getCurrentThrowable() {
    return currentThrowable;
  }
  public void setCurrentThrowable(Throwable currentThrowable) {
    this.currentThrowable = currentThrowable;
  }
  public Integer getExceptionLevel() {
    return exceptionLevel;
  }
  public void setExceptionLevel(Integer exceptionLevel) {
    this.exceptionLevel = exceptionLevel;
  }
  public RuntimeDefinition getRuntimeDefinition() {
    return runtimeDefinition;
  }
  public void setRuntimeDefinition(RuntimeDefinition runtimeDefinition) {
    this.runtimeDefinition = runtimeDefinition;
  }
  public Set<RuntimeProcessorLinked> getRuntimeProcessorLinkeds() {
    return runtimeProcessorLinkeds;
  }
  public void addRuntimeProcessorLinkeds(RuntimeProcessorLinked runtimeProcessorLinked) {
    this.runtimeProcessorLinkeds.add(runtimeProcessorLinked);
  }
  public EndRuleableCallback getEndRuleableCallback() {
    return endRuleableCallback;
  }
  public void setEndRuleableCallback(EndRuleableCallback endRuleableCallback) {
    this.endRuleableCallback = endRuleableCallback;
  }
  /**
   * 该静态工具方法，用于向当前已创建的规则上下文管理器中获取已有的上下文信息
   * @param instanceId
   * @return
   */
  public static RuleRuntimeContext getRuleRuntimeContext(String instanceId) {
    WeakReference<RuleRuntimeContext> refs = RuleRuntimeContext.ruleRuntimeContexts.get(instanceId);
    return refs.get();
  }
  
  /**
   * 该方法将一个存在的规则运行时保存到上下文映射工具中，以便该运行时能够在后续的处理过程中被各种ruleable运行器使用
   * @param instanceId 指定的规则实例id
   * @param ruleRuntimeContext 规则运行时
   */
  public static void setRuleRuntimeContext(String instanceId , RuleRuntimeContext ruleRuntimeContext) {
    Validate.notNull(ruleRuntimeContext , "错误的规则实例上下文信息[null]，请检查!!");
    // TODO 需要一个引用队列信息
    WeakReference<RuleRuntimeContext> refs = new WeakReference<>(ruleRuntimeContext, referenceQueue);
    RuleRuntimeContext.ruleRuntimeContexts.put(instanceId, refs);
  }
  
  /** 
   * 该静态工具方法，用于清理那些已经失效的规则运行上下文的信息——使用规则实例id进行清理
   * @param reference
   */ 
  public static void clearRuleRuntimeContext(Reference<? extends RuleRuntimeContext> reference) {
    Set<Entry<String, WeakReference<RuleRuntimeContext>>> enties = ruleRuntimeContexts.entrySet();
    enties.spliterator().forEachRemaining(item -> {
      // 如果条件成立，说明找到了需要被清理的弱引用
      if(item.getValue() == reference) {
        String key = item.getKey();
        RuleRuntimeContext.ruleRuntimeContexts.remove(key);
        LOGGER.info("遗留规则实例上下文【" + key + "】已被清理=====");
        return;
      }
    });
  }
}
