package com.bizunited.platform.mars.policy.process.runtime.service;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.mars.entity.RuleDefinitionEntity;
import com.bizunited.platform.mars.entity.RuleNodeEntity;
import com.bizunited.platform.mars.policy.process.cache.Mappable;
import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNode;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeMapping;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeDelayNode;
import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.service.RuleDefinitionListener;
import com.bizunited.platform.mars.service.RuleDefinitionService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.ehcache.Cache;
import org.ehcache.CacheManager;
import org.ehcache.config.CacheConfiguration;
import org.ehcache.config.ResourcePools;
import org.ehcache.config.builders.CacheConfigurationBuilder;
import org.ehcache.config.builders.CacheManagerBuilder;
import org.ehcache.config.builders.ExpiryPolicyBuilder;
import org.ehcache.config.builders.ResourcePoolsBuilder;
import org.ehcache.config.units.MemoryUnit;
import org.ehcache.expiry.ExpiryPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 规则定义运行时服务的实现
 *
 * @author yinwenjie
 */
@Component("simpleRuntimeDefinitionService")
public class SimpleRuntimeDefinitionService extends AbstractRuntimeService implements RuntimeDefinitionService, RuleDefinitionListener {

  @Autowired
  private RuleDefinitionService ruleDefinitionService;
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  @Qualifier("_delayTimerQueue")
  private DelayQueue<RuntimeDelayNode> delayQueue;
  /**
   * 使用ecache作为规则定义对象的本地缓存
   * 缓存时间为5秒
   */
  private static Cache<String, RuleDefinitionEntity> cached;
  /**
   * 针对当前规则定义的code和version对应的可重入锁
   */
  private static Map<String, ReentrantLock> runtimeDefinitionQueryLocks = new ConcurrentHashMap<>();
  /**
   * key前缀
   */
  private static final String PRE = "_MARS";
  private static final Logger LOGGER = LoggerFactory.getLogger(SimpleRuntimeDefinitionService.class);

  public SimpleRuntimeDefinitionService() {
    synchronized (SimpleRuntimeDefinitionService.class) {
      while (cached == null) {
        // 设置集合最大数据量
        ResourcePools resourcePools = ResourcePoolsBuilder.heap(500).build();
        // 设置过期策略为：失效前允许存活24小时（放心，由于本类实现了RuleDefinitionListener接口，所以只要数据层有变化，缓存信息都会被即时清楚）
        ExpiryPolicy<Object, Object> expiryPolicy = ExpiryPolicyBuilder.timeToLiveExpiration(Duration.ofHours(24));
        CacheConfigurationBuilder<String, RuleDefinitionEntity> cacheConfigurationBuilder = CacheConfigurationBuilder.newCacheConfigurationBuilder(String.class, RuleDefinitionEntity.class, resourcePools);
        CacheConfiguration<String, RuleDefinitionEntity> cacheConfiguration = cacheConfigurationBuilder.withExpiry(expiryPolicy).build();

        // 开始构建缓存
        CacheManagerBuilder<CacheManager> build = CacheManagerBuilder.newCacheManagerBuilder();
        CacheManager cacheManager =
                build.withCache("pre_RuleDefinition_Configured", cacheConfiguration)
                        .withDefaultSizeOfMaxObjectSize(1024L, MemoryUnit.MB)
                        .build(true);
        cached = cacheManager.getCache("pre_RuleDefinition_Configured", String.class, RuleDefinitionEntity.class);
      }
    }
  }

  @Override
  public Object process(String code, String version, Map<String, Object> params) throws RuntimeException {
    return this.process(code, version, params, null);
  }

  @Override
  public Object process(String code, String version, Map<String, Object> params,
      ProcessCallback processCallback) throws RuntimeException {
    return this.process(code, version, params, processCallback, null);
  }

  @Override
  public Object process(String code, String version, Map<String, Object> params,
      ProcessCallback processCallback, EndRuleableCallback endRuleableCallback) throws RuntimeException {
    /*
     * 执行过程为：
     * 调用start私有方法，开始一个新的规则实例运行的前提下，给当前调用线程阻塞信号
     * blocker设定为当前实例上下文
     * */
    
    RuleRuntimeContext runtimeContext = this.start(code, version, params, true , endRuleableCallback);
    LockSupport.park(runtimeContext);
    if (runtimeContext.getStatus() == RuleRunTimeStatus.EXCEPTION) {
      throw new IllegalArgumentException(runtimeContext.getCurrentThrowable().getMessage());
    }

    if(processCallback != null) {
      processCallback.doInvocation(runtimeContext);
    }
    return runtimeContext.get_return();
  }

  /**
   * 根据规则定义的编号和版本号，创建一个独立的规则实例。规则实例由独立的uuid编号，
   * 还有独立的运行时上下文
   *
   * @param code
   * @param version
   * @return
   */
  @SuppressWarnings("unchecked")
  private RuntimeDefinition createByCodeAndVersion(String code, String version) {
    /*
     * 处理过程为：
     * 1、首先根据code和version查询数据层指定的规则定义详情，包括所有的节点、节点扩展信息和连线信息
     * （注意，这里有一个缓存机制，一般来说都是直接从缓存中获取）
     * 2、接着根据查询出来的RuleDefinitionEntity信息，开始进行RuntimeDefinition信息的构造，具体过程为：
     *  2.1、首先点赋予生成的instanceid，并构造RuntimeDefinition基本信息
     *  2.2、然后构造各个RuntimeNode信息和扩展的RuntimeNode信息，以及各个节点中的RuntimeNodeParams信息
     *  2.3、最后构造连线信息
     * 3、完成构造后的RuntimeDefinition 对象将会被返回
     * */

    // 1、==========
    Validate.notBlank(code, "必须传入规则定义编号信息!!");
    Validate.notBlank(version, "必须传入规则定义版本信息!!");
    String instanceId = UUID.randomUUID().toString();
    RuleDefinitionEntity ruleDefinition = this.findDetailsByCodeAndVersion(code, version);
    Validate.notNull(ruleDefinition, "错误的规则定义信息[code=%s;version=%s]，未找到对应的规则定义，请检查!!", code, version);

    // 2.1、==========
    // 注意，只处理基本信息，因为规则运行时中，根据节点类型的不一样，类情况是完全不一样的
    RuntimeDefinition resultRuntimeDefinition = this.nebulaToolkitService.copyObjectByWhiteList(ruleDefinition, RuntimeDefinition.class, LinkedHashSet.class, ArrayList.class);
    resultRuntimeDefinition.setInstanceId(instanceId);
    // 2.2、==========
    /*
     * 采用继承结构 + 泛型 + 中考解决代码中大量if的问题，代码get/set冗余问题，
     * 以及特定对象中特殊值的处理问题，
     * 属于工厂方法模式的一种变形
     * */
    Set<RuntimeNode> runtimeNodes = new HashSet<>();
    for (RuleNodeEntity ruleNode : ruleDefinition.getNodes()) {
      int type = ruleNode.getType();
      Class<? extends RuntimeNode> runtimeNodeClass = null;
      if (type == 2) {
        runtimeNodeClass = RuntimeNodeMapping.MAPPING.get(type * 10 + ruleNode.getSourceType());
      } else {
        runtimeNodeClass = RuntimeNodeMapping.MAPPING.get(type);
      }

      Validate.notNull(runtimeNodeClass, "找到节点类型为[%d]的节点定义，但是该值所对应的节点类型，目前规则引擎不支持，请检查!!", type);
      RuntimeNode currrentRuntimeNode = null;
      try {
        currrentRuntimeNode = runtimeNodeClass.newInstance();
      } catch (InstantiationException | IllegalAccessException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(e.getMessage(), e);
      }
      Validate.isTrue(currrentRuntimeNode instanceof Mappable, "当前节点类型[%s]下的具体节点定义，没有实现Mappable接口，无法实现节点定义映射，请检查!!", type);
      Mappable<RuleNodeEntity, ? extends RuntimeNode> currentMappable = (Mappable<RuleNodeEntity, ? extends RuntimeNode>) currrentRuntimeNode;
      RuntimeNode currentRuntimeNode = currentMappable.mappingTransform(ruleNode);
      currentRuntimeNode.setInstanceId(instanceId);
      runtimeNodes.add(currentRuntimeNode);
    }
    resultRuntimeDefinition.setNodes(runtimeNodes);
    return resultRuntimeDefinition;
  }

  /**
   * 该私有方法用于开始一个新的规则定义实例
   * @param sync    是否是同步调用
   * @param endRuleableCallback 结束节点的回调
   * @return
   */
  private RuleRuntimeContext start(String code, String version, Map<String, Object> params, boolean sync , EndRuleableCallback endRuleableCallback) {
    /*
     * 执行过程：
     * 1、验证指定的规则定义是否符合进行同步执行的要求，如果不符合执行要求，则抛出异常
     * 2、基于当前规则实例，创建规则实例运行上下文，以及所需的当前规则运行时所需要独立使用、记录的信息
     * 并且基于当前规则实例的开始节点，进行执行器通知并开始执行规则实例
     * 注意，上下文信息要和规则实例id进行映射
     * */

    // 1、======
    RuntimeDefinition currentDefinition = this.createByCodeAndVersion(code, version);
    RuntimeNode startNode = this.runtimeNodeService.findStartByDefinition(currentDefinition);
    Set<RuntimeNode> endNodes = this.runtimeNodeService.findEndByDefinition(currentDefinition);
    Validate.notNull(startNode, "未找到规则定义中的开始节点，请检查规则定义!!");
    if (sync) {
      Validate.isTrue(!CollectionUtils.isEmpty(endNodes), "未找到规定定义中有任何结束节点，请检查规则定义!!");
      Validate.isTrue(endNodes.size() == 1, "当前规则定义中找到不止一个结束节点，不能使用同步调用方式进行规则实例的调用!!");
    }

    // 2、======
    final RuleRuntimeContext runtimeContext = new RuleRuntimeContext(Thread.currentThread());
    runtimeContext.setEndRuleableCallback(endRuleableCallback);
    runtimeContext.setStatus(RuleRunTimeStatus.NORMAL);
    if (params != null) {
      runtimeContext.setParams(new HashMap<>(params));
    }
    //用户信息以及jps session 写入上下文中
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    runtimeContext.getParams().put("_authentication", authentication);
    runtimeContext.setRuntimeDefinition(currentDefinition);
    RuleRuntimeContext.setRuleRuntimeContext(currentDefinition.getInstanceId(), runtimeContext);
    RuntimeDelayNode runtimeDelayNode = (RuntimeDelayNode) startNode;
    runtimeDelayNode.setRuleRuntimeContext(runtimeContext);
    delayQueue.add(runtimeDelayNode);
    return runtimeContext;
  }

  /**
   * 该私有方法用于真正查询数据库中的指定规则定义详情
   * 通过给定的规则定义编号和版本信息，查询完整的规则描述。
   * 规则描述信息将基于持久层存储的规则定义信息生成，并被缓存到内存中
   *
   * @param code    指定的规则定义编号
   * @param version 指定的规则定义版本
   * @return
   */
  private RuleDefinitionEntity findDetailsByCodeAndVersion(String code, String version) { 
    if (StringUtils.isBlank(code) || StringUtils.isBlank(version)) {
      return null;
    }
    String key = StringUtils.join(PRE, "_", code, "_", version);
    // 如果有缓存，则从缓存中查询
    RuleDefinitionEntity cacheRuleDefinitionEntity = this.findDetailsByCodeAndVersionOnCache(code, version);
    if (cacheRuleDefinitionEntity != null) {
      LOGGER.info("// === 从缓存中加载!!cacheRuleDefinitionEntity = " + code);
      return cacheRuleDefinitionEntity;
    }
    synchronized (SimpleRuntimeDefinitionService.class) {
      while (runtimeDefinitionQueryLocks.get(key) == null) {
        runtimeDefinitionQueryLocks.put(key, new ReentrantLock());
      }
    }

    // 从数据库进行查询，如果查询到，则再次存入缓存中
    // 这样保证在本地cache失效的情况下，也只会查询一次
    ReentrantLock lock = null;
    RuleDefinitionEntity ruleDefinitionEntity = null;
    try {
      lock = runtimeDefinitionQueryLocks.get(key);
      lock.lock();
      while ((ruleDefinitionEntity = this.findDetailsByCodeAndVersionOnCache(code, version)) == null) {
        ruleDefinitionEntity = this.ruleDefinitionService.findDetailsByCodeAndVersion(code, version);
        if (ruleDefinitionEntity == null) {
          return null;
        }
        // 注意，从数据库查询得到的信息，是一个JPA信息，所以要进行中度拷贝
        RuleDefinitionEntity target = this.nebulaToolkitService.copyObjectByWhiteList(ruleDefinitionEntity, RuleDefinitionEntity.class, LinkedHashSet.class, ArrayList.class, new String[]{"nodes", "nodes.templateNode", "nodes.templateNode.sourceDataView", "nodes.templateNode.sourceServicable", "nodes.templateNode.sourceAggregateDataView", "nodes.templateNode.sourceScript", "nodes.ruleLockExt",
                "nodes.ruleAggregationExt", "nodes.ruleTimerExt",
                "nodes.inputs", "nodes.outputs", "nodes.nexts",
                "nodes.nexts.fromNode", "nodes.nexts.toNode"});
        cached.put(key, target);
        return target;
      }
      return ruleDefinitionEntity;
    } finally {
      lock.unlock();
    }
  }

  // 该私有方法试图从本地缓存中查询出规则定义信息
  private RuleDefinitionEntity findDetailsByCodeAndVersionOnCache(String code, String version) {
    String key = StringUtils.join(PRE, "_", code, "_", version);
    RuleDefinitionEntity ruleDefinitionEntity = cached.get(key);
    return ruleDefinitionEntity;
  }

  @Override
  public Object processAsync(String code, String version, Map<String, Object> params) {
    return this.process(code, version, params, null);
  }

  @Override
  public Object processAsync(String code, String version, Map<String, Object> params, ProcessCallback processCallback) {
    return this.processAsync(code, version, params, processCallback, null);
  }

  @Override
  public Object processAsync(String code, String version, Map<String, Object> params,
      ProcessCallback processCallback, EndRuleableCallback endRuleableCallback) {
    // 和SimpleRuntimeDefinitionService.process方法执行过程类似
    RuleRuntimeContext runtimeContext = this.start(code, version, params, false,endRuleableCallback);
    DefinitionProcessFuture future = null;
    if(processCallback != null) {
      future = new DefinitionProcessFuture(runtimeContext , processCallback);
    } else {
      future = new DefinitionProcessFuture(runtimeContext);
    }
    try {
      return future.get();
    } catch (ExecutionException | InterruptedException e) {
      // TODO 终止信号本身，暂不处理
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  @Override
  public void onDefinitionCreate(RuleDefinitionEntity ruleDefinition) {
    // 无需做任何业务动作
  }

  @Override
  public void onDefinitionModify(RuleDefinitionEntity ruleDefinition) {
    this.onDefinitionDelete(ruleDefinition);
  }

  @Override
  public void onDefinitionDelete(RuleDefinitionEntity ruleDefinition) {
    // 一旦数据层的规则定义被删除或者被修改，则本服务实现中缓存的规则定义信息，就直接被清理
    String code = ruleDefinition.getCode();
    String version = ruleDefinition.getCverion();
    String key = StringUtils.join(PRE, "_", code, "_", version);
    cached.remove(key);
  }
}
