package com.bizunited.platform.mars.policy.process.rule.conditions;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.compress.utils.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeConditionNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeDefinitionService;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeNodeService;

/**
 * 简单的条件规则的顶层定义
 * @author yinwenjie
 */
@Component("simpleConditionRuleable")
public class SimpleConditionRuleable<T extends RuntimeConditionNode> implements ConditionRuleable<RuntimeConditionNode> {
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  @Autowired
  private ScriptService scriptService;

  @Override
  public int getType() {
    return RuntimeNodeType.CONDITION.getValue();
  }

  @Override
  public void doProcess(RuntimeConditionNode currentNode ,RuleRuntimeContext context , RuntimeProcessorLinked runtimeProcessorLinked , ProcessorChain processChain) {
    processChain.doProcessNode(context, runtimeProcessorLinked);
  }

  @Override
  public Set<RuntimeProcessorLinked> createProcessorLinkeds(RuntimeConditionNode currentNode, RuleRuntimeContext context) {
    /*
     * 在接收到继续的通知，正式开始继续执行前，SimpleConditionRuleable节点会依次检查所有的条件分支，
     * 找到最匹配的分支运行条件后，继续执行，
     *
     * 如果没有找到任何匹配的条件分支，则会抛出错误，要求结束规则实例的执行过程
     *
     * 处理过程为:
     * 1、准备数据，判定是否是条件类型的节点
     * 2、如果当前节点是条件节点，则做如下处理：
     *   2.1、nexts信息需要排序，然后再根据判定表达式，依次执行判定
     *   2.2、每次执行判定时，都要分析表单式的所使用的对象，并从context上下文中进行获取
     *   2.3、如果以上两个步骤没有找到符合条件的分支，则查找有没有没有conditions设置的默认分支
     * 3、根据获得的下一节点运行时信息，构建规则链路分支
     * */
    // 1、========
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    RuntimeNodeType type = currentNode.getType();
    Validate.isTrue(type == RuntimeNodeType.CONDITION , "当前节点不是条件类型的运行时节点，请检查!!");

    // 2、===== 如果执行到这里，就需要进行条件判定或者并行分支了
    Set<RuntimeNodeNexts> sortNexts = nexts.stream().filter(item -> item.getLineType() == 1).sorted((source , target) -> source.getSort() - target.getSort()).collect(Collectors.toSet());
    RuntimeNodeNexts currentNext = null;
    for(RuntimeNodeNexts next : sortNexts) {
      String conditions = next.getConditions();
      if(StringUtils.isBlank(conditions)) {
        continue;
      }
      // 执行
      Object evalResult = this.executeGroovyScript(conditions, context.getParams());
      if(evalResult == null || Boolean.class != evalResult.getClass()) {
        throw new IllegalArgumentException("条件判定分支的返回值，只能是boolean型，请检查!!");
      }
      // 如果条件成立，说明符合条件，就是这个运行分支
      if((Boolean)evalResult) {
        currentNext = next;
        break;
      }
    }
    // 2.3、===
    if(currentNext == null) {
      currentNext = sortNexts.stream().filter(item -> StringUtils.isBlank(item.getConditions())).findFirst().orElse(null);
    }
    Validate.notNull(currentNext , "未发现任何符合条件的运行分支，也没有发现未设置运行条件的默认分支，请检查!!");

    // 3、=====
    RuntimeProcessorLinked runtimeProcessorLinked = new RuntimeProcessorLinked(currentNode, currentNext);
    return Sets.newHashSet(runtimeProcessorLinked);
  }

  /**
   * 在全动态后端groovy脚本的场景下，运行实际的动态脚本代码
   * @param scriptContxt
   * @param contentMap
   * @param binding 可能来源于调用者的脚本变量绑定信息
   * @return 可能有返回值，也可能没有任何返回值
   */
  private Object executeGroovyScript(String scriptContxt, Map<String , Object> contentMap) {
    // 进行上下文已有参数绑定后执行
    Map<String , Object> scriptInputsMap = new HashMap<>(contentMap);
    scriptInputsMap.put("runtimeNodeService", runtimeNodeService);
    scriptInputsMap.put("runtimeDefinitionService", runtimeDefinitionService);
    return this.scriptService.invoke(scriptContxt, scriptInputsMap);
  }

  @Override
  public Class<RuntimeConditionNode> mapping() {
    return RuntimeConditionNode.class;
  }
}