package com.bizunited.platform.mars.policy.process.rule.logic;

import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.mars.entity.RuleSourceScriptEntity;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeParams;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.logic.RuntimeScriptNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeDefinitionService;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeNodeService;
import com.bizunited.platform.mars.service.RuleSourceScriptService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 基于groovy脚本的逻辑处理组件（执行器）
 *
 * @author yinwenjie
 */
@Component("groovyScriptProcessRuleable")
public class GroovyScriptProcessRuleable<T extends RuntimeScriptNode> extends AbstractLogicRuleable<RuntimeScriptNode> implements LogicRuleable<RuntimeScriptNode> {
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  @Autowired
  private RuleSourceScriptService ruleSourceScriptService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(GroovyScriptProcessRuleable.class);

  @Override
  public void doProcess(RuntimeScriptNode currentNode, RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked, ProcessorChain processChain) {
    super.process(currentNode, runtimeProcessorLinked);
    processChain.doProcessNode(context, runtimeProcessorLinked);
  }

  @Override
  protected void execute(RuntimeScriptNode currentNode, Map<String, Object> inputParamValues, RuleRuntimeContext context) {
    String code = currentNode.getCode();
    Validate.notBlank(code, "在进行节点定义执行时，节点定义的code必须有值，请检查入参!!");
    // 取得并判定处理源信息
    Validate.notNull(currentNode, "运行节点[%s]时，未发现处理源信息，请检查!!", code);
    String scriptId = currentNode.getScriptId();
    Validate.notBlank(scriptId, "运行节点[%s]时，未发现指定的脚本Id信息，请检查!!", code);
    RuleSourceScriptEntity ruleSourceScript = ruleSourceScriptService.findById(scriptId);
    Validate.notNull(ruleSourceScript, "运行节点[%s]时，未发现指定的脚本Id信息，请检查!!", code);
    ScriptEntity scriptEntity = ruleSourceScript.getScript();
    String scriptContent = scriptService.findContentById(scriptEntity.getId());
    Validate.notBlank(scriptContent, "运行节点[%s]时，未发现指定的脚本信息，请检查!!", code);
    String language = scriptEntity.getLanguage();
    Validate.isTrue(StringUtils.equals(language, "groovy"), "运行节点[%s]时，发现脚本类型不匹配，请检查!!", code);

    // 2、======
    inputParamValues.putAll(context.getParams());
    inputParamValues.put("runtimeNodeService", runtimeNodeService);
    inputParamValues.put("runtimeDefinitionService", runtimeDefinitionService);
    Map<String, Object> outputParamValues = null;
    try {
      outputParamValues = this.scriptService.invoke(new String[]{scriptEntity.getId()}, inputParamValues);
    } catch (InvokeProxyException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    if (outputParamValues == null) {
      outputParamValues = new HashMap<>();
    }

    // 3、=====
    Map<String, Object> contentMap = context.getParams();
    Set<RuntimeNodeParams> outputParams = currentNode.getOutputs();
    if (outputParams != null) {
      for (RuntimeNodeParams item : outputParams) {
        boolean nullable = item.getNullable();
        // TODO 出参类型还要进行判定
        String templateParamName = item.getTemplateParamName();
        String contextParamName = item.getContextParamName();
        Object outputValue = outputParamValues.get(templateParamName);
        if (outputValue == null && nullable) {
          String errorMsg = String.format("分析运行节点[%s]返回值时，发现指定的出参[%s]设定为not nullable，但是执行结果却没有返回值，请检查", code, templateParamName);
          throw new IllegalArgumentException(errorMsg);
        }
        contentMap.put(contextParamName, outputValue);
      }
    }
  }

  @Override
  public Class<RuntimeScriptNode> mapping() {
    return RuntimeScriptNode.class;
  }
}