package com.bizunited.platform.mars.policy.process.rule.waiter;

import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeParallelBranchNode;
import com.bizunited.platform.mars.policy.process.executor.ProcessorChain;
import com.bizunited.platform.mars.policy.process.rule.starter.StarterRuleable;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

/**
 * 随机A/B并行的节点
 *
 * @author yinwenjie
 */
@Component("_ParallelBranchRuleable")
public class ParallelBranchRuleable<T extends RuntimeParallelBranchNode> implements StarterRuleable<RuntimeParallelBranchNode> {

  @Override
  public int getType() {
    return RuntimeNodeType.MERGE.getValue();
  }

  @Override
  public void doProcess(RuntimeParallelBranchNode currentNode, RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked, ProcessorChain processChain) {
    processChain.doProcessNode(context, runtimeProcessorLinked);
  }

  @Override
  public Set<RuntimeProcessorLinked> createProcessorLinkeds(RuntimeParallelBranchNode currentNode, RuleRuntimeContext context) {
    /*
     * 在接收到继续的通知，正式开始继续执行前，ParallelBranchRuleable节点会依次检查所有的条件分支，
     * 找到所有匹配条件的分支，并为这些分支的后续执行创建多个规则链路的初始状态，并向下执行
     *
     * 1、链路检查
     *  1.1、是否存在后续节点数据
     *  1.2、判断当前节点是否为A/B并行节点
     *  1.3、当前节点参数是否有效(集合或者数据类型)
     * 2、随机生成输出参数子集大小
     *  2.1、拆分参数子集
     *  2.2、根据出参名，若无出参名默认使用 "上下文参数名+下划线+数字"的形式例如user_1写入上下文中
     * 3、根据各分支获得下一节点运行时信息，构建规则链路分支
     * */

    // 1、链路检查
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    Validate.isTrue(CollectionUtils.isNotEmpty(nexts), "当前节点后续运行时连线数据为空，请检查！");
    RuntimeNodeType type = currentNode.getType();
    Validate.isTrue(type == RuntimeNodeType.CONCURRENCY, "当前节点不是A/B并发类型的运行时节点，请检查!!");
    Validate.notBlank(currentNode.getParamName(), "A/B并行节点参数名为空，请检查！");

    Set<RuntimeNodeNexts> sortNexts = nexts.stream().filter(item -> item.getLineType() == 1).sorted((source, target) -> source.getSort() - target.getSort()).collect(Collectors.toSet());
    Set<RuntimeProcessorLinked> runtimeProcessorLinkeds = Sets.newHashSet();
    List<?> params = getParams(context, currentNode.getParamName());
    Validate.isTrue(CollectionUtils.isNotEmpty(params), "A/B并行节点集合数据为空，请检查");
    // 2、拆分上下文参数
    int groupSize = sortNexts.size();
    List<List<?>> subLists = splitList(params, groupSize);
    int current = 0;
    for (RuntimeNodeNexts next : sortNexts) {
      String bindParam = next.getBindParam();
      //绑定上下文参数为空的情况下，采用约定默认命名方式
      if (StringUtils.isBlank(bindParam)) {
        bindParam = currentNode.getParamName() + "_" + current;
      }
      context.getParams().put(bindParam, subLists.get(current));
      // 3、根据各分支获得下一节点运行时信息，构建规则链路分支
      RuntimeProcessorLinked runtimeProcessorLinked = new RuntimeProcessorLinked(currentNode, next);
      runtimeProcessorLinkeds.add(runtimeProcessorLinked);
      current++;
    }
    return runtimeProcessorLinkeds;
  }

  /**
   * 根据groupSize 随机拆分传入的List数据
   *
   * @param params
   * @param groupSize
   * @return
   */
  private List<List<?>> splitList(List<?> params, Integer groupSize) {
    int dataSize = params.size();
    Validate.isTrue(dataSize >= groupSize, "输入数据集合小于需要分组数，无法随机分组，请检查！");
    List<List<?>> result = Lists.newArrayListWithCapacity(groupSize);
    List<?> subList = null;
    int[] splitSize = getRandomSplitSize(groupSize, dataSize);
    int start = 0;
    for (int s : splitSize) {
      subList = params.subList(start, start + s);
      start += s;
      result.add(subList);
    }
    return result;
  }

  /**
   * 根据dataSize与groupSize 生成groupSize大小的随机数组，数组长度之和固定为dataSize
   *
   * @param groupSize
   * @param dataSize
   * @return
   */
  private int[] getRandomSplitSize(Integer groupSize, Integer dataSize) {
    Random random = ThreadLocalRandom.current();
    int[] result = new int[groupSize];
    int randomNum = 0;
    int tmp = 0;
    for (int i = 0; i < dataSize; i++) {
      tmp = sum(result);
      if (i == groupSize - 1 || tmp == dataSize) {
        result[i] = dataSize - tmp;
        break;
      }
      randomNum = random.nextInt((dataSize - tmp) / (groupSize - i)) + 1;
      if (tmp + randomNum <= dataSize) {
        result[i] = randomNum;
      }
    }
    return result;
  }

  /**
   * 计算数组的和
   *
   * @param datas
   * @return
   */
  private int sum(int[] datas) {
    int sum = 0;
    for (int d : datas) {
      sum += d;
    }
    return sum;
  }

  /**
   * 从上下文中获取指定参数并验证有效性（集合或者数组）转型为List
   *
   * @param context   运行时上下文
   * @param paramName 参数名
   * @return
   */
  private List<?> getParams(RuleRuntimeContext context, String paramName) {
    Object params = context.getParams().get(paramName);
    Validate.notNull(params, "A/B并行节点上下文参数为空,请检查!");
    // 验证参数是否为集合或者数组
    Validate.isTrue(params.getClass().isArray() || Collection.class.isAssignableFrom(params.getClass()), "制定参数不是集合或者数组，请检查！");
    List<?> paramList = null;
    if (params.getClass().isArray()) {
      Object[] arr = (Object[]) params;
      paramList = Arrays.asList(arr);
    } else if (Collection.class.isAssignableFrom(params.getClass())) {
      Collection<?> collection = (Collection<?>) params;
      paramList = Lists.newLinkedList(collection);
    }
    return paramList;
  }

  @Override
  public Class<RuntimeParallelBranchNode> mapping() {
    return RuntimeParallelBranchNode.class;
  }
}
