package com.bizunited.platform.mars.policy.process.runtime.service;

import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import org.apache.commons.lang3.Validate;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.LockSupport;

/**
 * 这个future负责在规则定义进行异步执行的时候，为调用者提供一个观察执行状态和异步获取执行接口的一个窗口</br>
 * 由于规则定义的执行可能涉及多个线程，所以该future窗口不支持取消操作
 *
 * @author yinwenjie
 */
public class DefinitionProcessFuture implements Future<Object> {

  /**
   * 本次运行所涉及的上下文
   */
  private RuleRuntimeContext ruleRuntimeContext;

  /**
   * 可能有客户设定的一个回调
   */
  private ProcessCallback processCallback;

  DefinitionProcessFuture(RuleRuntimeContext ruleRuntimeContext) {
    this.ruleRuntimeContext = ruleRuntimeContext;
  }

  DefinitionProcessFuture(RuleRuntimeContext ruleRuntimeContext, ProcessCallback processCallback) {
    this.ruleRuntimeContext = ruleRuntimeContext;
    this.processCallback = processCallback;
  }

  @Override
  public boolean cancel(boolean mayInterruptIfRunning) {
    return false;
  }

  @Override
  public boolean isCancelled() {
    return false;
  }

  @Override
  public boolean isDone() {
    // 异常状态或者完成状态，都叫“运行完成”
    return this.ruleRuntimeContext.getStatus() != RuleRunTimeStatus.NORMAL;
  }

  @Override
  public Object get() throws InterruptedException, ExecutionException {
    LockSupport.park(this.ruleRuntimeContext);
    if (this.ruleRuntimeContext.getStatus() == RuleRunTimeStatus.EXCEPTION) {
      throw new IllegalArgumentException(ruleRuntimeContext.getCurrentThrowable().getMessage());
    }
    if (this.processCallback != null) {
      this.processCallback.doInvocation(ruleRuntimeContext);
    }
    return this.ruleRuntimeContext.get_return();
  }

  @Override
  public Object get(long timeout, TimeUnit unit)
          throws InterruptedException, ExecutionException, TimeoutException {
    Validate.isTrue(timeout > 0, "DefinitionProcessFuture.get，参考的阻塞时间值必须大于0");
    // 默认为毫秒
    TimeUnit currentUnit = unit;
    if (currentUnit == null) {
      currentUnit = TimeUnit.MILLISECONDS;
    }

    LockSupport.parkNanos(this.ruleRuntimeContext, TimeUnit.NANOSECONDS.convert(timeout, currentUnit));
    if (this.ruleRuntimeContext.getStatus() == RuleRunTimeStatus.EXCEPTION) {
      throw new IllegalArgumentException(ruleRuntimeContext.getCurrentThrowable().getMessage());
    }
    return this.ruleRuntimeContext.get_return();
  }
}
