package com.bizunited.platform.mars.policy.process.executor;

import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNode;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.ext.RuntimeExtNode;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeDelayNode;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeStartAble;
import com.bizunited.platform.mars.policy.process.rule.Ruleable;
import com.bizunited.platform.mars.policy.process.rule.ext.AbstractExtRuleable;
import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeNodeService;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeProcessorLinkedService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.orm.jpa.EntityManagerFactoryUtils;
import org.springframework.orm.jpa.EntityManagerHolder;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import java.util.List;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.locks.LockSupport;

/**
 * 注意：一个ProcessorLinkedChain对象对应规则链路的一次运行</br>
 * 可使用模板模式进行代码化简，下个版本进行
 *
 * @author yinwenjie
 */
@Primary
@Component("ProcessorLinkedChainRunnable")
@Scope("prototype")
public final class ProcessorLinkedChainRunnable implements ProcessorChain, Runnable {
  private static final Logger LOGGER = LoggerFactory.getLogger(ProcessorLinkedChainRunnable.class);
  /**
   * 当前正在运行的规则定义链所使用的运行时上下文信息
   */
  private RuleRuntimeContext ruleRuntimeContext;
  /**
   * 当前正在运行的规则定义链
   */
  private RuntimeProcessorLinked runtimeProcessorLinked;
  @Autowired
  @Qualifier("_delayTimerQueue")
  private DelayQueue<RuntimeDelayNode> delayQueue;
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeProcessorLinkedService runtimeProcessorLinkedService;
  /**
   * 当前规则链路中以运行的节点数量（递归层次数量）
   */
  private int count = 0;
  @Autowired
  private List<Ruleable<RuntimeNode>> ruleables;

  public ProcessorLinkedChainRunnable() {

  }

  /**
   * 构造函数，注释未写
   *
   * @param currentNode
   * @param ruleRuntimeContext
   * @param runtimeProcessorLinked
   */
  public ProcessorLinkedChainRunnable(RuleRuntimeContext ruleRuntimeContext, RuntimeProcessorLinked runtimeProcessorLinked) {
    this.ruleRuntimeContext = ruleRuntimeContext;
    this.runtimeProcessorLinked = runtimeProcessorLinked;
  }

  @Override
  public void run() {
    try {
      // 用户身份信息写入当前线程
      SecurityContext ctx = SecurityContextHolder.createEmptyContext();
      ctx.setAuthentication((Authentication) this.ruleRuntimeContext.getParams().get("_authentication"));
      SecurityContextHolder.setContext(ctx);
      // jpa session 写入当前线程
      EntityManagerFactory emFactory = (EntityManagerFactory) this.ruleRuntimeContext.getParams().get("_emFactory");
      if (!TransactionSynchronizationManager.hasResource(emFactory)) {
        EntityManager em = emFactory.createEntityManager();
        EntityManagerHolder emHolder = new EntityManagerHolder(em);
        TransactionSynchronizationManager.bindResource(emFactory, emHolder);
      }
      this.doProcessNode(this.ruleRuntimeContext, runtimeProcessorLinked);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      ruleRuntimeContext.setCurrentThrowable(e);
      ruleRuntimeContext.setExceptionLevel(0);
      ruleRuntimeContext.setStatus(RuleRunTimeStatus.EXCEPTION);
      Thread requestThread = ruleRuntimeContext.getRequestThread();
      if (requestThread != null) {
        LockSupport.unpark(requestThread);
      }
    } finally {
      // 清理当前异步线程中的用户信息
      SecurityContextHolder.clearContext();
      // 清理当前异步线程中的jpa session
      EntityManagerFactory emFactory = (EntityManagerFactory) this.ruleRuntimeContext.getParams().get("_emFactory");
      if (TransactionSynchronizationManager.hasResource(emFactory)) {
        EntityManagerHolder emHolder = (EntityManagerHolder)
                TransactionSynchronizationManager.unbindResource(emFactory);
        EntityManagerFactoryUtils.closeEntityManager(emHolder.getEntityManager());
      }
    }
  }

  // 由ProcessorLinkedChain驱动的多元递归
  public void doProcessNode(RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked) {
    Validate.notNull(context, "规则实例上下文信息不能为空!!");
    Validate.notNull(runtimeProcessorLinked, "规则链路信息不能为空!!");
    RuntimeNode currentNode = runtimeProcessorLinked.getCurrentNode();
    if (currentNode == null && this.count == 0) {
      currentNode = runtimeProcessorLinked.getFounder();
      this.runtimeProcessorLinkedService.next(runtimeProcessorLinked, currentNode);
    } else if (currentNode == null) {
      Validate.notNull(currentNode, "没有找到规则链路定义的开始节点，这种情况一般由于规则定义结构受到破坏，请检查!!");
    }

    /*
     *
     * 整个调用过程使用责任链模式，基于多元递归进行实现，在进行边界验证后，具体的实现方式如下：
     *
     * 1、确定是否是首次递归，如果是，则需要判定当前开始节点是否需要重新加载执行，既RuntimeStartAble.reloadStartNode()
     * 是否返回true；如果是则需要重加载执行。（进入递归，后续过程不会执行）
     *
     * 2、如果步骤1的判定不成立，且当前运行时节点已经是RuntimeEndAble处理性质的节点，则整个工作过程结束
     * （结束递归，后续过程不会执行）
     *
     * 3、如果不是首次递归，但又是RuntimeStartAble性质的运行节点，说明当前规则链路定义到了一个特殊的等待环节
     * 将当前节点在引用了context上下文后，送入延迟等待队列
     * （结束递归，后续过程不会执行）
     *
     * 4、根据当前节点的连线（正常连线）信息，对连线路由进行判定，以便确定下一个规则运行节点（特别是条件性质的运行节点）
     * 注意：此步骤可能出现错误，如果出现错误，则直接向上抛出异常，无需处理，因为第5步会处理
     *
     * 5、根据上一步骤找到的Node的处理性质，确定具体的处理类，不同的节点处理性质处理类完全不一样。
     * 处理过程主要由记录在RuntimeRuleMapping映射关系中的，当前节点对应的处理器进行处理
     * （递归结束，除非抛出异常，否则不会继续执行了）
     *
     * 6、TODO 异常线暂不处理
     * 如果调用过程抛出异常，则做如下处理：
     *  6.1、判定当前节点连线是否存在异常连线信息，如果有，则对线路进行判定（主要是异常类型），以便确定下一个规则运行节点
     *  注意：异常线路同样需要有结束节点的约束
     *  (此步骤，等于把第5步再执行一次)
     *  6.2、如果没有异常线路，或者异常线路判定不符合，则结束处理。
     *
     *  注意：如果整个处理过程存在异常，则异常一定会向最外层抛出
     * */
    Integer currentLevel = count++;
    Ruleable<RuntimeNode> currentRuleable = this.findRuleable(currentNode);
    Validate.notNull(currentRuleable, "没发现指定运行时节点[%s]，对应的执行器Runable，请检查!!", currentNode.getClass().getName());
    // 1、======
    // 如果条件成立，说明当前链路定义第一次执行
    if (currentLevel == 0) {
      RuntimeStartAble runtimeStartAble = (RuntimeStartAble) currentNode;
      if (runtimeStartAble.reloadStartNode()) {
        currentRuleable.doProcess(currentNode, context, runtimeProcessorLinked, this);
        return;
      }
    }
    // 2、====== 结束节点
    else if (currentNode.getType() == RuntimeNodeType.END) {
      // 如果条件成立，说明整个规则定义在之前的节点运行处理过程中
      // 就已经报错了，这里只不过是异常分支线处理完成了，所以依然需要抛出异常,而不能算正常结束
      if (context.getStatus() == RuleRunTimeStatus.EXCEPTION) {
        throw new IllegalArgumentException(context.getCurrentThrowable());
      }
      // 组织运行结果，并进行返回
      context.setStatus(RuleRunTimeStatus.DONE);
      RuntimeDefinition currentDefinition = context.getRuntimeDefinition();
      String returnParam = currentDefinition.getReturnParam();
      if (!StringUtils.isBlank(returnParam)) {
        context.set_return(context.getParams().get(returnParam));
      }
      // 解除可能被阻塞的请求线程的状态
      if (context.getRequestThread() != null) {
        LockSupport.unpark(context.getRequestThread());
      }
      LOGGER.info("运行结束节点【{}】,链路编号【{}】,链路版本号【{}】", currentNode.getCode(), currentDefinition.getCode(), currentDefinition.getCverion());
      return;
    }
    // 3、======
    else if (RuntimeStartAble.class.isAssignableFrom(currentNode.getClass())
            && (currentNode != runtimeProcessorLinked.getFounder())) {
      // 包裹context上下文引用后，放入延迟队列
      RuntimeDelayNode runtimeDelayNode = (RuntimeDelayNode) currentNode;
      runtimeDelayNode.setRuleRuntimeContext(context);
      this.runtimeProcessorLinkedService.end(runtimeProcessorLinked, currentNode);
      delayQueue.add(runtimeDelayNode);
      return;
    }

    // 4、=======寻找下一个节点运行时——在相同的规则链路中
    /*
     * 寻找节点分为多种情况：
     * a、如果当前节点是规则定义的开始节点，则需要从runtimeProcessorLinked的getFoundNext取出连接线
     * b、如果当前节点不是规则定义的开始节点，则需要从runtimeNodeService服务的findNextByDefinition方法中取出连接线
     * 以上两种获取途径，都应该能正确获取到连接线，如果为null，则需要抛出异常
     * */
    RuntimeNodeNexts runtimeNodeNext = null;
    if (currentNode == runtimeProcessorLinked.getFounder()) {
      runtimeNodeNext = runtimeProcessorLinked.getFoundNext();
      Validate.notNull(runtimeNodeNext, "根据当前规则运行时上下文，没有找到符合条件的下一运行时节点，请检查!!");
      currentNode = this.runtimeNodeService.findByCodeAndContext(runtimeNodeNext.getToNodeCode(), context);
    } else {
      currentNode = runtimeNodeService.findNextByContext(currentNode, context);
      Validate.notNull(currentNode, "根据当前规则运行时上下文，没有找到符合条件的下一运行时节点，请检查!!");
    }

    // 5.======
    LOGGER.info("当前执行节点code：nodeCode【{}】, type【{}】 ", currentNode.getCode(), currentNode.getType());
    currentRuleable = this.findRuleable(currentNode);
    Validate.notNull(currentRuleable, "没发现指定运行时节点[%s]，对应的执行器Runable，请检查!!", currentNode.getClass().getName());
    try {
      this.runtimeProcessorLinkedService.next(runtimeProcessorLinked, currentNode);
      currentRuleable.doProcess(currentNode, context, runtimeProcessorLinked, this);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      if (context.getStatus() == RuleRunTimeStatus.NORMAL) {
        Throwable currente = e;
        context.setCurrentThrowable(currente);
        context.setStatus(RuleRunTimeStatus.EXCEPTION);
        context.setExceptionLevel(currentLevel);
        this.runtimeProcessorLinkedService.exception(runtimeProcessorLinked, currentNode, e);
      }
    }

    // 6.====== TODO 异常线暂不处理
  }

  /**
   * 该私有方法为指定的运行时节点找到对应的ruleable处理器
   *
   * @param currentNode
   * @return
   */
  @SuppressWarnings("unchecked")
  private Ruleable<RuntimeNode> findRuleable(RuntimeNode currentNode) {
    /*
     * 分为两种找到ruleable的方式，一种是通过Ruleable.mapping()方法的实现去找
     * 如果当type == 9 时，就是通过AbstractExtRuleable.mappingExtType()方法的返回值去找
     * */
    RuntimeNodeType runtimeNodeType = currentNode.getType();
    // 如果条件成立，则满足type == 9 业务定义节点的处理器转换
    if (runtimeNodeType == RuntimeNodeType.CUSTOM && currentNode instanceof RuntimeExtNode) {
      RuntimeExtNode runtimeExtNode = (RuntimeExtNode) currentNode;
      for (Ruleable<? extends RuntimeNode> item : ruleables) {
        if (!AbstractExtRuleable.class.isAssignableFrom(item.getClass())) {
          continue;
        }

        AbstractExtRuleable extRuleable = (AbstractExtRuleable) item;
        int mappingExtType = extRuleable.mappingExtType();
        if (mappingExtType == runtimeExtNode.getExtType().intValue()) {
          return (Ruleable<RuntimeNode>) item;
        }
      }
    }
    // 其它type时，进行此分支的执行
    else {
      for (Ruleable<RuntimeNode> item : ruleables) {
        if (item.mapping() == currentNode.getClass()) {
          return item;
        }
      }
    }

    return null;
  }
}