package com.bizunited.platform.mars.service.process;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.mars.service.cache.RuntimeDefinition;
import com.bizunited.platform.mars.service.cache.RuntimeNode;
import com.bizunited.platform.mars.service.rule.Ruleable;
import com.bizunited.platform.mars.service.rule.conditions.ConditionRuleable;
import com.bizunited.platform.mars.service.rule.end.EndRuleable;
import com.bizunited.platform.mars.service.rule.mutex.MutexLockRuleable;
import com.bizunited.platform.mars.service.rule.processes.ProcessRuleable;
import com.bizunited.platform.mars.service.rule.start.StarterRuleable;

/**
 * 注意：一个ApplicationProcessChain对象在一次规则运行过程中就是一个新的实例。
 * @author yinwenjie
 */
@Primary
@Component("ApplicationProcessChain")
@Scope("prototype")
public final class ApplicationProcessChain implements ProcessChain {
  private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationProcessChain.class);
  /**
   * 为什么要将当前规则执行过程的上下文放到当前线程中？
   * 主要是为了方便脚本类的处理节点，再被代理的情况下，也能使用当前处理过程的上线文信息
   */
  private static ThreadLocal<RuleRuntimeContext> ruleRuntimeContext = new ThreadLocal<>();
  /**
   * 当前运行的规则定义
   */
  private RuntimeDefinition currentDefinition;
  /**
   * 当前正在运行的规则节点
   */
  private RuntimeNode currentNode;
  /**
   * 当前以运行的节点数量（递归层次数量）
   */
  private int count = 0;
  /**
   * spring上下文
   */
  private ApplicationContext applicationContext;
  
  public ApplicationProcessChain() {
    
  }
  
  // 由ApplicationProcessChain驱动的多元递归
  public void doProcessNode(Map<String, Object> invokeOutputs , ProcessChainInvocation invocation) {
    Validate.notNull(currentDefinition , "规则定义信息不能为空!!");
    Validate.notNull(applicationContext , "上下文信息不能为空!!");

    /*
     * 整个调用过程使用责任链模式，基于多元递归进行实现，在进行边界验证后，具体的实现方式如下：
     * 
     * 1、确定是否是首次递归，如果是，则进行开始节点、上下文等信息的初始化工作
     * 
     * 2、如果步骤1的判定不成立，且当前运行时节点已经是endRule处理性质的节点，则整个工作过程结束
     * 
     * 3、如果以上两个步骤都不成立，则说明该调用是由当前运行时节点中的doProcessNode方法调用引起的
     * 则首先根据当前传入的invokeOutputs信息（K-V信息），对上下文数据进行修正
     * 
     * 4、根据当前节点的连线（正常连线）信息，对连线路由进行判定，以便确定下一个规则运行节点
     * 注意：此步骤可能出现错误，如果出现错误，则直接向上抛出异常，无需处理，因为第5步会处理
     * 
     * 5、根据上一步骤找到的Node的处理性质，确定具体的处理类，不同的节点处理性质处理类完全不一样，处理如下：
     *   5.1、判定操作节点的处理器性质是否正确
     *   5.2、之后，调用Ruleable.doProcessNode进行节点处理
     * 
     * 6、如果调用过程抛出异常，则做如下处理：
     *  6.1、判定当前节点连线是否存在异常连线信息，如果有，则对线路进行判定（主要是异常类型），以便确定下一个规则运行节点
     *  注意：异常线路同样需要有结束节点的约束
     *  (此步骤，等于把第5步再执行一次)
     *  6.2、如果没有异常线路，或者异常线路判定不符合，则结束处理。
     *  
     *  注意：如果整个处理过程存在异常，则异常一定会向最外层抛出
     * */
    // 取得当前processDefinition中的开始节点
    Validate.notNull(currentDefinition , "未发现要运行的规则定义，不允许执行!!");
    RuntimeNodeService runtimeNodeService = applicationContext.getBean(RuntimeNodeService.class);
    ClassLoader appClassLoader = applicationContext.getClassLoader();
    Integer currentLevel = count++;
    
    // 1、======
    // 当前的运行的规则上下文定义(写入线程上下文的原因已进行说明)
    RuleRuntimeContext currentContext = ruleRuntimeContext.get();
    if(currentContext == null) {
      currentContext = new RuleRuntimeContext();
      currentContext.setStatus(RuleProcessStatus.NORMAL);
      currentContext.setCurrentDefinition(this.currentDefinition);
      currentContext.setCurrentThread(Thread.currentThread());
      if(invokeOutputs != null) {
        currentContext.setParams(new HashMap<>(invokeOutputs));
      } else {
        currentContext.setParams(new HashMap<>());
      }
      ruleRuntimeContext.set(currentContext);
    }
    if(currentNode == null) {
      currentNode = runtimeNodeService.findStartByCodeAndVersion(currentDefinition);
      Validate.notNull(currentNode , "没有找到规则定义的开始节点，请检查!!");
    }
    // 2、======
    // 结束节点
    else if(currentNode.getType() == 5){
      // 如果条件成立，说明整个规则运行实例在之前的规则节点处理过程中
      // 就已经报错了，这里只不过是异常分支线处理完成了，所以依然需要抛出异常
      // 而不能算正常结束
      if(currentContext.getStatus() == RuleProcessStatus.EXCEPTION) {
        throw new IllegalArgumentException(currentContext.getCurrentThrowable());
      }
      currentContext.setStatus(RuleProcessStatus.DONE);
      String returnParam = currentDefinition.getReturnParam();
      if(!StringUtils.isBlank(returnParam)) {
        currentContext.set_return(currentContext.getParams().get(returnParam));
      }
      return;
    }
    // 3、======
    else {
      Map<String, Object> contextParams = currentContext.getParams();
      if(!CollectionUtils.isEmpty(invokeOutputs)) {
        contextParams.putAll(invokeOutputs);
      }
    }
    // 4、=======进行下一个节点运算
    if(currentLevel != 0) {
      currentNode = runtimeNodeService.findNextByDefinition(currentNode, currentContext);
      // 如果条件成立，则代表出现异常，向上抛出
      Validate.notNull(currentNode , "根据当前规则运行时上下文，没有找到符合条件的下一运行时节点，请检查!!");
    }
    // 5.1、======
    Class<?> nodeRuleableClass = parseRuleable(appClassLoader);
    Ruleable ruleable  = (Ruleable)applicationContext.getBean(nodeRuleableClass);
    // 5.2、====== 开始调用(currentLevel == 0 时调用开始节点)
    try {
      ruleable.doProcessNode(currentNode, currentContext, this);
    } catch(Exception e) {
      if(currentContext.getStatus() == RuleProcessStatus.NORMAL) {
        Throwable currente = e;
        if(currente instanceof InvocationTargetException) {
          currente = ((InvocationTargetException)e).getTargetException();
        }
        currentContext.setCurrentThrowable(currente);
        currentContext.setStatus(RuleProcessStatus.EXCEPTION);
        currentContext.setExceptionLevel(currentLevel);
      }
    } finally {
      // 如果条件成立，说明整个规则定义已经执行完毕()
      // 并且需要将当前线程上线文中的规则上线文对象清理掉
      // 并且完成可能的回调，让外部调用者拿到可能的返回值
      if(currentLevel == 0) {
        String returnParam = currentDefinition.getReturnParam();
        if(!StringUtils.isBlank(returnParam)) {
          Object returnValue = currentContext.getParams().get(returnParam);
          if(returnValue != null) {
            currentContext.set_return(returnValue);
          }
          if(invocation != null) {
            invocation.success(currentContext);
          }
        } 
        ruleRuntimeContext.set(null);
      } 
    } 
    
    // 6.1、========
    if(currentContext.getStatus() == RuleProcessStatus.EXCEPTION
      && currentLevel == currentContext.getExceptionLevel()) {
      RuntimeNode errorNode = runtimeNodeService.findExceptionNextByDefinition(currentNode, currentContext);
      if(errorNode != null) {
        currentNode = errorNode;
        nodeRuleableClass = parseRuleable(appClassLoader);
        ruleable  = (Ruleable)applicationContext.getBean(nodeRuleableClass);
        // 和步骤5的运行区别在于，就算出现了异常也无需捕获了
        // 会以规则运行时，之前已经抛出的异常为准
        try {
          ruleable.doProcessNode(currentNode , currentContext, this);
        } catch(RuntimeException e) {
          LOGGER.error(e.getMessage() , e);
        } 
      } 
    } 
    // 6.2、=========
    if(currentContext.getStatus() == RuleProcessStatus.EXCEPTION) {
      throw new IllegalArgumentException(currentContext.getCurrentThrowable());
    } 
  } 
  
  /**
   * 负责基于当前规则节点的类型和已获得的处理类
   * 验证处理类的类型正确性
   */
  private Class<?> parseRuleable(ClassLoader appClassLoader) {
    Integer nodeType = currentNode.getType();
    String nodeRuleableClassName = currentNode.getRuleableClass();
    Class<?> nodeRuleableClass = null;
    try {
      nodeRuleableClass = appClassLoader.loadClass(nodeRuleableClassName);
    } catch(ClassNotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    // 5.1、======
    if(!(Ruleable.class.isAssignableFrom(nodeRuleableClass))) {
      throw new IllegalArgumentException(String.format("当前规则处理器[%s]并没有实现Ruleable接口，请检查!!", nodeRuleableClass.getName()));
    }
    switch (nodeType) {
      // 判定组件
      case 1:
        if(!(ConditionRuleable.class.isAssignableFrom(nodeRuleableClass))) {
          throw new IllegalArgumentException("判定组件的处理器必须实现ConditionRuleable接口，请检查!!");
        }
        break;
      // 逻辑处理组件
      case 2:
        if(!(ProcessRuleable.class.isAssignableFrom(nodeRuleableClass))) {
          throw new IllegalArgumentException("逻辑处理组件的处理器必须实现ProcessRuleable接口，请检查!!");
        }
        break;
      // 锁组件
      case 3:
        if(!(MutexLockRuleable.class.isAssignableFrom(nodeRuleableClass))) {
          throw new IllegalArgumentException("锁组件的处理器必须实现ProcessRuleable接口，请检查!!");
        }
        break;
      // 开始组件
      case 4:
        if(!(StarterRuleable.class.isAssignableFrom(nodeRuleableClass))) {
          throw new IllegalArgumentException("开始组件的处理器必须实现ProcessRuleable接口，请检查!!");
        }
        break;
      // 结束组件
      case 5:
        if(!(EndRuleable.class.isAssignableFrom(nodeRuleableClass))) {
          throw new IllegalArgumentException("结束组件的处理器必须实现ProcessRuleable接口，请检查!!");
        }
        break;
      default:
        break;
    }
    return nodeRuleableClass;
  }
  
  /**
   * 从当前线程上下文中获取规则执行上下文
   * @return
   */
  public static RuleRuntimeContext getRuleRuntimeContext() {
    return ApplicationProcessChain.ruleRuntimeContext.get();
  }

  public void setCurrentDefinition(RuntimeDefinition currentDefinition) {
    this.currentDefinition = currentDefinition;
  }

  public void setApplicationContext(ApplicationContext applicationContext) {
    this.applicationContext = applicationContext;
  }
}
