package com.bizunited.platform.mars.service.process;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.mars.entity.RuleDefinitionEntity;
import com.bizunited.platform.mars.entity.RuleNodeNextsEntity;
import com.bizunited.platform.mars.entity.RuleNodeParamsEntity;
import com.bizunited.platform.mars.entity.RuleNodeEntity;
import com.bizunited.platform.mars.entity.RuleTemplateNodeEntity;
import com.bizunited.platform.mars.entity.RuleTemplateParamsEntity;
import com.bizunited.platform.mars.service.RuleDefinitionService;
import com.bizunited.platform.mars.service.RuleDefinitionListener;
import com.bizunited.platform.mars.service.RuleNodeEntityService;
import com.bizunited.platform.mars.service.RuleNodeNextsEntityService;
import com.bizunited.platform.mars.service.RuleNodeParamsEntityService;
import com.bizunited.platform.mars.service.RuleTemplateNodeEntityService;
import com.bizunited.platform.mars.service.cache.RuntimeDefinition;
import com.bizunited.platform.mars.service.cache.RuntimeNode;
import com.bizunited.platform.mars.service.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.service.cache.RuntimeNodeParams;
import com.bizunited.platform.venus.common.service.file.VenusFileService;

/**
 * 规则定义运行时服务的实现
 * @author yinwenjie
 */
@Component("SimpleRuntimeDefinitionService")
public class SimpleRuntimeDefinitionService extends AbstractRuntimeService implements RuntimeDefinitionService , RuleDefinitionListener {
  
  @Autowired
  private RuleDefinitionService ruleDefinitionService;
  @Autowired
  private RuleNodeEntityService ruleNodeService;
  @Autowired
  private RuleNodeNextsEntityService ruleNodeNextsService;
  @Autowired
  private RuleNodeParamsEntityService ruleNodeParamsService;
  @Autowired
  private RuleTemplateNodeEntityService ruleTemplateNodeService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedissonClient redissonClient;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private ApplicationContext applicationContext;
  /**
   * key前缀
   */
  private static final String PRE = "_MARS";
  private static final Logger LOGGER = LoggerFactory.getLogger(SimpleRuntimeDefinitionService.class);
  
  @Override
  public RuntimeDefinition findDetailsByCodeAndVersion(String code, String version) {
    if(StringUtils.isBlank(code) || StringUtils.isBlank(version)) {
      return null;
    }
    // 如果有缓存，则从缓存中查询
    RuntimeDefinition cacheRuntimeDefinition = this.findRedisByCodeAndVersion(code, version);
    if(cacheRuntimeDefinition != null) {
      LOGGER.info("// === 从缓存中加载!!cacheRuntimeDefinition = " + cacheRuntimeDefinition);
      return cacheRuntimeDefinition;
    }
    
    /*
     * 该方法不仅基于持久化数据查找规则定义信息，而且还将信息缓存到redis中，处理步骤为：
     * 1、根据数据层查询结果构建节点信息
     * 2、根据数据层查询结果，构建节点间连线信息（1、2是一并处理的）
     * 3、根据目前支持的节点处理源的类型，构造节点处理源信息
     * 4、为每一个node构建连线信息
     * */
    RuleDefinitionEntity ruleDefinition = ruleDefinitionService.findDetailsByCodeAndVersion(code, version);
    if(ruleDefinition == null) {
      return null;
    }
    RuntimeDefinition runtimeDefinition = nebulaToolkitService.copyObjectByWhiteList(ruleDefinition, RuntimeDefinition.class, HashSet.class, ArrayList.class , "nodes", "nodes.templateNode" , "nodes.inputs" , "nodes.outputs");
    Set<RuntimeNode> runTimeNodes = runtimeDefinition.getNodes();
    if(CollectionUtils.isEmpty(runTimeNodes)) {
      return runtimeDefinition;
    }
    // 1、构造运行时节点基本信息
    // 2、节点定义的入参和出参已被中考，不用再进行额外处理
    for(RuntimeNode runtimeNode : runTimeNodes) {
      // 寻找对应的ruleNodeTemplate信息
      String runtimeNodeId = runtimeNode.getId();
      RuleNodeEntity currentRuleNode = ruleDefinition.getNodes().stream().filter(ruleNodeItem -> StringUtils.equals(ruleNodeItem.getId(), runtimeNodeId)).findFirst().orElse(null);
      String ruleCode = currentRuleNode.getCode();
      RuleTemplateNodeEntity currentRuleTemplateNode = currentRuleNode.getTemplateNode();
      runtimeNode.setTemplateNodeId(currentRuleTemplateNode.getId());
      runtimeNode.setRuleableClass(currentRuleTemplateNode.getRuleableClass());

      // 3、构造节点处理源信息(目前支持三种处理源，只有逻辑组件节点需要设定）
      int type = runtimeNode.getType();
      if(type == 2) {
        int sourceType = runtimeNode.getSourceType();
        // 数值：1、服务源；2、数据视图（聚集）、3、后台脚本；4、数据视图（集合）【暂时没有】；5、全动态脚本
        switch (sourceType) {
          case 1:
            runtimeNode.setSourceServicable(super.parseByRuntimeSourceServicable(currentRuleTemplateNode, ruleCode));
            break;
          case 2:
            runtimeNode.setSourceAggregateDataView(super.parseByRuntimeSourceAggregateDataView(currentRuleTemplateNode, ruleCode));
            break;
          case 3:
            runtimeNode.setSourceScript(super.parseByRuntimeSourceScript(currentRuleTemplateNode, ruleCode));
            break;
          case 5:
            // 全动态脚本无需进行扩展
            break;
            // TODO 其它暂未实现
          default:
            throw new IllegalArgumentException(String.format("解析节点定义[%s]的处理源时，发现错误的sourceType类型值，请检查!!" , ruleCode));
        }
      }
    }
    
    // 4、为每一个node构建连线信息（依据from节点为查询基准）
    runTimeNodes.stream().forEach(runTimeNode -> {
      String runTimeNodeId = runTimeNode.getId();
      Set<RuleNodeNextsEntity> nexts = this.ruleNodeNextsService.findDetailsByFromNode(runTimeNodeId);
      Set<RuntimeNodeNexts> runtimeNexts = new HashSet<>();
      if(nexts != null) {
        nexts.stream().forEach (nextItem -> {
          final String fromNodeId = nextItem.getFromNode().getId();
          final String toNodeId = nextItem.getToNode().getId();
          String toNodeCode = nextItem.getToNode().getCode();
          String conditions = nextItem.getConditions();
          Integer lineType = nextItem.getLineType();
          Integer sort = nextItem.getSort();
          String exceptions = nextItem.getExceptions();
          
          // 检验id的正确性
          if(runTimeNodes.stream().filter(item -> item.getId() == fromNodeId).count() == 0) {
            throw new IllegalArgumentException(String.format("正在构建规则定义，但是未发现当前连线的开始节点[%s]，并不在数据持久层查询到的node节点id信息", fromNodeId));
          }
          if(runTimeNodes.stream().filter(item -> item.getId() == toNodeId).count() == 0) {
            throw new IllegalArgumentException(String.format("正在构建规则定义，但是未发现当前连线的结束节点[%s]，并不在数据持久层查询到的node节点id信息", toNodeId));
          }
          RuntimeNodeNexts runtimeNext = new RuntimeNodeNexts();
          runtimeNext.setConditions(conditions);
          runtimeNext.setExceptions(exceptions);
          runtimeNext.setLineType(lineType);
          runtimeNext.setSort(sort);
          runtimeNext.setToNodeCode(toNodeCode);
          runtimeNexts.add(runtimeNext);
        });
      } 
      runTimeNode.setNexts(runtimeNexts);
    });
    
    // 4、======
    // 计入json缓存
    this.onDefinitionCreate(ruleDefinition);
    return runtimeDefinition;
  }
  
  // 该私有方法试图从redis中查询出规则定义信息
  private RuntimeDefinition findRedisByCodeAndVersion(String code , String version) {
    String key = StringUtils.join(PRE ,"_" , code , "_" , version);
    Map<String, String> map = redissonClient.getMap(key);
    String jsonValue = map.get(key);
    if(StringUtils.isBlank(jsonValue)) {
      return null;
    }
    
    try {
      return JSON.parseObject(jsonValue, RuntimeDefinition.class);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return null;
    }
  }
  
  public void onDefinitionCreate(RuleDefinitionEntity ruleDefinition) {
    if(ruleDefinition == null) {
      return;
    }
    String jsonValue = JSON.toJSONString(ruleDefinition);
    
    /*
     * 一旦规则定义发生变化，相关规则定义在redis中的缓存将会被清楚
     * */
    String code = ruleDefinition.getCode();
    String version = ruleDefinition.getCverion();
    String key = StringUtils.join(PRE ,"_" , code , "_" , version);;
    
    boolean isSuceess = false;
    RLock lock = null;
    lock = this.redissonClient.getLock(key);
    try {
      isSuceess = lock.tryLock(10, TimeUnit.SECONDS);
      Map<String, String> map = redissonClient.getMap(key);
      map.put(key, jsonValue);
    } catch(RuntimeException e) { 
      LOGGER.error(e.getMessage() , e);
    } catch(InterruptedException e)  {
      LOGGER.error(e.getMessage() , e);
      Thread.currentThread().interrupt();
    } finally {
      if(isSuceess) {
        lock.unlock();
      }
    }
  }
  
  /**
   * 当规则定义变化时（属性修改时），该监听器将被触发
   * @param ruleDefinition
   */
  public void onDefinitionModify(RuleDefinitionEntity ruleDefinition) {
    this.onDefinitionCreate(ruleDefinition);
  }
  
  /**
   * 当规则定义变化时（删除时），该监听器将被触发
   * @param ruleDefinition
   */
  public void onDefinitionDelete(RuleDefinitionEntity ruleDefinition) {
    /*
     * 一旦规则定义发生变化，相关规则定义在redis中的缓存将会被清楚
     * */
    String code = ruleDefinition.getCode();
    String version = ruleDefinition.getCverion();
    String key = StringUtils.join(PRE ,"_" , code , "_" , version);;
    
    boolean isSuceess = false;
    RLock lock = null;
    lock = this.redissonClient.getLock(key);
    try {
      isSuceess = lock.tryLock(10, TimeUnit.SECONDS);
      Map<String, String> map = redissonClient.getMap(key);
      map.put(key, null);
    } catch(RuntimeException e) { 
      LOGGER.error(e.getMessage() , e);
    } catch(InterruptedException e)  {
      LOGGER.error(e.getMessage() , e);
      Thread.currentThread().interrupt();
    } finally {
      if(isSuceess) {
        lock.unlock();
      }
    }
  }
  
  @Transactional
  public RuntimeDefinition create(RuntimeDefinition definition , byte[] fileContent) {
    Validate.notNull(definition, "创建规则定义时，定义信息必须传入");
    Validate.isTrue(fileContent != null && fileContent.length > 0 , "创建规则定义时，页面布局文件必须传入");
    String code = definition.getCode();
    String version = definition.getCverion();
    String desc = definition.getDesc();
    String id = definition.getId();
    Validate.isTrue(id == null, "创建规则定义时，其id信息不能有值!!");
    definition.setId(null);
    Validate.notBlank(desc , "创建规则定义时，其desc描述信息不能有值!!");
    RuleDefinitionEntity exsitRuleDefinition = this.ruleDefinitionService.findByCodeAndVersion(code, version);
    Validate.isTrue(exsitRuleDefinition == null , "创建规则定义时，指定的code【%s】和version【%s】已经存在于规则定义中，请重新设定" , code , version);
    Set<RuntimeNode> nodes = definition.getNodes();
    Validate.isTrue(!CollectionUtils.isEmpty(nodes) , "创建规则定义时，未发现任何规则节点定义，请检查!!");
    // 确认只有一个开始节点，有一个或者多个结束节点
    long startNodeCount = nodes.stream().filter(item->item.getType() == 4).count();
    long endNodeCount = nodes.stream().filter(item -> item.getType() == 5).count();
    Validate.isTrue(startNodeCount == 1 , "规则定义中必须有且只有一个开始节点，请检查!!");
    Validate.isTrue(endNodeCount >= 1 , "规则定义中必须至少有一个结束节点，请检查!!");
    
    /*
     * 在完成边界校验后，处理步骤为：
     * 1、首先进行基本信息的添加，包括页面布局文件内容信息
     * 2、然后进行规则节点信息的添加
     *  2.1、包括规则节点的出入参明细信息
     * 3、最后处理节点定义间的连线信息
     * */
    
    // 1、===============
    // 确认文件信息
    String relativePath = this.saveScriptContent();
    String fileName = StringUtils.join(code , "_" , version , ".txt");
    this.venusFileService.saveFile(relativePath, fileName, fileName, fileContent);
    definition.setTemplateFilename(fileName);
    definition.setTemplateRelativepath(relativePath);
    RuleDefinitionEntity ruleDefinition = this.nebulaToolkitService.copyObjectByWhiteList(definition, RuleDefinitionEntity.class, HashSet.class, ArrayList.class);
    // 基本信息保存
    RuleDefinitionEntity newDefinition = this.ruleDefinitionService.create(ruleDefinition);
    definition.setId(newDefinition.getId()); 
    ClassLoader classLoader = applicationContext.getClassLoader();
    
    // 2、===============
    Map<String, RuleNodeEntity> nodeCodeMapping = new HashMap<>();
    nodes.stream().forEach(node -> {
      // 处理节点基本信息
      this.validateAndHandRuntimeNode(ruleDefinition, node, classLoader, nodeCodeMapping);
      // 2.1、处理节点参数信息(参数和使用的处理源情况挂钩)
      this.validateAndHandRuntimeNodeParams(ruleDefinition, node , nodeCodeMapping);
    });
    
    // 3、==========
    // 最后是进行连线
    this.validateAndHandNexts(nodes, nodeCodeMapping);
    return definition;
  }
  
  /**
   * 该私有方法在创建规则定义时，对每一个传入的运行时节点定义信息进行检查和处理
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndHandRuntimeNode(RuleDefinitionEntity ruleDefinition , RuntimeNode node , ClassLoader classLoader , Map<String, RuleNodeEntity> nodeCodeMapping) {
    // 以下是边界校验信息
    String nodeId = node.getId();
    Validate.isTrue(nodeId == null , "创建规则节点定义时，其id信息不能有值!!");
    node.setId(null);
    String nodeCode = node.getCode();
    String templateNodeId = node.getTemplateNodeId();
    Validate.notBlank(templateNodeId , "创建规则节点定义时，其使用的节点模板编号需要指定");
    RuleTemplateNodeEntity templateNode = this.ruleTemplateNodeService.findDetailsById(templateNodeId);
    Validate.notNull(templateNode , "创建规则节点定义时，未发现指定的节点模板信息[%s]" , templateNodeId);
    Validate.isTrue(templateNode.getNodeStatus() == 1 , "创建规则节点定义时，发现至少一个节点模板状态不可用，请检查！！");
    // 模板信息和节点定义信息必须匹配
    RuleNodeEntity exsitRuleNode = this.ruleNodeService.findByCode(nodeCode);
    Validate.isTrue(exsitRuleNode == null , "创建规则节点定义时，发现重复的节点定义code，请检查!!");
    Integer type = node.getType();
    Validate.notNull(type , "创建规则节点定义时，节点类型必须填写");
    Validate.isTrue(type == templateNode.getType() , "创建规则定义时，发现至少一个节点定义的类型和对应的模板节点中的不一致");
    // 如果是逻辑组件，则必须填写处理器扩展信息
    if(type == 2) {
      Integer sourceType = node.getSourceType();
      Validate.notNull(sourceType , "创建规则节点定义时，节点工作逻辑类型必须填写");
      Validate.isTrue(sourceType == templateNode.getSourceType() , "创建规则定义时，发现至少一个节点定义的数据源类型和对应的模板节点中的不一致");
    }
    String ruleableClass = node.getRuleableClass();
    Validate.notBlank(ruleableClass , "创建规则节点定义时，其完整处理器必须指定");
    Validate.isTrue(StringUtils.equals(ruleableClass, templateNode.getRuleableClass()) , "创建规则定义时，发现至少一个节点定义的类型和对应的模板节点中的不一致");
    try {
      Class.forName(ruleableClass, false, classLoader);
    } catch(ClassNotFoundException e) {
      String errorMsg = String.format("创建规则节点定义时，未找到指定的处理器[%s]，请检查!!", ruleableClass);
      throw new IllegalArgumentException(errorMsg, e);
    }
    node.setRuleableClass(ruleableClass);
    
    // 处理规则定义的基本信息（不包括入参和出参信息）
    RuleNodeEntity ruleNode = this.nebulaToolkitService.copyObjectByWhiteList(node, RuleNodeEntity.class, HashSet.class, ArrayList.class);
    ruleNode.setTemplateNode(templateNode);
    ruleNode.setDefinition(ruleDefinition);
    ruleNode = this.ruleNodeService.create(ruleNode);
    node.setId(ruleNode.getId());
    nodeCodeMapping.put(nodeCode, ruleNode); 
  }

  /**
   * 该私有方法在创建规则定义时，对每一个传入的运行时节点定义的参数信息（包括入参和出参）进行检查和处理
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndHandRuntimeNodeParams(RuleDefinitionEntity ruleDefinition , RuntimeNode node , Map<String, RuleNodeEntity> nodeCodeMapping) {
    String templateNodeId = node.getTemplateNodeId();
    String nodeCode = node.getCode();
    RuleTemplateNodeEntity templateNode = this.ruleTemplateNodeService.findDetailsById(templateNodeId);
    // 检验可能的入参和可能的出参信息
    Set<RuleTemplateParamsEntity> inputTemplates = templateNode.getInputs();
    Set<RuleTemplateParamsEntity> outputTemplates = templateNode.getOutputs();
    final Set<RuntimeNodeParams> inputs = node.getInputs() == null?new HashSet<>():node.getInputs();
    final Set<RuntimeNodeParams> outputs = node.getOutputs() == null?new HashSet<>():node.getOutputs();    
    Set<String> inputTemplateParamNames = inputs.stream().map(RuntimeNodeParams::getTemplateParamName).collect(Collectors.toSet());
    Set<String> outputTemplateParamNames = outputs.stream().map(RuntimeNodeParams::getTemplateParamName).collect(Collectors.toSet());
    // 检验模板中规定的入参，是否都进行了设定
    if(!CollectionUtils.isEmpty(inputTemplates)) {
      long count = inputTemplates.stream().filter(item -> inputTemplateParamNames.contains(item.getParamName())).count();
      Validate.isTrue(count == inputTemplates.size() && inputs.size() == inputTemplates.size(), "至少一个节点模板设定的入参，在节点定义中并没有绑定上下文信息，请检查!!");
    }
    // 检验模板中规定的出参，是否都进行了设定
    if(!CollectionUtils.isEmpty(outputTemplates)) {
      long count = outputTemplates.stream().filter(item -> outputTemplateParamNames.contains(item.getParamName())).count();
      Validate.isTrue(count == outputTemplates.size() && outputs.size() == outputTemplates.size() , "至少一个节点模板设定的出参，在节点定义中并没有绑定上下文信息，请检查!!");
    }
    // 接着检查和处理可能的入参
    inputs.stream().forEach(inputItem -> {
      String paramDesc = inputItem.getParamDesc();
      String paramType = inputItem.getParamType();
      String templateParamName = inputItem.getTemplateParamName();
      inputItem.setType(1);
      Validate.notBlank(paramDesc , "入参数描述信息必须填写!!");
      Validate.notBlank(templateParamName , "入参【实参】-模板参数名信息必须填写!!");
      Validate.notNull(paramType , "入参数类型信息必须填写!!");
      RuleNodeParamsEntity paramEntity = this.nebulaToolkitService.copyObjectByWhiteList(inputItem, RuleNodeParamsEntity.class, HashSet.class, ArrayList.class);
      paramEntity.setNode(nodeCodeMapping.get(nodeCode));
      this.ruleNodeParamsService.create(paramEntity);
    });
    // 然后检查和处理可能的出参
    outputs.stream().forEach(outputItem -> {
      String paramDesc = outputItem.getParamDesc();
      String paramType = outputItem.getParamType();
      String templateParamName = outputItem.getTemplateParamName();
      outputItem.setType(2);
      Validate.notBlank(paramDesc , "出参数描述信息必须填写!!");
      Validate.notBlank(templateParamName , "出参【实参】-模板参数名信息必须填写!!");
      Validate.notNull(paramType , "出参数类型信息必须填写!!");
      RuleNodeParamsEntity paramEntity = this.nebulaToolkitService.copyObjectByWhiteList(outputItem, RuleNodeParamsEntity.class, HashSet.class, ArrayList.class);
      paramEntity.setNode(nodeCodeMapping.get(nodeCode));
      this.ruleNodeParamsService.create(paramEntity);
    });
  }

  /**
   * 该私有方法在创建规则定义时，对节点间连线信息进行检查和处理
   * @param ruleDefinition
   * @param node
   * @param classLoader
   * @param nodeCodeMapping
   */
  private void validateAndHandNexts(Set<RuntimeNode> nodes , Map<String, RuleNodeEntity> nodeCodeMapping) {
    nodes.stream().forEach(node -> {
      Integer type = node.getType();
      String nodeCode = node.getCode();
      Set<RuntimeNodeNexts> nexts = node.getNexts() == null?new HashSet<>():node.getNexts();
      Set<RuntimeNodeNexts> normalNexts = nexts.stream().filter(item -> item.getLineType() == 1).collect(Collectors.toSet());
      RuleNodeEntity fromNode = nodeCodeMapping.get(node.getCode());
      String fromNodeCode = fromNode.getCode();
      /*
       * a、如果当前节点类型是条件节点，则才支持一个或者多个分支
       * b、如果当前节点类型是终结，则才支持没有分支
       * b、其余类型的节点，都只支持一个分支路径
       * */
      switch (type) {
        case 1:
          Validate.isTrue(normalNexts.size() >= 1 , "业务编号为[%s]的节点定义,其至少有一个（正常的）后续分支连线，请检查!!" , nodeCode);
          break;
        case 5:
          Validate.isTrue(normalNexts == null || normalNexts.size() == 0 , "业务编号为[%s]的节点定义,其不能有任何（正常的）后续分支连线，请检查!!" , nodeCode);
          break;
        default:
          Validate.isTrue(normalNexts.size() == 1 , "业务编号为[%s]的节点定义,其必须有一个（正常的）后续分支连线，请检查!!" , nodeCode);
          break;
      }
      nexts.stream().forEach(nextItem -> {
        String conditions = nextItem.getConditions();
        String exceptions = nextItem.getExceptions();
        Integer lineType = nextItem.getLineType();
        Integer sort = nextItem.getSort() == null?100:nextItem.getSort();
        String toNodeCode = nextItem.getToNodeCode();
        RuleNodeEntity toNode = nodeCodeMapping.get(toNodeCode);
        Validate.notNull(toNode , "指定的目标定义节点[%s]未被发现，请检查!!" , toNodeCode);
        // 在一个规则定义中，一个定义节点到另一个定义节点，只能有一条连线
        RuleNodeNextsEntity exsitNext = this.ruleNodeNextsService.findByFromAndTo(fromNodeCode, toNodeCode);
        Validate.isTrue(exsitNext == null , "指定的连线从[%s]到[%s]已经存在，不能重复连接，请检查!!" , fromNodeCode , toNodeCode);
        // 开始进行添加
        RuleNodeNextsEntity ruleNext = new RuleNodeNextsEntity();
        ruleNext.setConditions(conditions);
        ruleNext.setExceptions(exceptions);
        ruleNext.setFromNode(fromNode);
        ruleNext.setToNode(toNode);
        ruleNext.setLineType(lineType);
        ruleNext.setSort(sort);
        this.ruleNodeNextsService.create(ruleNext);
      });
    });
  }
  
  @Override
  @Transactional
  public RuntimeDefinition update(RuntimeDefinition definition, byte[] fileContent) {
    Validate.notNull(definition, "修改规则定义时，定义信息必须传入");
    Validate.isTrue(fileContent != null && fileContent.length > 0 , "修改规则定义时，页面布局文件必须传入");
    String desc = definition.getDesc();
    String definitionId = definition.getId();
    Validate.notBlank(definitionId, "修改规则定义时，其id信息必须传入!!");
    Validate.notBlank(desc , "修改规则定义时，其desc描述信息不能没有值!!");
    RuleDefinitionEntity exsitRuleDefinition = this.ruleDefinitionService.findById(definitionId);
    Validate.notNull(exsitRuleDefinition, "修改规则定义时，指定的规则定义未找到[%s]，请重新设定" , definitionId);
    
    Set<RuntimeNode> nodes = definition.getNodes();
    // 确认只有一个开始节点，有一个或者多个结束节点
    long startNodeCount = nodes.stream().filter(item->item.getType() == 4).count();
    long endNodeCount = nodes.stream().filter(item -> item.getType() == 5).count();
    Validate.isTrue(startNodeCount == 1 , "规则定义中必须有且只有一个开始节点，请检查!!");
    Validate.isTrue(endNodeCount >= 1 , "规则定义中必须至少有一个结束节点，请检查!!");
    
    /*
     * 规则定义的全量修改，可以简单理解为：
     * 1、根据指定的id进行参数定义、连线定义、节点定义的删除
     * 2、在进行规则定义基本信息的修改后
     * 3、再重新进行节点定义、参数定义信息的添加操作
     * 4、最后重新进行连线定义信息的添加操作
     * */
    
    // 1、======
    Set<RuleNodeEntity> ruleNodes = this.ruleNodeService.findDetailsByDefinition(definitionId);
    if(!CollectionUtils.isEmpty(ruleNodes)) {
      Set<String> ruleNodeIds = ruleNodes.stream().map(RuleNodeEntity::getId).collect(Collectors.toSet());
      // 进行所有明细信息的删除
      this.ruleNodeParamsService.deleteByRuleNodes(ruleNodeIds.toArray(new String[] {}));
      // 进行所有连线的删除
      this.ruleNodeNextsService.deleteByFromNodes(ruleNodeIds.toArray(new String[] {}));
      // 删除定义下所有的节点信息
      this.ruleNodeService.deleteByIds(ruleNodeIds.toArray(new String[] {}));
    }
    
    // 2、======
    // 确认文件信息
    String relativePath = this.saveScriptContent();
    String fileName = StringUtils.join(exsitRuleDefinition.getCode() , "_" , exsitRuleDefinition.getCverion() , ".txt");
    this.venusFileService.saveFile(relativePath, fileName, fileName, fileContent);
    definition.setTemplateFilename(fileName);
    definition.setTemplateRelativepath(relativePath);
    RuleDefinitionEntity ruleDefinition = this.nebulaToolkitService.copyObjectByWhiteList(definition, RuleDefinitionEntity.class, HashSet.class, ArrayList.class);
    this.ruleDefinitionService.update(ruleDefinition);
    
    // 3、======
    ClassLoader classLoader = applicationContext.getClassLoader();
    Map<String, RuleNodeEntity> nodeCodeMapping = new HashMap<>();
    nodes.stream().forEach(node -> {
      // 处理节点基本信息
      this.validateAndHandRuntimeNode(ruleDefinition, node, classLoader, nodeCodeMapping);
      // 处理节点参数信息
      this.validateAndHandRuntimeNodeParams(ruleDefinition, node , nodeCodeMapping);
    });
    
    // 4、======
    this.validateAndHandNexts(nodes, nodeCodeMapping);
    return definition;
  }

  /**
   * 重复的代码，进行私有方法封装，文件相对路径
   * @param scriptContent
   * @return
   */
  private String saveScriptContent() {
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String relativePath = StringUtils.join("/ruleengine/", folderName, "/", (new Random().nextInt(100) % 10));
    return relativePath;
  }

  @Override
  public Object process(String code, String version, Map<String, Object> params) {
    RuntimeDefinition currentDefinition = this.findDetailsByCodeAndVersion(code, version);
    Validate.notNull(currentDefinition , "错误的规则定义信息[code=%s;version=%s]，请检查!!" , code , version);
    ApplicationProcessChain processChain = applicationContext.getBean(ApplicationProcessChain.class);
    processChain.setApplicationContext(applicationContext);
    processChain.setCurrentDefinition(currentDefinition);
    final RuleRuntimeContext runtimeContext = new RuleRuntimeContext();
    // 开始执行规则定义
    processChain.doProcessNode(params , context -> {
      runtimeContext.set_return(context.get_return());
    });
    return runtimeContext.get_return();
  }
}
