package com.bizunited.platform.mars.service.process.executor;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import com.bizunited.platform.mars.service.cache.RuntimeNode;
import com.bizunited.platform.mars.service.cache.RuntimeNodeParams;
import com.bizunited.platform.mars.service.cache.RuntimeSourceAggregateDataView;
import com.bizunited.platform.mars.service.process.RuleRuntimeContext;

/**
 * 执行器，专门用于执行逻辑处理节点的具体业务过程。该处理器设计用来执行数据视图（聚集）设定下的逻辑过程
 * @author yinwenjie
 */
@Component("aggregateDataViewProcessExecutor")
public class AggregateDataViewProcessExecutor implements ProcessExecutor {
  @Autowired
  private DataViewService dataViewService;
  
  public int getSourceType() {
    return 2;
  }
  public void execute(RuntimeNode currentNode , Map<String , Object> inputParamValues , RuleRuntimeContext context) {
    String code = currentNode.getCode();
    // 取得并判定处理源信息
    RuntimeSourceAggregateDataView runtimeSourceAggregateDataView = currentNode.getSourceAggregateDataView();
    Validate.notNull(runtimeSourceAggregateDataView , "运行节点[%s]时，未发现处理源信息，请检查!!" , code);
    // 取得并判定数据视图
    String viewCode = runtimeSourceAggregateDataView.getViewCode();
    Validate.notNull(viewCode , "在基于数据视图的条件判定处理器工作时，未发现指定的codeView信息，请检查配置");
    DataViewEntity currentDataView =  dataViewService.findDetailsByCode(viewCode);
    Validate.notNull(currentDataView, "未发现指定的数据视图信息[%s]，请检查!!", viewCode);
    // 取得并判定执行源信息
    String aggregateField = runtimeSourceAggregateDataView.getAggregateField();
    Validate.notBlank(aggregateField , "未设定操作列");
    Integer aggregateType = runtimeSourceAggregateDataView.getAggregateType();
    Validate.notNull(aggregateType , "操作类型必须设定");
    // 开始进行数据视图的调用
    ExecuteParamModel dataViewParamModel = new ExecuteParamModel();
    dataViewParamModel.setSystemParams(inputParamValues);
    dataViewParamModel.setDataViewCode(currentDataView.getCode());
    List<JSONObject> result = null;
    try {
      result = this.dataViewService.executeQuery(dataViewParamModel);
    } catch(RuntimeException e) {
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    
    // 开始配定返回值，如果outputParams == null说明没有设定返回值
    Set<RuntimeNodeParams> outputParams = currentNode.getOutputs();
    Validate.isTrue(outputParams == null || (outputParams != null && outputParams.size() == 1) , "运行节点[%s]时，发现其设定的出参不符合要求(要么不设定返回值，要么只设置一个返回值参数)!!");
    if(outputParams == null) {
      return;
    }
    RuntimeNodeParams outputParam = outputParams.iterator().next();
    String contextParamName = outputParam.getContextParamName();
    Integer scale = runtimeSourceAggregateDataView.getScale() == null?0:runtimeSourceAggregateDataView.getScale();
    
    BigDecimal sum = new BigDecimal(0);
    BigDecimal max = new BigDecimal(Integer.MIN_VALUE);
    BigDecimal min = new BigDecimal(Integer.MAX_VALUE);
    int count = 0;
    for(; result != null && count < result.size() ; count++) {
      JSONObject item = result.get(count);
      Double aggregateFieldValue = null;
      try {
        aggregateFieldValue = item.getDouble(aggregateField);
        if(aggregateFieldValue == null) {
          throw new IllegalArgumentException();
        }
      } catch(RuntimeException e) {
        throw new IllegalArgumentException(String.format("在执行数据视图[%s].[%s]时，发现不为数字(或没有设定默认值)的输出结果，请检查数据视图!!", viewCode , aggregateField), e);
      }
      
      // 判定最大值，最小值、累加值
      sum = sum.add(new BigDecimal(aggregateFieldValue));
      if(max.doubleValue() < aggregateFieldValue) {
        max = new BigDecimal(aggregateFieldValue);
      }
      if(min.doubleValue() > aggregateFieldValue) {
        min = new BigDecimal(aggregateFieldValue);
      }
    }
    
    // 1、count；2、sum；3、avg；4、max；5、min
    Map<String , Object> contentMap = context.getParams();
    switch (aggregateType) {
      case 1:
        contentMap.put(contextParamName, count);
        break;
      case 2:
        contentMap.put(contextParamName, sum.setScale(scale , RoundingMode.HALF_UP).doubleValue());
        break;
      case 3:
        if(count == 0) {
          contentMap.put(contextParamName , 0);
        } else {
          contentMap.put(contextParamName, sum.divide(new BigDecimal(count), scale , RoundingMode.HALF_UP).setScale(scale , RoundingMode.HALF_UP).doubleValue());
        }
        break;
      case 4:
        contentMap.put(contextParamName, max.setScale(scale , RoundingMode.HALF_UP).doubleValue());
        break;
      case 5:
        contentMap.put(contextParamName, min.setScale(scale , RoundingMode.HALF_UP).doubleValue());
        break;
      default:
        break;
    }
  }
}
