package com.bizunited.platform.mars.service.process.executor;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.mars.service.cache.RuntimeNode;
import com.bizunited.platform.mars.service.cache.RuntimeNodeParams;
import com.bizunited.platform.mars.service.cache.RuntimeSourceScript;
import com.bizunited.platform.mars.service.process.RuleRuntimeContext;
import com.bizunited.platform.mars.service.process.RuntimeDefinitionService;
import com.bizunited.platform.mars.service.process.RuntimeNodeService;

/**
 * 执行器，专门用于执行逻辑处理节点的具体业务过程。该处理器设计用来执行后端固定groovy脚本设定下的逻辑过程
 * @author yinwenjie
 */
@Component("groovyScriptProcessExecutor")
public class GroovyScriptProcessExecutor implements ProcessExecutor {
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  @Autowired
  private ScriptService scriptService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(GroovyScriptProcessExecutor.class);
  public int getSourceType() {
    return 3;
  }
  public void execute(RuntimeNode currentNode , Map<String , Object> inputParamValues , RuleRuntimeContext context) {
    Validate.notNull(currentNode , "在进行节点定义执行时，必须要有至少一个节点定义RuntimeNode信息，请检查入参!!");
    String code = currentNode.getCode();
    Validate.notBlank(code , "在进行节点定义执行时，节点定义的code必须有值，请检查入参!!");
    // 取得并判定处理源信息
    RuntimeSourceScript runtimeSourceGroovy = currentNode.getSourceScript();
    Validate.notNull(runtimeSourceGroovy , "运行节点[%s]时，未发现处理源信息，请检查!!" , code);
    String language = runtimeSourceGroovy.getLanguage();
    Validate.isTrue(StringUtils.equals(language, "groovy") , "运行节点[%s]时，发现脚本类型不匹配，请检查!!" , code);
    String scriptContent = runtimeSourceGroovy.getScriptContent();
    Validate.notBlank(scriptContent , "运行节点[%s]时，未发现指定的脚本信息，请检查!!" , code);
    String scriptId = runtimeSourceGroovy.getScriptId();
    Validate.notBlank(scriptId , "运行节点[%s]时，未发现指定的脚本Id信息，请检查!!" , code);
    
    // 2、======
    inputParamValues.put("runtimeNodeService", runtimeNodeService);
    inputParamValues.put("runtimeDefinitionService", runtimeDefinitionService);
    Map<String , Object> outputParamValues = null;
    try {
      outputParamValues = this.scriptService.invoke(new String[] {scriptId}, inputParamValues);
    } catch (InvokeProxyException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    if(outputParamValues == null) {
      outputParamValues = new HashMap<>();
    }
    
    // 3、=====
    Map<String , Object> contentMap = context.getParams();
    Set<RuntimeNodeParams> outputParams = currentNode.getOutputs();
    if(outputParams != null) {
      for(RuntimeNodeParams item : outputParams) {
        boolean nullable = item.getNullable();
        // TODO 出参类型还要进行判定
        String templateParamName = item.getTemplateParamName();
        String contextParamName = item.getContextParamName();
        Object outputValue = outputParamValues.get(templateParamName);
        if(outputValue == null && nullable) {
          String errorMsg = String.format("分析运行节点[%s]返回值时，发现指定的出参[%s]设定为not nullable，但是执行结果却没有返回值，请检查", code , templateParamName);
          throw new IllegalArgumentException(errorMsg);
        }
        contentMap.put(contextParamName, outputValue);
      }
    }
  }
}
