package com.bizunited.platform.mars.policy.process.executor;

import java.util.concurrent.DelayQueue;
import java.util.concurrent.locks.LockSupport;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNode;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeDelayNode;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeStartAble;
import com.bizunited.platform.mars.policy.process.rule.Ruleable;
import com.bizunited.platform.mars.policy.process.runtime.RuleRunTimeStatus;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeNodeService;
import com.bizunited.platform.mars.policy.process.runtime.service.RuntimeProcessorLinkedService;

/**
 * 注意：一个ProcessorLinkedChain对象对应规则链路的一次运行</br>
 * 可使用模板模式进行代码化简，下个版本进行
 * @author yinwenjie
 */
@Primary
@Component("ProcessorLinkedChain")
@Scope("prototype")
public final class ProcessorLinkedChain implements ProcessorChain , Runnable {
  private static final Logger LOGGER = LoggerFactory.getLogger(ProcessorLinkedChain.class);
  /**
   * 当前正在运行的规则节点
   */
  private RuntimeNode currentNode;
  /**
   * TODO 注释未写
   */
  private RuleRuntimeContext ruleRuntimeContext;
  /**
   * TODO 注释未写
   */
  private RuntimeProcessorLinked runtimeProcessorLinked;
  @Autowired
  @Qualifier("_delayTimerQueue")
  private DelayQueue<RuntimeDelayNode> delayQueue;
  /**
   * spring上下文
   */
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeProcessorLinkedService runtimeProcessorLinkedService;
  /**
   * 当前规则链路中以运行的节点数量（递归层次数量）
   */
  private int count = 0;
  
  public ProcessorLinkedChain() {
    
  }

  /**
   * 构造函数，注释未写
   * @param currentNode
   * @param ruleRuntimeContext
   * @param runtimeProcessorLinked
   */
  public ProcessorLinkedChain(RuntimeNode currentNode, RuleRuntimeContext ruleRuntimeContext, RuntimeProcessorLinked runtimeProcessorLinked) {
    this.currentNode = currentNode;
    this.ruleRuntimeContext = ruleRuntimeContext;
    this.runtimeProcessorLinked = runtimeProcessorLinked;
  }

  @Override
  public void run() {
    
  }

  // 由ProcessorLinkedChain驱动的多元递归
  @SuppressWarnings({"rawtypes", "unchecked"})
  public void doProcessNode(RuleRuntimeContext context, RuntimeProcessorLinked runtimeProcessorLinked)  {
    Validate.notNull(ruleRuntimeContext , "规则实例上下文信息不能为空!!");
    Validate.notNull(runtimeProcessorLinked , "规则链路信息不能为空!!");
    RuntimeNode currentNode = runtimeProcessorLinked.getCurrentNode();
    if(currentNode == null && this.count == 0) {
      currentNode = runtimeProcessorLinked.getFounder();
      runtimeProcessorLinked.setCurrentNode(currentNode);
    } else if(currentNode == null) {
      Validate.notNull(currentNode , "没有找到规则链路定义的开始节点，这种情况一般由于规则定义结构受到破坏，请检查!!");
    }

    /* 
     * 
     * 整个调用过程使用责任链模式，基于多元递归进行实现，在进行边界验证后，具体的实现方式如下：
     * 
     * 1、确定是否是首次递归，如果是，则需要判定当前开始节点是否需要重新加载执行，既RuntimeStartAble.reloadStartNode()
     * 是否返回true；如果是则需要重加载执行。（进入递归，后续过程不会执行）
     * 
     * 2、如果步骤1的判定不成立，且当前运行时节点已经是RuntimeEndAble处理性质的节点，则整个工作过程结束
     * （结束递归，后续过程不会执行）
     * 
     * 3、如果不是首次递归，但又是RuntimeStartAble性质的运行节点，说明当前规则链路定义到了一个特殊的等待环节
     * 将当前节点在引用了context上下文后，送入延迟等待队列
     * （结束递归，后续过程不会执行）
     * 
     * 4、根据当前节点的连线（正常连线）信息，对连线路由进行判定，以便确定下一个规则运行节点（特别是条件性质的运行节点）
     * 注意：此步骤可能出现错误，如果出现错误，则直接向上抛出异常，无需处理，因为第5步会处理
     * 
     * 5、根据上一步骤找到的Node的处理性质，确定具体的处理类，不同的节点处理性质处理类完全不一样。
     * 处理过程主要由记录在RuntimeRuleMapping映射关系中的，当前节点对应的处理器进行处理
     * （递归结束，除非抛出异常，否则不会继续执行了）
     * 
     * 6、TODO 异常线暂不处理
     * 如果调用过程抛出异常，则做如下处理：
     *  6.1、判定当前节点连线是否存在异常连线信息，如果有，则对线路进行判定（主要是异常类型），以便确定下一个规则运行节点
     *  注意：异常线路同样需要有结束节点的约束
     *  (此步骤，等于把第5步再执行一次)
     *  6.2、如果没有异常线路，或者异常线路判定不符合，则结束处理。
     *  
     *  注意：如果整个处理过程存在异常，则异常一定会向最外层抛出
     * */
    Integer currentLevel = count++;
    Class<? extends Ruleable> currentRuleableClass = RuntimeRuleMapping.MAPPING.get(currentNode.getClass());
    Ruleable currentRuleable = this.applicationContext.getBean(currentRuleableClass);
    
    // 1、======
    // 如果条件成立，说明当前链路定义第一次执行
    if(currentLevel == 0) {
      RuntimeStartAble runtimeStartAble = (RuntimeStartAble)currentNode;
      if(runtimeStartAble.reloadStartNode()) {
        currentRuleable.doProcess(currentNode, context, runtimeProcessorLinked, this);
        return;
      }
    }
    // 2、====== 结束节点
    else if(currentNode.getType() == RuntimeNodeType.END){
      // 如果条件成立，说明整个规则定义在之前的节点运行处理过程中
      // 就已经报错了，这里只不过是异常分支线处理完成了，所以依然需要抛出异常,而不能算正常结束
      if(context.getStatus() == RuleRunTimeStatus.EXCEPTION) {
        throw new IllegalArgumentException(context.getCurrentThrowable());
      }
      // 组织运行结果，并进行返回
      context.setStatus(RuleRunTimeStatus.DONE);
      RuntimeDefinition currentDefinition = context.getRuntimeDefinition();
      String returnParam = currentDefinition.getReturnParam();
      if(!StringUtils.isBlank(returnParam)) {
        context.set_return(context.getParams().get(returnParam));
      }
      // 解除可能被阻塞的请求线程的状态
      if(context.getRequestThread() != null) {
        LockSupport.unpark(context.getRequestThread());
      }
      return;
    } 
    // 3、======
    else if(currentLevel != 0 && RuntimeStartAble.class.isAssignableFrom(currentRuleableClass)) {
      // 包裹context上下文引用后，放入延迟队列
      RuntimeDelayNode runtimeDelayNode = (RuntimeDelayNode)currentNode;
      delayQueue.add(runtimeDelayNode);
      this.runtimeProcessorLinkedService.end(runtimeProcessorLinked, currentNode);
      return;
    } 
    
    // 4、=======寻找下一个节点运行时——在相同的规则链路中
    /*
     * 寻找节点分为多种情况：
     * a、如果当前节点是规则定义的开始节点，则需要从runtimeProcessorLinked的getFoundNext取出连接线
     * b、如果当前节点不是规则定义的开始节点，则需要从runtimeNodeService服务的findNextByDefinition方法中取出连接线
     * 以上两种获取途径，都应该能正确获取到连接线，如果为null，则需要抛出异常
     * */
    RuntimeNodeNexts runtimeNodeNext = null;
    if(this.currentNode == this.runtimeProcessorLinked.getFounder()) {
      runtimeNodeNext = this.runtimeProcessorLinked.getFoundNext();
      Validate.notNull(runtimeNodeNext , "根据当前规则运行时上下文，没有找到符合条件的下一运行时节点，请检查!!");
      currentNode = this.runtimeNodeService.findByCodeAndContext(runtimeNodeNext.getToNodeCode(), context);
    } else {
      currentNode = runtimeNodeService.findNextByContext(currentNode, context);
      Validate.notNull(currentNode , "根据当前规则运行时上下文，没有找到符合条件的下一运行时节点，请检查!!");
    } 
    
    // 5.======
    currentRuleableClass = RuntimeRuleMapping.MAPPING.get(currentNode.getClass());
    currentRuleable = this.applicationContext.getBean(currentRuleableClass);
    try {
      currentRuleable.doProcess(currentNode, context, runtimeProcessorLinked, this);
      this.runtimeProcessorLinkedService.next(runtimeProcessorLinked, currentNode);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      if(context.getStatus() == RuleRunTimeStatus.NORMAL) {
        Throwable currente = e;
        context.setCurrentThrowable(currente);
        context.setStatus(RuleRunTimeStatus.EXCEPTION);
        context.setExceptionLevel(currentLevel);
        this.runtimeProcessorLinkedService.exception(this.runtimeProcessorLinked, currentNode, e);
      }
    }
    
    // 6.====== TODO 异常线暂不处理
  } 
}