package com.bizunited.platform.mars.policy.process.executor;

import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ThreadPoolExecutor;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.cache.waiter.RuntimeDelayNode;
import com.bizunited.platform.mars.policy.process.rule.Ruleable;
import com.bizunited.platform.mars.policy.process.rule.starter.StarterRuleable;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;

/**
 * 这个线程运行器工作于mars_timer_thread_pool线程池中，
 * 主要作用是当某个继承了RuntimeDelayNode类的节点的已经到达阻塞等待时间，
 * 并且离开DelayQueue队列后，就由该线程运行器进行运行。</p>
 * 
 * 运行中的执行器，主要将进行 
 * @author yinwenjie 
 */
public class TimerNotifyRunnable implements Runnable {
  @Autowired
  @Qualifier("_delayTimerQueue")
  private DelayQueue<RuntimeDelayNode> delayQueue;
  @Autowired
  private ApplicationContext applicationContext;
  
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(TimerNotifyRunnable.class);
  
  @SuppressWarnings({"rawtypes", "unchecked"})
  @Override
  public void run() {
    /* 
     * 这个线程，只要不是发现了线程终止信号（已知的场景就只可能是操作系统 kill process id）
     * 都会坚持循环执行
     * */
    Long count = 0l;
    while(count++ < Long.MAX_VALUE) {
      Thread currentThread = Thread.currentThread();
      // TODO 收到终止信号，就进行落盘处理，然后结束
      if(currentThread.isInterrupted()) {
        break;
      }
      
      RuntimeDelayNode currentNode = null;
      try {
        currentNode = this.delayQueue.take();
        // TODO 继续做业务逻辑的处理
      } catch(InterruptedException e) {
        // TODO 在后续版本中，mars模块将再退出前完成未处理任务的落盘备份处理!!
        LOGGER.error("发现终止信号，请避免使用kill -9的方式强制终结进程，在后续版本中，mars模块将再退出前完成未处理任务的落盘备份处理!!");
        Thread.currentThread().interrupt();
        break;
      }
      String instanceId = currentNode.getInstanceId();
      RuleRuntimeContext ruleRuntimeContext = RuleRuntimeContext.getRuleRuntimeContext(instanceId);
      
      /*
       * 执行过程为：
       * 1、首先根据RuntimeDelayMapping映射信息，找到当前节点对应的运行器
       * 注意，由于这里都是继承了RuntimeDelayNode类的节点，所以其运行器一定实现了StarterRuleable接口
       * 否则会报错。
       * 
       * 2、接着调用StarterRuleable接口，以便根据不同的开始节点完成规则链路的创建
       * 由于这个开始节点可能存在多个链路同时执行的情况，所以本消息消费者，需要将执行过程送入到另一个线程池
       * （mars_extcutor_thread_pool）中进行执行
       * */
      // 1、=====
      Class<? extends Ruleable> starterRuleableClass = RuntimeRuleMapping.MAPPING.get(currentNode.getClass());
      Validate.notNull(starterRuleableClass , "没发现指定节点定义类型[%s]，对应的执行器Runable，请检查!!" , currentNode.getClass().getName());
      Ruleable currentRuleable = this.applicationContext.getBean(starterRuleableClass);
      Validate.isTrue(currentRuleable instanceof StarterRuleable , "当前节点并不是一种链路开始节点，请检查设定!!");
      
      // 2、=====
      StarterRuleable<RuntimeDelayNode> currentStarterRuleable = (StarterRuleable<RuntimeDelayNode>)currentRuleable;
      Set<RuntimeProcessorLinked> runtimeProcessorLinkeds = currentStarterRuleable.createProcessorLinkeds(currentNode, ruleRuntimeContext);
      Validate.isTrue(!CollectionUtils.isEmpty(runtimeProcessorLinkeds) , "未发现规则实例[%s]中，匹配任务规则链路的连线信息可以被执行!!");
      // 放入真实的链路执行线程池，开始执行
      ThreadPoolExecutor marsExtcutorExecutor = applicationContext.getBean("_mars_extcutor_thread_pool" , ThreadPoolExecutor.class);
      for (RuntimeProcessorLinked runtimeProcessorLinked : runtimeProcessorLinkeds) {
        ProcessorLinkedChain processorLinkedChain = (ProcessorLinkedChain)applicationContext.getBean("mars_extcutor_thread_pool", currentNode , ruleRuntimeContext , runtimeProcessorLinked);
        marsExtcutorExecutor.submit(processorLinkedChain);
      }
      
      // TODO 需要进行验证检查
//      // 开始执行规则定义
//      processChain.doProcessNode(params , context -> {
//        runtimeContext.set_return(context.get_return());
//      });
//      return runtimeContext.get_return();
    }
  }
}