package com.bizunited.platform.mars.policy.process.runtime.service;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.mars.policy.process.cache.RuntimeDefinition;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNode;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeNexts;
import com.bizunited.platform.mars.policy.process.cache.RuntimeNodeType;
import com.bizunited.platform.mars.policy.process.cache.RuntimeProcessorLinked;
import com.bizunited.platform.mars.policy.process.runtime.contexts.RuleRuntimeContext;

/**
 * 规则节点定义运行时服务的实现
 * @author yinwenjie
 *
 */
@Component("SimpleRuntimeNodeService")
public class SimpleRuntimeNodeService extends AbstractRuntimeService implements RuntimeNodeService {
  
  private static final Logger LOGGER = LoggerFactory.getLogger(SimpleRuntimeNodeService.class);
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private RuntimeNodeService runtimeNodeService;
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  @Autowired
  private ScriptService scriptService;
  
  @Override
  public RuntimeNode findStartByDefinition(RuntimeDefinition currentDefinition) {
    if(currentDefinition == null) {
      return null;
    }
    Set<RuntimeNode> nodes = currentDefinition.getNodes();
    if(CollectionUtils.isEmpty(nodes)) {
      return null;
    }
    
    // 查询开始节点
    return nodes.stream().filter(item -> item.getType() == RuntimeNodeType.BEGIN).findFirst().orElse(null);
  }
  
  @Override
  public RuntimeNode findStartByRuntimeProcessorLinked(RuntimeProcessorLinked runtimeProcessorLinked) {
    if(runtimeProcessorLinked == null) {
      return null;
    }
    return runtimeProcessorLinked.getFounder();
  }

  public Set<RuntimeNode> findEndByDefinition(RuntimeDefinition currentDefinition) {
    if(currentDefinition == null) {
      return null;
    }
    Set<RuntimeNode> nodes = currentDefinition.getNodes();
    if(CollectionUtils.isEmpty(nodes)) {
      return null;
    }
    
    // 查询开始节点
    return nodes.stream().filter(item -> item.getType() == RuntimeNodeType.END).collect(Collectors.toSet());
  }

  @Override
  public RuntimeNode findByCodeAndContext(String nodeCode, RuleRuntimeContext context) {
    if(StringUtils.isBlank(nodeCode)|| context == null) {
      return null;
    }
    RuntimeDefinition runtimeDefinition = context.getRuntimeDefinition();
    Set<RuntimeNode> runtimeNodes = runtimeDefinition.getNodes();
    RuntimeNode currentNode = runtimeNodes.stream().filter(item -> StringUtils.equals(item.getCode(), nodeCode)).findFirst().orElse(null);
    if(!this.validate(currentNode, context)) {
      return null;
    }
    return currentNode;
  }

  @Override
  public RuntimeNode findNextByContext(RuntimeNode currentNode, RuleRuntimeContext context) {
    if(!this.validate(currentNode, context)) {
      return null;
    }
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    Set<RuntimeNode> nodes = context.getRuntimeDefinition().getNodes();
    RuntimeNodeType type = currentNode.getType();
    
    /*
     * 处理过程为:
     * 1、如果当前节点不是条件节点、不是并发节点，则只可能有一个next连线（如果多了，则报错）
     * 2、如果当前节点是条件节点，则做如下处理：
     *   2.1、nexts信息需要排序，然后再根据判定表达式，依次执行判定
     *   2.2、每次执行判定时，都要分析表单式的所使用的对象，并从context上下文中进行获取
     *   2.3、如果以上两个步骤没有找到符合条件的分支，则查找有没有没有conditions设置的默认分支
     * */
    // 1、========
    if(type != RuntimeNodeType.CONDITION && type != RuntimeNodeType.CONCURRENCY) {
      Validate.isTrue(nexts.stream().filter(item -> item.getLineType() == 1).count() == 1l , "当前运行的规则节点定义，不是条件类型或者并行分支类型的节点，不能有多个常规连线，请检查!!");
      RuntimeNodeNexts next = nexts.stream().filter(item -> item.getLineType() == 1).findFirst().orElse(null);
      String nodeCode = next.getToNodeCode();
      // 寻找id对应的节点——在当前定义的所有节点中
      LOGGER.info("规则引擎正在查找下一节点：nodeCode = " + nodeCode);
      RuntimeNode targetNode = null;
      try {
        targetNode = nodes.stream().filter(item -> StringUtils.equals(item.getCode(), nodeCode)).findFirst().orElse(null);
      } catch(RuntimeException e) {
        if(nodes != null) {
          LOGGER.info("96行错误，当前nodes的情况为 " + JSONArray.toJSONString(nodes));
        } else {
          LOGGER.info("96行错误，当前nodes的情况为 null ");
        }
      }
      return targetNode;
    }
    
    // 2、===== 如果执行到这里，就需要进行条件判定或者并行分支了
    Set<RuntimeNodeNexts> sortNexts = nexts.stream().filter(item -> item.getLineType() == 1).sorted((source , target) -> source.getSort() - target.getSort()).collect(Collectors.toSet());
    RuntimeNodeNexts currentNext = null;
    for(RuntimeNodeNexts next : sortNexts) {
      String conditions = next.getConditions();
      if(StringUtils.isBlank(conditions)) {
        continue;
      }
      // 执行
      Object evalResult = this.executeGroovyScript(conditions, context.getParams());
      if(evalResult == null || Boolean.class != evalResult.getClass()) {
        throw new IllegalArgumentException("条件判定分支的返回值，只能是boolean型，请检查!!");
      }
      // 如果条件成立，说明符合条件，就是这个运行分支
      if((Boolean)evalResult) {
        currentNext = next;
        break;
      }
    }
    // 2.3、===
    if(currentNext == null) {
      currentNext = sortNexts.stream().filter(item -> StringUtils.isBlank(item.getConditions())).findFirst().orElse(null);
    }
    Validate.notNull(currentNext , "未发现任何符合条件的运行分支，也没有发现未设置运行条件的默认分支，请检查!!");
    String toNodeCode = currentNext.getToNodeCode();
    RuntimeNode targetNode = nodes.stream().filter(item -> item.getCode() == toNodeCode).findFirst().orElse(null);
    Validate.notNull(targetNode , "没有发现指定的节点信息[%s]，请检查!!" , toNodeCode);
    return targetNode;
  }

  /**
   * 在全动态后端groovy脚本的场景下，运行实际的动态脚本代码
   * @param scriptContxt
   * @param contentMap
   * @param binding 可能来源于调用者的脚本变量绑定信息
   * @return 可能有返回值，也可能没有任何返回值
   */
  private Object executeGroovyScript(String scriptContxt, Map<String , Object> contentMap) {
    // 进行上下文已有参数绑定后执行
    Map<String , Object> scriptInputsMap = new HashMap<>(contentMap);
    scriptInputsMap.put("runtimeNodeService", runtimeNodeService);
    scriptInputsMap.put("runtimeDefinitionService", runtimeDefinitionService);
    return this.scriptService.invoke(scriptContxt, scriptInputsMap);
  }
  
  // 如果验证后，如果过程中发现不能继续，则返回false
  private boolean validate(RuntimeNode currentNode , RuleRuntimeContext context) {
    if(currentNode == null || context == null) {
      return false;
    }
    String fromId = currentNode.getId();
    RuntimeNodeType type = currentNode.getType();
    if(StringUtils.isBlank(fromId)) {
      return false;
    }
    // 如果当前节点是结束节点，则不用进行查找后续结点
    if(type == RuntimeNodeType.END) {
      return false;
    }
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    if(CollectionUtils.isEmpty(nexts)) {
      throw new IllegalArgumentException("未发现指定节点的后续节点定义，请检查规则模板设定!!");
    }
    return true;
  }
  
  @Override
  public RuntimeNode findExceptionNextByContext(RuntimeNode currentNode, RuleRuntimeContext context) {
    if(!this.validate(currentNode, context)) {
      return null;
    } 
    Set<RuntimeNodeNexts> nexts = currentNode.getNexts();
    Set<RuntimeNode> nodes = context.getRuntimeDefinition().getNodes();
    Throwable currentThrowable = context.getCurrentThrowable();
    if(currentThrowable == null) {
      return null;
    }
    if(currentThrowable instanceof InvocationTargetException) {
      currentThrowable = ((InvocationTargetException)currentThrowable).getTargetException();
    }
    
    /*
     * 处理方式为：
     * 1、首先nexts信息需要排序，然后再根据判定表达式，依次执行判定
     * 注意，只需要对异常线进行排序
     * 2、判定当前上下文记录的异常和分支允许的异常是否一致，只有一致才进行处理
     * 3、如果以上过程没有发现任何符合要求的分支节点（包括没有默认分支），则返回null
     * (和findNextByDefinition类似，请参考)
     * */
    
    // 1、=======
    Set<RuntimeNodeNexts> sortNexts = nexts.stream().filter(item -> item.getLineType() == 2).sorted((source , target) -> source.getSort() - target.getSort()).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(sortNexts)) {
      return null;
    } 
    
    // 2、=======
    ClassLoader classLoader = applicationContext.getClassLoader();
    RuntimeNodeNexts currentNext = null;
    for(RuntimeNodeNexts next : sortNexts) {
      String exceptions = next.getExceptions();
      if(StringUtils.isBlank(exceptions)) {
        continue;
      }
      String exceptionValues[] = StringUtils.split(exceptions, ",");
      for (String exceptionValue : exceptionValues) {
        Class<?> exceptionClass = null;
        try {
          exceptionClass = classLoader.loadClass(exceptionValue);
        } catch(Exception e) {
          LOGGER.error(e.getMessage() , e);
          throw new IllegalArgumentException(String.format("未发现指定的异常定义类[%s]，请检查!!", exceptionValue), e); 
        }
        
        // 如果条件成立，说明找到了匹配的分支
        if(exceptionClass.isAssignableFrom(currentThrowable.getClass())) {
          currentNext = next;
        }
      }
    }
    
    // 3、========
    // 如果没有找到任何匹配的异常分支线，则寻找没有填写任何异常条件的默认分支
    if(currentNext == null) {
      currentNext = sortNexts.stream().filter(item -> StringUtils.isBlank(item.getConditions())).findFirst().orElse(null);
    }
    if(currentNext == null) {
      return null;
    }
    String toNodeCode = currentNext.getToNodeCode();
    RuntimeNode targetNode = nodes.stream().filter(item -> item.getCode() == toNodeCode).findFirst().orElse(null);
    Validate.notNull(targetNode , "没有发现指定的节点[%s]信息，请检查!!" , toNodeCode);
    return targetNode;
  }
  
//  @Override
//  public Object process(String templateNodeCode , String returnParamName) {
//    return this.process(templateNodeCode, null , returnParamName);
//  }
//
//  @Override
//  public Object process(String templateNodeCode, Map<String, Object> params, String returnParamName) {
//    Validate.notBlank(templateNodeCode , "错误的模板节点编号信息!!");
//    RuleTemplateNodeEntity ruleTemplateNode = this.ruleTemplateNodeService.findDetailsByCode(templateNodeCode);
//    Validate.notNull(ruleTemplateNode , "未发现指定code[%s]的节点模板信息，请检查!!" , templateNodeCode);
//    
//    /*
//     * 类似于这种基于节点模板code进行执行的动态节点定义
//     * 其效果就是按照当前source的信息，直接进行执行
//     * 它并不会参与到整个规则定义链中。
//     * 
//     * 但是动态节点的调用，当前线程必须存在规则上下文
//     * 最后，再根据returnParamName的设定，从上下文中提取可能的返回值
//     * */
//    
//    RuntimeNode currentNode = new RuntimeNode();
//    int sourceType = ruleTemplateNode.getSourceType();
//    int type = ruleTemplateNode.getType();
//    Validate.isTrue(type == 2 , "在进行基于节点模板的节点动态执行时，只支持逻辑组件的使用。否则请直接使用ApplicationProcessChain进行规则定义的调用!!");
//    currentNode.setSourceType(sourceType);
//    currentNode.setTemplateNodeId(ruleTemplateNode.getId());
//    currentNode.setType(ruleTemplateNode.getType());
//    // 业务编号随机生成
//    currentNode.setCode(UUID.randomUUID().toString());
//    // 可能的入参
//    Set<RuleTemplateParamsEntity> templateInputsParams = ruleTemplateNode.getInputs();
//    Set<RuntimeNodeParams> inputs = new HashSet<>();
//    if(!CollectionUtils.isEmpty(templateInputsParams)) {
//      for (RuleTemplateParamsEntity templateInputsParam : templateInputsParams) {
//        inputs.add(this.bindingRuntimeNodeParams(templateInputsParam));
//      }
//    }
//    currentNode.setInputs(inputs);
//    // 可能的出参
//    Set<RuleTemplateParamsEntity> templateOutputsParams = ruleTemplateNode.getOutputs();
//    Set<RuntimeNodeParams> outputs = new HashSet<>();
//    if(!CollectionUtils.isEmpty(templateOutputsParams)) {
//      for (RuleTemplateParamsEntity templateOutputsParam : templateOutputsParams) {
//        outputs.add(this.bindingRuntimeNodeParams(templateOutputsParam));
//      }
//    }
//    currentNode.setOutputs(outputs);
//    // 可能的处理器信息 1:服务源；2：数据视图（聚集）；3：后台脚本（groovy）
//    switch (sourceType) {
//      case 1:
//        currentNode.setSourceServicable(super.parseByRuntimeSourceServicable(ruleTemplateNode, templateNodeCode));
//        break;
//      case 2:
//        currentNode.setSourceAggregateDataView(super.parseByRuntimeSourceAggregateDataView(ruleTemplateNode, templateNodeCode));
//        break;
//      case 3:
//        currentNode.setSourceScript(super.parseByRuntimeSourceScript(ruleTemplateNode, templateNodeCode));
//        break;
//      default:
//        
//    }
//    // 开始执行
//    this.process(currentNode, params);
//    
//    // 提取可能的返回值信息
//    if(StringUtils.isBlank(returnParamName)) {
//      return null;
//    }
//    // 取得上下文信息
//    RuleRuntimeContext context = ProcessorLinkedChain.getRuleRuntimeContext();
//    return context.getParams().get(returnParamName);
//  }
  
//  /**
//   * 根据数组层的参数情况，生成运行时参数（可能是入参也可能是出参）
//   * @param templateParam
//   * @return
//   */
//  private RuntimeNodeParams bindingRuntimeNodeParams(RuleTemplateParamsEntity templateParam) {
//    boolean nullable = templateParam.getNullable();
//    String paramDesc = templateParam.getParamDesc();
//    String paramName = templateParam.getParamName();
//    String paramType = templateParam.getParamType();
//    Integer type = templateParam.getType();
//    
//    RuntimeNodeParams param = new RuntimeNodeParams();
//    param.setContextParamName(paramName);
//    param.setNullable(nullable);
//    param.setParamDesc(paramDesc);
//    param.setParamType(paramType);
//    param.setTemplateParamName(paramName);
//    param.setType(type);
//    return param;
//  }
}
