package com.bizunited.platform.mars.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseCode;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.mars.service.cache.RuntimeDefinition;
import com.bizunited.platform.mars.service.process.RuntimeDefinitionService;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import java.io.IOException;
import java.lang.String;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

/**
 * 规则运行时信息相关的http restful接口定义
 * @author yinwenjei
 */
@RestController
@RequestMapping("/v1/runtimeDefinitions")
public class RuntimeDefinitionController extends BaseController { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RuntimeDefinitionController.class);
  
  @Autowired
  private RuntimeDefinitionService runtimeDefinitionService;
  
  @ApiOperation(value = "针对一个新的规则定义，进行创建操作。注意：创建操作时，规则定义的基本信息和附属信息，都不能包含ID信息，具体的信息结构可参见文档示例")
  @PostMapping(value="")
  public ResponseModel create(@RequestParam("runtimeDefinition") @ApiParam(name="runtimeDefinition" , value = "符合前后端规则定义数据结构描述的json信息") String runtimeDefinitionJson ,
                              @RequestParam("file") @ApiParam(name="file" , value = "前端布局文件排版信息") MultipartFile file) {
    try {
      byte[] fileContent = file.getBytes();
      RuntimeDefinition definition = JSON.toJavaObject(JSON.parseObject(runtimeDefinitionJson), RuntimeDefinition.class);
      definition = this.runtimeDefinitionService.create(definition, fileContent);
      return this.buildHttpResultW(definition, new String[]{});
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      ResponseModel error = new ResponseModel();
      error.setErrorMsg("错误的文件上传内容，请检查!!");
      error.setMessage(e.getMessage());
      error.setResponseCode(ResponseCode.E503);
      error.setSuccess(false);
      return error;
    }
  }
  
  @ApiOperation(value = "针对一个已有的规则定义，进行该规则定义的全量修改。所谓全量修改，就是之前的规则定义信息会全部作废")
  @PatchMapping(value="")
  public ResponseModel update(@RequestParam("runtimeDefinition") @ApiParam(name="runtimeDefinition" , value = "符合前后端规则定义数据结构描述的json信息") String runtimeDefinitionJson ,
                              @RequestParam("file") @ApiParam(name="file" , value = "前端布局文件排版信息") MultipartFile file) {
    try {
      byte[] fileContent = file.getBytes();
      RuntimeDefinition definition = JSON.toJavaObject(JSON.parseObject(runtimeDefinitionJson), RuntimeDefinition.class);
      definition = this.runtimeDefinitionService.update(definition, fileContent);
      return this.buildHttpResultW(definition, new String[]{});
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      ResponseModel error = new ResponseModel();
      error.setErrorMsg("错误的文件上传内容，请检查!!");
      error.setMessage(e.getMessage());
      error.setResponseCode(ResponseCode.E503);
      error.setSuccess(false);
      return error;
    }
  }
  
  @ApiOperation(value = "通过给定的规则定义编号和版本信息，查询完整的规则描述。规则描述信息将基于持久层存储的规则定义信息生成，并被缓存到内存中")
  @RequestMapping(value="/findDetailsByCodeAndVersion" , method={RequestMethod.GET})
  public ResponseModel findDetailsByCodeAndVersion(@RequestParam("code") @ApiParam(name="code" , value = "前端布局文件排版信息") String code,
                                                   @RequestParam("version") @ApiParam(name="version" , value = "前端布局文件排版信息") String version) {
    try { 
      RuntimeDefinition result = this.runtimeDefinitionService.findDetailsByCodeAndVersion(code, version);
      return this.buildHttpResultW(result, "nodes" , "nodes.inputs" , "nodes.outputs" , "nodes.nexts"); 
    } catch(RuntimeException e) { 
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    } 
  } 
  
  /**
   * 对一个指定规则定义发起请求调用，一次调用将使用一个独立的ProcessChain对象进行调用过程管理
   * @param code 指定的规则定义的业务编号
   * @param version 指定的规则定义的版本信息
   * @param params 如果有外部的参数，则从这里进行传入
   * @return 如果当前规则定义指定了存在调用结果，在调用成功的前提下，调用结果将被返回
   */
  @ApiOperation(value = "对一个指定规则定义发起请求调用，一次调用将使用一个独立的ProcessChain对象进行调用过程管理")
  @RequestMapping(value="/process" , method={RequestMethod.POST})
  public ResponseModel process(@RequestParam("code") String code ,
                               @RequestParam("version") String version , HttpServletRequest request , HttpServletResponse response) {
    try {
      // request中的param值全部通过K-V形式进行读取
      Map<String, Object> inputs = new HashMap<>();
      Map<String, String[]> parameterMap = request.getParameterMap();
      if(parameterMap != null) {
        Set<Entry<String, String[]>> entrySet = parameterMap.entrySet();
        for (Entry<String, String[]> entry : entrySet) {
          String key = entry.getKey();
          String[] values = entry.getValue();
          if(values == null) {
            continue;
          }
          if(values.length == 1) {
            inputs.put(key, values[0]);
          } else {
            inputs.put(key, values);
          }
        }
      }
      // 开始调用
      Object result = this.runtimeDefinitionService.process(code, version, inputs);
      if(result instanceof Collection<?>) {
        Collection<?> results = (Collection<?>)result;
        return this.buildHttpResultW(results);
      } else {
        return this.buildHttpResultW(result);
      }
    } catch(RuntimeException e) { 
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
} 
