package com.bizunited.platform.core.common.constant;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 目前Kuiper规定的能够扫描的基本类型
 *用于方法参数的一些常量信息
 *部分java基本类型对应的映射关系，以及javassist的类与jvm的映射关系
 *目前支持到二维
 */
public class ParamClassTypeConst {

  private ParamClassTypeConst(){}

  /**
   * 记录java八大基本类型对应的类信息，以及它们的数组和二维数组在jvm中类信息
   */
  private static final Map<String,Class<?>> BASE_CLASS_MAP = new HashMap<>();
  /**
   * 记录java八大基本类型与对应的自动装箱后的类型映射关系，以及它们的数组和二维数组对应的自动装箱后的类型映射关系
   */
  private static final Map<String,String> BASE_CLASS_MAPPING = new HashMap<>();
  /**
   * 记录java八大基本类型与其数组形式，在javassist与jvm之间的映射关系
   */
  private static final Map<String,String> JAVASSIST_TO_BASE_MAPPING = new HashMap<>();
  /**
   * 记录java八大基本类型的装箱对象的数组形式，在javassist与jvm之间的映射关系
   */
  private static final Map<String,String> JAVASSIST_TO_BASE_REF_MAPPING = new HashMap<>();
  
  static {
    fillBaseClassMap();
    fillBaseClassMapping();
    fillJavassistToBaseMapping();
    fillJavassistToBaseRefMapping();
  }

  public static Boolean baseClassMapContainsKey(String key){
    return BASE_CLASS_MAP.containsKey(key);
  }

  public static Class<?> getBaseClass(String key){
    return BASE_CLASS_MAP.get(key);
  }

  public static String getJavassistBaseClass(String key){
    return JAVASSIST_TO_BASE_MAPPING.get(key);
  }
  /**
   * 目前Kuiper规定的能够扫描的基本类型
   * 用于分析参数的类型是否符合我们规定的类型 ， 只匹配类名(除Date外，都是final，没有继承关系)
   */
  private static final List<String> BASE_PARAM_CLASS_TYPE_LIST = Arrays.asList(new String[]{
    "java.lang.String","java.lang.Boolean",
    "java.lang.Integer","java.lang.Long","java.lang.Double","java.lang.Byte","java.lang.Short","java.lang.Float",
    "int","long","double","byte","short","boolean","char","float",
    "java.lang.Character","java.util.Date"});

  public static final List<String> getBaseParamClassTypeList(){
    return BASE_PARAM_CLASS_TYPE_LIST;
  }
  /**
   * 目前Kuiper规定的能够扫描的基本类型
   * 用于分析参数的类型是符合我们规定的类型，匹配特殊的指定的类型（含有继承关系或实现）
   */
  private static final List<String> BASE_PARAM_SPECIAL_TYPE_LIST = Arrays.asList("java.util.Date","java.lang.Number","java.text.Format","java.lang.CharSequence");

  public static final List<String> getBaseParamSpecialTypeList(){
    return BASE_PARAM_SPECIAL_TYPE_LIST;
  }

  private static void fillBaseClassMap(){
    BASE_CLASS_MAP.put("int", int.class);
    BASE_CLASS_MAP.put("long", long.class);
    BASE_CLASS_MAP.put("double", double.class);
    BASE_CLASS_MAP.put("byte", byte.class);
    BASE_CLASS_MAP.put("short", short.class);
    BASE_CLASS_MAP.put("boolean", boolean.class);
    BASE_CLASS_MAP.put("char", char.class);
    BASE_CLASS_MAP.put("float", float.class);

    BASE_CLASS_MAP.put("[I", int[].class);
    BASE_CLASS_MAP.put("[J", long[].class);
    BASE_CLASS_MAP.put("[D", double[].class);
    BASE_CLASS_MAP.put("[B", byte[].class);
    BASE_CLASS_MAP.put("[S", short[].class);
    BASE_CLASS_MAP.put("[Z", boolean[].class);
    BASE_CLASS_MAP.put("[C", char[].class);
    BASE_CLASS_MAP.put("[F", float[].class);

    BASE_CLASS_MAP.put("[[I", int[][].class);
    BASE_CLASS_MAP.put("[[J", long[][].class);
    BASE_CLASS_MAP.put("[[D", double[][].class);
    BASE_CLASS_MAP.put("[[B", byte[][].class);
    BASE_CLASS_MAP.put("[[S", short[][].class);
    BASE_CLASS_MAP.put("[[Z", boolean[][].class);
    BASE_CLASS_MAP.put("[[C", char[][].class);
    BASE_CLASS_MAP.put("[[F", float[][].class);
  }

  private static void fillBaseClassMapping(){
    BASE_CLASS_MAPPING.put("int", "java.lang.Integer");
    BASE_CLASS_MAPPING.put("long", "java.lang.Long");
    BASE_CLASS_MAPPING.put("double", "java.lang.Double");
    BASE_CLASS_MAPPING.put("byte", "java.lang.Byte");
    BASE_CLASS_MAPPING.put("short", "java.lang.Short");
    BASE_CLASS_MAPPING.put("boolean", "java.lang.Boolean");
    BASE_CLASS_MAPPING.put("char", "java.lang.Character");
    BASE_CLASS_MAPPING.put("float", "java.lang.Float");

    BASE_CLASS_MAPPING.put("[I", "[Ljava.lang.Integer;");
    BASE_CLASS_MAPPING.put("[J", "[Ljava.lang.Long;");
    BASE_CLASS_MAPPING.put("[D", "[Ljava.lang.Double;");
    BASE_CLASS_MAPPING.put("[B", "[Ljava.lang.Byte;");
    BASE_CLASS_MAPPING.put("[S", "[Ljava.lang.Short;");
    BASE_CLASS_MAPPING.put("[Z", "[Ljava.lang.Boolean;");
    BASE_CLASS_MAPPING.put("[C", "[Ljava.lang.Character;");
    BASE_CLASS_MAPPING.put("[F", "[Ljava.lang.Float;");

    BASE_CLASS_MAPPING.put("[[I", "[[Ljava.lang.Integer;");
    BASE_CLASS_MAPPING.put("[[J", "[[Ljava.lang.Long;");
    BASE_CLASS_MAPPING.put("[[D", "[[Ljava.lang.Double;");
    BASE_CLASS_MAPPING.put("[[B", "[[Ljava.lang.Byte;");
    BASE_CLASS_MAPPING.put("[[S", "[[Ljava.lang.Short;");
    BASE_CLASS_MAPPING.put("[[Z", "[[Ljava.lang.Boolean;");
    BASE_CLASS_MAPPING.put("[[C", "[[Ljava.lang.Character;");
    BASE_CLASS_MAPPING.put("[[F", "[[Ljava.lang.Float;");
  }

  private static void fillJavassistToBaseMapping(){
    JAVASSIST_TO_BASE_MAPPING.put("int", "int");
    JAVASSIST_TO_BASE_MAPPING.put("long", "long");
    JAVASSIST_TO_BASE_MAPPING.put("double", "double");
    JAVASSIST_TO_BASE_MAPPING.put("byte", "byte");
    JAVASSIST_TO_BASE_MAPPING.put("short", "short");
    JAVASSIST_TO_BASE_MAPPING.put("boolean", "boolean");
    JAVASSIST_TO_BASE_MAPPING.put("char", "char");
    JAVASSIST_TO_BASE_MAPPING.put("float", "float");

    JAVASSIST_TO_BASE_MAPPING.put("int[]", "[I");
    JAVASSIST_TO_BASE_MAPPING.put("long[]", "[J");
    JAVASSIST_TO_BASE_MAPPING.put("double[]", "[D");
    JAVASSIST_TO_BASE_MAPPING.put("byte[]", "[B");
    JAVASSIST_TO_BASE_MAPPING.put("short[]", "[S");
    JAVASSIST_TO_BASE_MAPPING.put("boolean[]", "[Z");
    JAVASSIST_TO_BASE_MAPPING.put("char[]", "[C");
    JAVASSIST_TO_BASE_MAPPING.put("float[]", "[F");

    JAVASSIST_TO_BASE_MAPPING.put("int[][]", "[[I");
    JAVASSIST_TO_BASE_MAPPING.put("long[][]", "[[J");
    JAVASSIST_TO_BASE_MAPPING.put("double[][]", "[[D");
    JAVASSIST_TO_BASE_MAPPING.put("byte[][]", "[[B");
    JAVASSIST_TO_BASE_MAPPING.put("short[][]", "[[S");
    JAVASSIST_TO_BASE_MAPPING.put("boolean[][]", "[[Z");
    JAVASSIST_TO_BASE_MAPPING.put("char[][]", "[[C");
    JAVASSIST_TO_BASE_MAPPING.put("float[][]", "[[F");
  }

  private static void fillJavassistToBaseRefMapping(){
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Integer[]", "[Ljava.lang.Integer;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Long[]", "[Ljava.lang.Long;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Double[]", "[Ljava.lang.Double;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Byte[]", "[Ljava.lang.Byte;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Short[]", "[Ljava.lang.Short;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Boolean[]", "[Ljava.lang.Boolean;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Character[]", "[Ljava.lang.Character;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Float[]", "[Ljava.lang.Float;");

    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Integer[][]", "[[Ljava.lang.Integer;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Long[][]", "[[Ljava.lang.Long;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Double[][]", "[[Ljava.lang.Double;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Byte[][]", "[[Ljava.lang.Byte;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Short[][]", "[[Ljava.lang.Short;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Boolean[][]", "[[Ljava.lang.Boolean;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Character[][]", "[[Ljava.lang.Character;");
    JAVASSIST_TO_BASE_REF_MAPPING.put("java.lang.Float[][]", "[[Ljava.lang.Float;");
  }
}
