package com.bizunited.platform.core.configuration;

import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.invoke.DefaultHandleChain;
import com.bizunited.platform.core.service.invoke.InvokeProxy;
import com.bizunited.platform.core.service.invoke.handle.request.TimerRecordHandle;
import com.bizunited.platform.core.service.scheduler.DynamicTaskNotifyMaster;
import com.bizunited.platform.core.service.scheduler.handle.DynamicTaskInvokeHandle;
import com.bizunited.platform.core.service.scheduler.handle.DynamicTaskInvokeLogHandle;

/**
 * 动态任务管理器的配置，动态任务管理器所管理的任务全部通过groovy进行驱动，并且可以实现动态启停、错误日志记录等操作
 * @author yinwenjie
 */
@Configuration
@EnableAsync
@Component("_schedulerConfig")
public class SchedulerConfig implements CommandLineRunner  {
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DynamicTaskNotifyMaster dynamicTaskNotifyMaster;
  @Autowired
  private TimerRecordHandle timerRecordHandle;
  @Autowired
  private DynamicTaskInvokeHandle dynamicTaskInvokeHandle;
  @Autowired
  private DynamicTaskInvokeLogHandle dynamicTaskInvokeLogHandle;
  
  @Bean("platform_dynamicTaskScheduler")
  public ThreadPoolTaskScheduler getThreadPoolTaskScheduler() {
    ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
    // 配置参数暂时不开放给开发人员
    scheduler.setBeanName("platform_dynamicTaskScheduler");
    scheduler.setDaemon(false);
    scheduler.setPoolSize(50);
    scheduler.setThreadFactory(getThreadFactory());
    scheduler.setThreadNamePrefix("platform_");
    return scheduler;
  }
  
  /**
   * 自定义ThreadFactory，主要是为了指定classLoader和一个可以很好做区别的threadName
   * @return
   */
  private ThreadFactory getThreadFactory() {
    ThreadFactory dynamicTaskSchedulerFactory = new ThreadFactory() {
      private AtomicInteger count = new AtomicInteger();
      
      @Override
      public Thread newThread(Runnable r) {
        Thread currentThread = new Thread(r);
        ClassLoader classLoader = applicationContext.getClassLoader();
        currentThread.setDaemon(false);
        currentThread.setName("dynamicTask_" + count.getAndIncrement());
        currentThread.setContextClassLoader(classLoader);
        return currentThread;
      }
    };
    return dynamicTaskSchedulerFactory;
  }
  
  @Bean("schedulerTaskProxy")
  @Scope(value = "prototype")
  public InvokeProxy getSchedulerTaskProxyInvoke() {
    InvokeProxy.Build build = new InvokeProxy.Build();
    InvokeProxy invokeProxy = build
        .addInvokeRequestFilter(this.timerRecordHandle , this.dynamicTaskInvokeHandle)
        .addInvokeResponseFilter(this.dynamicTaskInvokeLogHandle)
        .setTargetHandleChain(DefaultHandleChain.class)
        .addClassLoader(this.applicationContext.getClassLoader())
        .build();
    return invokeProxy;
  }

  @Override
  public void run(String... args) throws Exception {
    // 当启动完成后，开始对动态调度任务的执行权的抢占
    dynamicTaskNotifyMaster.notifyMaster();
  }
}