package com.bizunited.platform.core.configuration;

import java.util.List;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.init.InitProcessService;
import com.bizunited.platform.rbac.server.service.redis.RedisMutexService;

@Component("SystemInitConfig")
public class SystemInitConfig implements CommandLineRunner {
  private static final Logger LOGGER = LoggerFactory.getLogger(SystemInitConfig.class);
  /**
   * 所有在本系统中实现了InitProcessService接口的初始化动作
   * 都将在这里被识别
   */
  @Autowired(required = false)
  private List<InitProcessService> initProcessServices;
  @Autowired
  private RedisMutexService redisMutexService;
  
  @Override
  @Transactional
  public void run(String... args) throws Exception {
    if(initProcessServices == null || initProcessServices.isEmpty()) {
      return;
    }
    /*
     * 整个系统的初始化过程包括以下步骤：
     * 1、在完成验证后，首先initProcessServices集合中的所有服务对象将按照InitProcessService接口的sort()方法进行重新排序，
     * sort()方法返回值越小的service服务将被排列在越前面（但是，不允许返回负数）
     * 2、接着，为了避免多个业务节点在同时启动时，同时运行初始化动作造成的重复初始化过程，在正式进行初始化操作前，要加分布式锁
     * 分布式锁将在系统主数据库事务提交后被解除
     * 3、接着按照步骤1处理排序集合中的每一个InitProcessService接口的实现类；注意doProcess()方法和stopOnException()方法的返回值定义
     * */
    // 1、=======
    List<InitProcessService> queue = this.initProcessServices.stream().sorted((t , s) -> t.sort() - s.sort()).collect(Collectors.toList());
    
    // 2、=======
    try {
      redisMutexService.lock("_starting");
      this.init(queue);
      LOGGER.info("....system init process dome!");
    } catch(RuntimeException e) {
      LOGGER.info("....system init process error!");
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    } finally {
      redisMutexService.unlock("_starting");
    }
  }
  
  /**
   * 正式的初始化动作在这里 
   * @param queue 
   */
  private void init(List<InitProcessService> queue) {
    // 3、=======
    for (InitProcessService initProcessService : queue) {
      try {
        boolean doProcess = initProcessService.doProcess();
        // 如果条件成立，那么说明需要在本次进程启动时，进行这个初始化动作
        if(doProcess) {
          initProcessService.init();
        }
      } catch(RuntimeException e) {
        boolean stopOnException = initProcessService.stopOnException();
        // 如果条件成立，说明一旦当前初始化动作出现异常，也不需要终止整个启动过程
        if(!stopOnException) {
          LOGGER.warn(e.getMessage() , e);
          continue;
        }
        // 否则代码将走到这里，说明一旦当前初始化动作出现异常，则需要终止整个启动过程
        LOGGER.error(e.getMessage() , e);
        throw e;
      }
    }
  }
}
