package com.bizunited.platform.core.controller;

import com.bizunited.platform.core.controller.model.ResponseCode;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.text.Format;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;

/**
 * 基础controller
 * @author yinwenjie
 */
@ApiIgnore
public class BaseController {  
  /**
   * 日志
   */
  private static final Logger LOG = LoggerFactory.getLogger(BaseController.class);

  /**
   * 获取登陆者账号
   * @return
   */
  protected String getPrincipalAccount() {
    Principal userPrincipal = this.getPrincipal();
    // 通过operator确定当前的登录者信息
    String account = userPrincipal.getName();
    Validate.notBlank(account, "not found op user!");
    return account;
  }
  /**
   * 获取登陆者账号对象
   * @return
   */
  protected Principal getPrincipal() {
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    Validate.notNull(authentication , "未获取到当前系统的登录人");
    return authentication;
  }

  /**
   * 当异常状况时，使用该方法构造返回值
   * @param e 错误的异常对象描述
   * @return 组装好的异常结果
   */
  protected ResponseModel buildHttpReslutForException(Exception e) {
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E501, e);
    return result;
  }
  
  /**
   * 该方法不返回任何信息，只是告诉调用者，调用业务成功了。
   * @return
   */
  protected ResponseModel buildHttpReslut() {
    return new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
  }
  
  /**
   * 该方法不返回任何信息，只是告诉调用者，调用业务成功了。
   * @return
   */
  protected ResponseModel buildHttpReslut(Object data) {
    return new ResponseModel(new Date().getTime(), data, ResponseCode.E0, null);
  }

  /**
   * 基于白名单关联属性过滤的查询结果对象构建方法<br>
   * 例如返回user下面的基本信息，包括user下的所有roles关联信息，那么白名单中需要包括roles属性即可，其它的关联属性都将被剔除。
   * @see #buildHttpReslutW(entity, String...) 
   * @param properties 
   * @return 
   */
  protected <T> ResponseModel buildHttpReslutW(Iterable<T> page, String... properties) {
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
    if (page == null) {
      result.setData(page);
      return result;
    }
    try {
      for (T item : page) {
        filterPropertyW(item, item.getClass(), new LinkedList<String>(), properties);
      }
      result.setData(page);
    } catch(Exception e) {
      result.setErrorMsg(e.getMessage());
      result.setMessage(e.getMessage());
      result.setResponseCode(ResponseCode.E501);
      result.setSuccess(false);
    }
    
    return result;
  }

  /**
   * 基于白名单关联属性过滤的查询结果对象构建方法<br>
   * 例如返回user下面的基本信息，包括user下的所有roles关联信息，那么白名单中需要包括roles属性即可，其它的关联属性都将被剔除。
   * @see #buildHttpReslutW(entity, String...)
   * @return
   */
  protected <T> ResponseModel buildHttpReslutW(Collection<T> entities, String... properties) {
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
    if (entities == null || entities.isEmpty()) {
      result.setData(entities);
      return result;
    }
    try {
      for (T item : entities) {
        filterPropertyW(item, item.getClass(), new LinkedList<String>(), properties);
      }
      result.setData(entities);
    } catch(Exception e) {
      result.setErrorMsg(e.getMessage());
      result.setMessage(e.getMessage());
      result.setResponseCode(ResponseCode.E501);
      result.setSuccess(false);
    }
    return result;
  }
  
  /**
   * 构建属性信息，通过白名单的方式
   * @param entity
   * @see #buildHttpReslutW(entity, String...)
   * @param properties 这是白名单属性列表（属性列表支持级联），只有在这个属性列表中的关联属性才会进行输出
   */
  protected <T> ResponseModel buildHttpReslutW(T entity, String... properties) { 
    ResponseModel result = new ResponseModel(new Date().getTime(), null, ResponseCode.E0, null);
    if (entity == null) {
      return result;
    }
    
    // 过滤
    Class<?> objectClass = entity.getClass();
    try {
      this.filterPropertyW(entity , objectClass , new LinkedList<String>() , properties);
    } catch(Exception e) {
      result.setErrorMsg(e.getMessage());
      result.setMessage(e.getMessage());
      result.setResponseCode(ResponseCode.E501);
      result.setSuccess(false);
    }
    result.setData(entity);
    return result;
  }
  
  /**
   * 过滤对象中的属性，只有存在于properties中的非基本类型属性才会进行下一层过滤处理
   * （所谓基本类型，就是java中的八大基本类型以及这些八大基础类型的类表示）
   * @param property 属性对象
   * @param targetClass 当前的class，可能是父级class也可能就是本级class
   * @param parentProperties 当前递归属性的上一级属性
   * @param properties 在白名单上的属性
   * @throws NoSuchMethodException
   * @throws SecurityException
   * @throws IllegalAccessException
   * @throws IllegalArgumentException
   * @throws InvocationTargetException
   */
  private void filterPropertyW(Object currentObject , Class<?> currentClass , LinkedList<String> parentProperties , String... properties) 
      throws SecurityException, IllegalAccessException, InvocationTargetException {
    if(parentProperties == null) {
      parentProperties = Lists.newLinkedList();
    }
    /*
     * - 处理过程如下：
     * 1、取得当前对象的所有属性（注意还有父级属性）
     * 2、如果当前属性属于基本类型的属性支持，则可以忽略不处理
     * 3、如果当前属性是一个集合类型（单泛型集合），且当前属性在properties白名单中，且当前属性有值，则向下递归每一个集合中的元素。其它情况全部设定为null，
     * 4、如果当前属性的类不是基本类型，也不是集合，且当前属性在properties白名单中，那么就向下一级进行递归处理，其它情况全部设定为null
     * 5、如果当前属性不属于以上所有情况，则无论是否在properties白名单中，都设置为null
     * */
    // 1、=======
    if(currentObject == null) {
      return;
    }
    Field[] fields = currentClass.getDeclaredFields();
    if(fields == null || fields.length == 0) {
      return;
    }
    // 处理父级
    Class<?> superClass = null;
    if((superClass = currentClass.getSuperclass()) != null) {
      this.filterPropertyW(currentObject, superClass, parentProperties, properties);
    }
    
    for (Field fieldItem : fields) {
      Class<?> fieldClass;
      String currentFieldName = fieldItem.getName();
      Method getMethod = null;
      Method setMethod = null;
      fieldClass = fieldItem.getType();
      // 寻找当前字段的set、get方法
      try {
        char[] chars = currentFieldName.toCharArray();
        chars[0] -= 32;
        getMethod = currentClass.getMethod("get" + String.valueOf(chars));
        chars = currentFieldName.toCharArray();
        chars[0] -= 32;
        setMethod = currentClass.getMethod("set" + String.valueOf(chars) , fieldClass);
      } catch(RuntimeException | NoSuchMethodException e) {
        LOG.debug(e.getMessage());
      } 
      // 如果没有设定set、get方法，则忽略本次处理
      if(getMethod == null || setMethod == null) {
        continue;
      }
      String currentFieldNames = null;
      if(parentProperties != null && !parentProperties.isEmpty()) {
        String[] fieldNames = parentProperties.toArray(new String[]{});
        currentFieldNames =  StringUtils.join(StringUtils.join(fieldNames, ".") , "." , currentFieldName);
      } else {
        currentFieldNames = currentFieldName;
      }
      
      // 2、========
      if(this.baseForPropertyW(fieldItem)) {
        continue;
      }
      // 3、=======
      else if(Collection.class.isAssignableFrom(fieldClass)) {
        // 比较当前属性是否在白名单中，如果不存在于properties集合中，则设置为null
        if(properties == null || properties.length == 0 || !Arrays.asList(properties).contains(currentFieldNames)) {
          setMethod.invoke(currentObject , new Object[] {null});
          continue;
        } 
        Object fieldValue = getMethod.invoke(currentObject);
        if(fieldValue == null) {
          continue;
        }
        
        // 运行到这里，就要取出集合中的每一个值，然后进行向下的递归遍历了
        parentProperties.addLast(currentFieldName);
        Collection<?> collections = (Collection<?>) fieldValue;
        for (Object propertyObject : collections) {
          Class<?> propertyClass = propertyObject.getClass();
          this.filterPropertyW(propertyObject, propertyClass, parentProperties, properties);
        }
        parentProperties.removeLast();
      }
      // 4、=======
      else if(!this.baseForPropertyW(fieldItem) && Arrays.asList(properties).contains(currentFieldNames)) {
        if(properties == null || properties.length == 0) {
          setMethod.invoke(currentObject , new Object[] {null});
          continue;
        } 
        Object currentFieldValue = getMethod.invoke(currentObject);
        if(currentFieldValue == null) {
          continue;
        }
        
        parentProperties.addLast(currentFieldName);
        this.filterPropertyW(currentFieldValue, currentFieldValue.getClass(), parentProperties, properties);
        parentProperties.removeLast(); 
      } 
      // 其它情况全部设置为null
      else {
        setMethod.invoke(currentObject , new Object[] {null});
      }
    }
  }
  
  /**
   * 向response中写入giftupian
   * @param response HttpServletResponse
   * @param httpType http信息的格式。根据不同的文件类型，http信息格式不一样<br>
   *        1：表示是一张图片 2：表示是一段语音 3：表示是可播放的mpg4视频
   * @param result 写入的byte信息
   */
  protected void writeResponseFile(HttpServletResponse response, int httpType, byte[] result) {
    if (httpType == 1) {
      response.setContentType("image/jpeg;charset=utf-8");
    } else if (httpType == 2) {
      response.setContentType("audio/mp3;charset=utf-8");
    } else if (httpType == 3) {
      response.setContentType("video/mpeg4;charset=utf-8");
    } else {
      throw new IllegalArgumentException("未知文件类型");
    }
  
    OutputStream out = null;
    try {
      out = response.getOutputStream();
      out.write(result);
    } catch (IOException e) {
      LOG.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }
  /**
   * 向响应中写入一般文件附件（一般文件下载）
   * 
   * @param result 写入byte信息
   */
  protected void writeResponseFile(HttpServletRequest request, HttpServletResponse response, byte[] result, String fileName) {
    // 对fileName进行http编码
    String currentFileName = "";
    try {
      if(StringUtils.isBlank(fileName)) {
        currentFileName = "temp";
      } else {
        currentFileName = URLEncoder.encode(fileName , StandardCharsets.UTF_8.name());
      }
    } catch(UnsupportedEncodingException e) {
      LOG.error(e.getMessage() , e);
      currentFileName = "temp";
    }
    
    response.setContentType("application/octet-stream;charset=utf-8");
    // 设置文件名
    response.addHeader("Content-Disposition", "attachment;fileName=" + this.encodeFileName(request, currentFileName));
    OutputStream out = null;
    try {
      out = response.getOutputStream();
      out.write(result);
    } catch (IOException e) {
      LOG.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }
  
  /**
   * 编译文件名，解决中文乱码问题
   * @param fileName
   * @return
   */
  private String encodeFileName(HttpServletRequest request, String fileName){
    String agent = request.getHeader("User-Agent");
    if(StringUtils.isBlank(agent)) {
      try {
        return URLEncoder.encode(fileName, StandardCharsets.UTF_8.name());
      } catch (UnsupportedEncodingException e) {
        LOG.warn(e.getMessage(), e);
      }
    }
    
    // 如果条件成立，说明是Firefox或者Safari浏览器
    if(agent.contains("Firefox") || agent.contains("Safari")){
      fileName = new String(fileName.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1);
    } else if(agent.contains("Chrome")) {
      try {
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8.name());
      } catch (UnsupportedEncodingException e) {
        LOG.warn(e.getMessage(), e);
      }
    }
    return fileName;
  }

  /**
   * 这个私有方法用于判断指定的字段信息的类型是否是所谓的“基本类型”的属性。
   * @param fieldItem
   * @return
   */
  private boolean baseForPropertyW(Field fieldItem) {
    /*
     * -什么叫做基本类型属性的支持呢？包括：
     * a、属于java.lang包下面的八大基础类型和对应的原生类型
     * (byte、short、int、long、float、double、char、boolean)
     * b、实现了CharSequence接口的各种类型
     * c、一部分工具类，java.util.Date及其子类、java.lang.Number及其子类、java.text.Format及其子类
     * d、其它类型特别是集合类型，都需要进行在
     * */
    Class<?> fieldItemClass = fieldItem.getType();
    // a、===
    if(fieldItemClass.isPrimitive()
        || fieldItemClass == Byte.TYPE || fieldItemClass == Short.TYPE || fieldItemClass == Integer.TYPE
        || fieldItemClass == Long.TYPE || fieldItemClass == Float.TYPE || fieldItemClass == Double.TYPE 
        || fieldItemClass == Character.TYPE || fieldItemClass == Boolean.TYPE 
        || fieldItemClass == Byte.class || fieldItemClass == Short.class || fieldItemClass == Integer.class
        || fieldItemClass == Long.class || fieldItemClass == Float.class || fieldItemClass == Double.class 
        || fieldItemClass == Character.class || fieldItemClass == Boolean.class ) {
      return true;
    }
    // b、===
    if(CharSequence.class.isAssignableFrom(fieldItemClass)) {
      return true;
    }
    // c、===
    if(Date.class.isAssignableFrom(fieldItemClass) || Number.class.isAssignableFrom(fieldItemClass) || Format.class.isAssignableFrom(fieldItemClass)) {
      return true;
    }
    return false;
  }
}
